/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.rights;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.RightsException;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDAO;
import org.ametys.plugins.core.impl.right.AbstractHierarchicalAccessController;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;

/**
 * {@link AccessController} for a {@link Content}
 */
public class SynchronizedContentAccessController extends AbstractHierarchicalAccessController<Object>
{
    /** The synchronize collection DAO */
    private SynchronizableContentsCollectionDAO _collectionsDAO;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _collectionsDAO = (SynchronizableContentsCollectionDAO) manager.lookup(SynchronizableContentsCollectionDAO.ROLE);
    }
    
    public boolean supports(Object object)
    {
        String collectionId = null;
        if (object instanceof Content content && content.getInternalDataHolder().hasValue(SynchronizableContentsCollection.COLLECTION_ID_DATA_NAME))
        {
            collectionId = content.getInternalDataHolder().<String[]>getValue(SynchronizableContentsCollection.COLLECTION_ID_DATA_NAME)[0];
        }
        else if (object instanceof String str && str.startsWith(SynchronizeContentRightAssignmentContext.ROOT_CONTEXT_PREFIX))
        {
            collectionId = StringUtils.substringAfter(str, SynchronizeContentRightAssignmentContext.ROOT_CONTEXT_PREFIX);
        }
        
        if (collectionId != null)
        {
            // Check that SCC still exists
            return _collectionsDAO.getSynchronizableContentsCollection(collectionId) != null;
        }
        
        return false;
    }
    
    @Override
    protected Set<Object> _getParents(Object object)
    {
        if (object instanceof Content)
        {
            String[] collectionIds = ((Content) object).getInternalDataHolder().getValue(SynchronizableContentsCollection.COLLECTION_ID_DATA_NAME);
            return Set.of(SynchronizeContentRightAssignmentContext.ROOT_CONTEXT_PREFIX + collectionIds[0]);
        }
        
        return null;
    }
    
    @Override
    protected Set< ? extends Object> _convertWorkspaceToRootRightContexts(Set<Object> workspacesContexts)
    {
        try
        {
            if (workspacesContexts.contains("/cms"))
            {
                Set<Object> rootContexts = new HashSet<>();
                
                for (SynchronizableContentsCollection synchronizableContentsCollection : _collectionsDAO.getSynchronizableContentsCollections())
                {
                    if (synchronizableContentsCollection.handleRightAssignmentContext())
                    {
                        rootContexts.add(SynchronizeContentRightAssignmentContext.ROOT_CONTEXT_PREFIX + synchronizableContentsCollection.getId());
                    }
                }
                return rootContexts;
            }
            return null;
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().debug("Root node for synchronized contents does not exist. Could not determine the contents root node to obtain all permissions");
            return null;
        }
    }
    
    @Override
    protected I18nizableText getObjectLabelForExplanation(Object object) throws RightsException
    {
        if (object instanceof Content)
        {
            Map<String, I18nizableTextParameter> params = Map.of("title", getObjectLabel(object));
            return new I18nizableText("plugin.cms", "PLUGINS_CMS_CONTENT_ACCESS_CONTROLLER_CONTENT_CONTEXT_EXPLANATION_LABEL", params);
        }
        else if (object instanceof String str && str.startsWith(SynchronizeContentRightAssignmentContext.ROOT_CONTEXT_PREFIX))
        {
            String collectionId = StringUtils.substringAfter(str, SynchronizeContentRightAssignmentContext.ROOT_CONTEXT_PREFIX);
            SynchronizableContentsCollection collection = _collectionsDAO.getSynchronizableContentsCollection(collectionId);
            return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_SCC_ACCESS_CONTROLLER_COLLECTION_CONTEXT_EXPLANATION_LABEL", Map.of("collection", collection.getLabel()));
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    public I18nizableText getObjectLabel(Object object) throws RightsException
    {
        if (object instanceof Content content)
        {
            return new I18nizableText(content.getTitle());
        }
        else if (object instanceof String str && str.startsWith(SynchronizeContentRightAssignmentContext.ROOT_CONTEXT_PREFIX))
        {
            String collectionId = StringUtils.substringAfter(str, SynchronizeContentRightAssignmentContext.ROOT_CONTEXT_PREFIX);
            SynchronizableContentsCollection collection = _collectionsDAO.getSynchronizableContentsCollection(collectionId);
            return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_SCC_ACCESS_CONTROLLER_COLLECTION_CONTEXT_LABEL", Map.of("collection", collection.getLabel()));
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    public I18nizableText getObjectCategory(Object object)
    {
        String collectionId = null;
        if (object instanceof Content content)
        {
            collectionId = content.getInternalDataHolder().<String[]>getValue(SynchronizableContentsCollection.COLLECTION_ID_DATA_NAME)[0];
        }
        else if (object instanceof String str && str.startsWith(SynchronizeContentRightAssignmentContext.ROOT_CONTEXT_PREFIX))
        {
            collectionId = StringUtils.substringAfter(str, SynchronizeContentRightAssignmentContext.ROOT_CONTEXT_PREFIX);
        }
        
        SynchronizableContentsCollection collection = _collectionsDAO.getSynchronizableContentsCollection(collectionId);
        Map<String, I18nizableTextParameter> params = Map.of("collection", collection != null ? collection.getLabel() : new I18nizableText(collectionId));
        return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_RIGHT_ASSIGNMENT_CONTEXT_SYNCHRONIZED_CONTENTS_LABEL", params);
    }
    
    public int getObjectPriority(Object object)
    {
        if (object instanceof String)
        {
            return 10;
        }
        return super.getObjectPriority(object);
    }
}
