/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.search;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.core.ui.ClientSideElement;
import org.ametys.core.ui.ClientSideElement.Script;
import org.ametys.core.ui.RibbonControlsManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDAO;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionHelper;
import org.ametys.runtime.authentication.AccessDeniedException;


/**
 * Helper for SCC callables.
 */
public class SCCSearchToolHelper implements Component, Serviceable
{
    /** SCC DAO */
    protected SynchronizableContentsCollectionDAO _synchronizableContentsCollectionDAO;
    /** SCC helper */
    protected SynchronizableContentsCollectionHelper _synchronizableContentsCollectionHelper;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** Ribbon controls manager */
    protected RibbonControlsManager _ribbonControlsManager;

    public void service(ServiceManager manager) throws ServiceException
    {
        _synchronizableContentsCollectionDAO = (SynchronizableContentsCollectionDAO) manager.lookup(SynchronizableContentsCollectionDAO.ROLE);
        _synchronizableContentsCollectionHelper = (SynchronizableContentsCollectionHelper) manager.lookup(SynchronizableContentsCollectionHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _ribbonControlsManager = (RibbonControlsManager) manager.lookup(RibbonControlsManager.ROLE);
    }
    
    /**
     * Get the {@link SCCSearchModelConfiguration} from the collectionId.
     * @param collectionId Collection ID
     * @return The {@link SCCSearchModelConfiguration}
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED) // model configuration is public
    public Map<String, Object> getSearchModelConfiguration(String collectionId)
    {
        SynchronizableContentsCollection collection = _synchronizableContentsCollectionDAO.getSynchronizableContentsCollection(collectionId);
        SCCSearchModelConfiguration searchModelConfiguration = collection.getSearchModelConfiguration();
        return searchModelConfiguration.toJSON();
    }

    /**
     * Import the content specified by the id in the specified collection.
     * @param controllerId The identifier of the controller that opened the tool
     * @param collectionId Collection ID
     * @param id Synchronization ID of the content
     * @param additionalParameters Additional parameters
     * @return Imported contents
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> importContent(String controllerId, String collectionId, String id, Map<String, Object> additionalParameters)
    {
        _checkArgumentsAndRights(controllerId, collectionId);
        return _synchronizableContentsCollectionHelper.importContent(collectionId, id, additionalParameters);
    }
    
    /**
     * Get the value of the synchronization field.
     * @param controllerId The identifier of the controller that opened the tool
     * @param collectionId Collection ID
     * @param contentId Content ID
     * @return The value of the synchronization field
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public String getSyncCode(String controllerId, String contentId, String collectionId)
    {
        _checkArgumentsAndRights(controllerId, collectionId);
        return _synchronizableContentsCollectionHelper.getSyncCode(contentId, collectionId);
    }
    
    private void _checkArgumentsAndRights(String controllerId, String collectionId) throws IllegalArgumentException, AccessDeniedException
    {
        if (StringUtils.isEmpty(controllerId))
        {
            throw new IllegalArgumentException("User " + _currentUserProvider.getUser() + " ' tried to import content from an empty controller for the collection '" + collectionId + "'.");
        }
        
        ClientSideElement controller = _ribbonControlsManager.getExtension(controllerId);
        
        // #getScripts method will check rights configured on the controller
        List<Script> scripts = controller.getScripts(false, new HashMap<>());
        
        // Get the id of the id collection configured on the controller with the given id
        String configuredCollectionId = scripts.stream()
                                               .map(Script::getParameters)
                                               .filter(params -> params.containsKey("sccModelId"))
                                               .map(params -> params.get("sccModelId"))
                                               .filter(String.class::isInstance)
                                               .map(String.class::cast)
                                               .findFirst()
                                               .map(_synchronizableContentsCollectionHelper::getSCCFromModelId)
                                               .map(SynchronizableContentsCollection::getId)
                                               .orElse(null);
        
        // Check that the given collectionId is the same as the one configured on the controller 
        if (!collectionId.equals(configuredCollectionId))
        {
            throw new IllegalArgumentException("User " + _currentUserProvider.getUser() + " ' tried to import content from the controller '" + controllerId + "' that does not match the excepted controller for the collection '" + collectionId + "'.");
        }
    }
}
