/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.workflow;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.slf4j.Logger;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.data.ContentSynchronizationContext;
import org.ametys.cms.data.ContentSynchronizationResult;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.workflow.AbstractContentFunction;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.core.observation.Event;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.AvalonLoggerAdapter;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDataProvider;
import org.ametys.plugins.repository.data.external.ExternalizableDataProvider.ExternalizableDataStatus;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper.SynchronizableValuesStatusInfos;
import org.ametys.plugins.repository.data.holder.values.SynchronizationResult;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.View;
import org.ametys.runtime.parameter.ValidationResult;
import org.ametys.runtime.parameter.ValidationResults;

import com.opensymphony.workflow.WorkflowException;

/**
 * OSWorkflow function to edit (import or synchronize) a synchronized content.
 */
public class EditSynchronizedContentFunction extends EditContentFunction
{
    /** Constant for storing the action id for editing invert relations on synchronized contents. */
    public static final String SYNCHRO_INVERT_EDIT_ACTION_ID_KEY = EditSynchronizedContentFunction.class.getName() + "$synchroInvertEditActionId";
    /** Constant for storing the action id for editing invert relations. */
    public static final String INVERT_EDIT_ACTION_ID_KEY = EditSynchronizedContentFunction.class.getName() + "$invertEditActionId";
    
    /** Constant for storing the ids of the contents related to the current content but that are not part of the synchronization **/
    public static final String NOT_SYNCHRONIZED_RELATED_CONTENT_IDS_KEY = EditSynchronizedContentFunction.class.getName() + "$notSynchronizedContents";
    
    /** Constant for storing the scc into the transient variables map. */
    public static final String SCC_KEY = SynchronizableContentsCollection.class.getName();
    /** Constant for storing the scc logger into the transient variables map. */
    public static final String SCC_LOGGER_KEY = SynchronizableContentsCollection.class.getName() + "$logger";
    /** Constant for storing the import parameters into the transient variables map. */
    public static final String ADDITIONAL_PARAMS_KEY = SynchronizableContentsCollection.class.getName() + "$additionalParams";
    
    /** Default action id of editing invert relations on synchronized contents. */
    public static final int DEFAULT_SYNCHRO_INVERT_EDIT_ACTION_ID = 800;
    /** Default action id of editing revert relations. */
    public static final int DEFAULT_INVERT_EDIT_ACTION_ID = 22;
    
    /** 
     * Import mode parameter.
     * If the parameter is present and set to <code>true</code>, the content is considered as imported.
     * Otherwise, it is considered as synchronized.
     */
    public static final String IMPORT = "import";
    
    @Override
    protected ContentSynchronizationContext getSynchronizationContext(Map transientVars)
    {
        ContentSynchronizationContext synchronizationContext = super.getSynchronizationContext(transientVars)
                .withStatus(ExternalizableDataStatus.EXTERNAL);
        
        getSynchronizableContentsCollection(transientVars)
            .map(SynchronizableContentsCollection::getId)
            .ifPresent(sccId -> synchronizationContext.withExternalizableDataContextEntry(SynchronizableContentsCollectionDataProvider.SCC_ID_CONTEXT_KEY, sccId));
        
        return synchronizationContext;
    }
    
    @Override
    protected Map<String, Object> convertValues(ModifiableContent content, View view, Map<String, Object> values, Map transientVars)
    {
        Map<String, Object> externalizableContext = new HashMap<>(); 
        getSynchronizableContentsCollection(transientVars)
                .map(SynchronizableContentsCollection::getId)
                .ifPresent(sccId -> externalizableContext.put(SynchronizableContentsCollectionDataProvider.SCC_ID_CONTEXT_KEY, sccId));
        
        SynchronizableValuesStatusInfos statusInfos = new SynchronizableValuesStatusInfos(content, externalizableContext);
        
        return DataHolderHelper.convertValuesWithSynchronizableValues(view, values, DataHolderHelper::convertValueIgnoringIncompatibleOnes, Optional.of(statusInfos));
    }
    
    @Override
    protected ValidationResults validateValues(View view, ModifiableContent content, Map<String, Object> values, Map transientVars) throws WorkflowException
    {
        // Do not validate synchronized values.
        // Sometimes want to be able to get invalid data at the end of the synchronization.
        // The validation workflow will fail and the content won't be valid anymore. That's normal
        return new ValidationResults();
    }
    
    @Override
    protected ValidationResult globalValidate(View view, Content content, Map<String, Object> values)
    {
        // Do not validate synchronized values.
        // Sometimes want to be able to get invalid data at the end of the synchronization.
        // The validation workflow will fail and the content won't be valid anymore. That's normal
        return ValidationResult.empty();
    }
    
    @Override
    protected ContentSynchronizationResult additionalOperations(ModifiableContent content, Map transientVars) throws WorkflowException
    {
        ContentSynchronizationResult result = super.additionalOperations(content, transientVars);

        Optional<SynchronizableContentsCollection> scc = getSynchronizableContentsCollection(transientVars);
        // There is no SCC in transient var when the modification is due to an invert relation
        if (scc.isPresent())
        {
            Map<String, Object> additionalParams = getAdditionalParameters(transientVars);
            ContentSynchronizationResult additionalResult;

            // Do the SCC additional operations
            if (isImportMode(transientVars))
            {
                additionalResult = scc.get().additionalImportOperations(content, additionalParams, getLogger(transientVars));
            }
            else
            {
                additionalResult = scc.get().additionalSynchronizeOperations(content, additionalParams, getLogger(transientVars));
            }
            
            result.aggregateResult(additionalResult);
        }
        
        return result;
    }
    
    /**
     * Determine if the content is being imported or synchronized
     * @param transientVars the parameters from the call.
     * @return <code>true</code> if the content is being imported, <code>false</code> otherwise
     * @throws WorkflowException if the object model or the request is not present
     */
    protected boolean isImportMode(Map transientVars) throws WorkflowException
    {
        Map<String, Object> parameters = getContextParameters(transientVars);
        Boolean importContent = (Boolean) parameters.get(IMPORT);
        return Boolean.TRUE.equals(importContent);
    }
    
    /**
     * Retrieve the scc associated with the workflow.
     * @param transientVars the parameters from the call.
     * @return the scc.
     */
    protected Optional<SynchronizableContentsCollection> getSynchronizableContentsCollection(Map transientVars)
    {
        return Optional.of(transientVars)
                .map(vars -> vars.get(SCC_KEY))
                .filter(SynchronizableContentsCollection.class::isInstance)
                .map(SynchronizableContentsCollection.class::cast);
    }
    
    /**
     * Retrieve the scc logger
     * @param transientVars the parameters from the call.
     * @return the scc logger
     */
    protected Logger getLogger(Map transientVars)
    {
        return Optional.of(transientVars)
                .map(vars -> vars.get(SCC_LOGGER_KEY))
                .filter(Logger.class::isInstance)
                .map(Logger.class::cast)
                .orElseGet(() -> new AvalonLoggerAdapter(_logger));
    }
    
    /**
     * Retrieve the synchronization additional parameters
     * @param transientVars the parameters from the call.
     * @return the synchronization additional parameters
     */
    protected Map<String, Object> getAdditionalParameters(Map transientVars)
    {
        return Optional.of(transientVars)
                .map(vars -> vars.get(ADDITIONAL_PARAMS_KEY))
                .filter(Map.class::isInstance)
                .map(Map.class::cast)
                .orElseGet(HashMap::new);
    }
    
    @Override
    protected int getInvertEditActionId(Map transientVars, Content referencedContent)
    {
        int synchroInvertActionId = (int) transientVars.getOrDefault(SYNCHRO_INVERT_EDIT_ACTION_ID_KEY, DEFAULT_SYNCHRO_INVERT_EDIT_ACTION_ID);
        int invertActionId = (int) transientVars.getOrDefault(INVERT_EDIT_ACTION_ID_KEY, DEFAULT_INVERT_EDIT_ACTION_ID);
        
        @SuppressWarnings("unchecked")
        Set<String> notSynchronizedContents = (Set<String>) transientVars.getOrDefault(NOT_SYNCHRONIZED_RELATED_CONTENT_IDS_KEY, Set.of());
        return notSynchronizedContents.contains(referencedContent.getId()) ? invertActionId : synchroInvertActionId;
    }
    
    @Override
    protected void notifyContentModifying(Content content, Map<String, Object> values, Map transientVars) throws WorkflowException
    {
        // Do nothing
    }
    
    @Override
    protected void updateCommonMetadata(ModifiableContent content, UserIdentity user, SynchronizationResult synchronizationResult) throws WorkflowException
    {
        if (synchronizationResult.hasChanged())
        {
            super.updateCommonMetadata(content, user, synchronizationResult);
        }
    }
    
    @Override
    protected void extractOutgoingReferences(ModifiableContent content, SynchronizationResult synchronizationResult)
    {
        if (synchronizationResult.hasChanged())
        {
            super.extractOutgoingReferences(content, synchronizationResult);
        }
    }
    
    @Override
    protected void prepareOrNotifyContentModified(Content content, Map transientVars, Map args, SynchronizationResult synchronizationResult) throws WorkflowException
    {
        if (synchronizationResult.hasChanged())
        {
            boolean notify = Boolean.parseBoolean((String) args.getOrDefault(KEY_NOTIFY_ARGUMENTS, "true"));
            if (notify)
            {
                Map<String, Object> eventParams = new HashMap<>();
                eventParams.put(ObservationConstants.ARGS_CONTENT, content);
                eventParams.put(ObservationConstants.ARGS_CONTENT_ID, content.getId());
                
                _observationManager.notify(new Event(org.ametys.plugins.contentio.synchronize.observation.ObservationConstants.EVENT_CONTENT_SYNCHRONIZED, getUser(transientVars), eventParams));
            }
            else // if notify is false, just prepare the event to send. A post notifyFunction will notify this event...
            {
                transientVars.put(AbstractContentFunction.EVENT_TO_NOTIFY_KEY, org.ametys.plugins.contentio.synchronize.observation.ObservationConstants.EVENT_CONTENT_SYNCHRONIZED);
            }
        }
    }
    
    @Override
    protected boolean canWriteModelItem(ModelItem modelItem, Content content, Map transientVars)
    {
        if (getSynchronizableContentsCollection(transientVars)
                .map(SynchronizableContentsCollection::ignoreRestrictions)
                .orElse(false))
        {
            return true;
        }
        return super.canWriteModelItem(modelItem, content, transientVars);
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_EDIT_CONTENT_FUNCTION_LABEL");
    }
}
