/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.util.function.Predicate;

import org.apache.excalibur.xml.sax.ContentHandlerProxy;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

/**
 * ContentHandler filtering first level elements from the JCR system view export of a plugin Node.
 */
public class SystemViewHandler extends ContentHandlerProxy
{
    private static final String __SV_NS = "http://www.jcp.org/jcr/sv/1.0";
    private static final String __SV_NODE = "node";
    private static final String __SV_PROPERTY = "property";
    private static final String __SV_NAME = "sv:name";
    
    private boolean _inExcludedTag;
    private int _depth;
    
    private Predicate<String> _nodeNamePredicate;
    private Predicate<String> _propertyNamePredicate;
    
    /**
     * Constructor.
     * @param handler the destination {@link ContentHandler}.
     * @param nodeNamePredicate first level nodes matching this Predicate will be filtered out
     * @param propertyNamePredicate  first level properties matching this Predicate will be filtered out
     */
    public SystemViewHandler(ContentHandler handler, Predicate<String> nodeNamePredicate, Predicate<String> propertyNamePredicate)
    {
        super(handler);
        _nodeNamePredicate = nodeNamePredicate;
        _propertyNamePredicate = propertyNamePredicate;
    }

    @Override
    public void startElement(String uri, String loc, String raw, Attributes a) throws SAXException
    {
        _depth++;
        
        if (_inExcludedTag)
        {
            return;
        }
        else if (_depth == 2 && __SV_NS.equals(uri) && (__SV_NODE.equals(loc) && _nodeNamePredicate.test(a.getValue(__SV_NAME))
                                                        || __SV_PROPERTY.equals(loc) && _propertyNamePredicate.test(a.getValue(__SV_NAME))))
        {
            _inExcludedTag = true;
            return;
        }
        
        super.startElement(uri, loc, raw, a);
    }
    
    @Override
    public void characters(char[] ch, int start, int len) throws SAXException
    {
        if (_inExcludedTag)
        {
            return;
        }
        
        super.characters(ch, start, len);
    }
    
    @Override
    public void endElement(String uri, String loc, String raw) throws SAXException
    {
        _depth--;
        
        if (_depth == 1 && __SV_NS.equals(uri))
        {
            if (_inExcludedTag)
            {
                _inExcludedTag = false;
                return;
            }
        }
        else if (_inExcludedTag)
        {
            return;
        }
        
        super.endElement(uri, loc, raw);
    }
}
