/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.in;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.WorkflowException;

/**
 * Abstract {@link ContentImporter} class which provides base importer configuration and logic.&lt;br&gt;
 * Configuration options:
 * &lt;ul&gt;
 *   &lt;li&gt;Importer priority&lt;/li&gt;
 *   &lt;li&gt;Allowed extensions, without leading dot and comma-separated&lt;/li&gt;
 *   &lt;li&gt;Content types and mixins of the created contents&lt;/li&gt;
 *   &lt;li&gt;Language of the created contents&lt;/li&gt;
 *   &lt;li&gt;Content workflow name and creation action ID&lt;/li&gt;
 * &lt;/ul&gt;&lt;br&gt;
 * Example configuration handled by the configure method:
 * &lt;pre&gt;
 * &lt;extension point="org.ametys.plugins.contentio.ContentImporterExtensionPoint"
 *               id="my.content.importer"
 *               class="..."&gt;
 *     &lt;priority&gt;500&lt;/priority&gt;
 *     &lt;extensions&gt;ext,ext2&lt;/extensions&gt;
 *     &lt;content-creation&gt;
 *         &lt;content-types&gt;My.ContentType.1,My.ContentType.2&lt;/content-types&gt;
 *         &lt;mixins&gt;My.Mixin.1,My.Mixin.2&lt;/mixins&gt;
 *         &lt;language&gt;en&lt;/language&gt;
 *         &lt;workflow name="content" createActionId="1" editActionId="2"/&gt;
 *     &lt;/content-creation&gt;
 * &lt;/extension&gt;
 * &lt;/pre&gt;
 */
public abstract class AbstractContentImporter extends AbstractLogEnabled implements ContentImporter, Serviceable, Configurable
{
    
    /** The default importer priority. */
    protected static final int DEFAULT_PRIORITY = 5000;
    
    /** Map used to store the mapping from "local" ID to content ID, when actually imported. */
    protected static final String _CONTENT_ID_MAP_KEY = AbstractContentImporter.class.getName() + "$contentIdMap";
    
    /** Map used to store the content references, indexed by content and metadata path. */
    protected static final String _CONTENT_LINK_MAP_KEY = AbstractContentImporter.class.getName() + "$contentLinkMap";
    
    /** Map used to store the content repeater sizes. */
    protected static final String _CONTENT_REPEATER_SIZE_MAP = AbstractContentImporter.class.getName() + "$contentRepeaterSizeMap";
    
    /** The AmetysObject resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The content workflow helper. */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    
    /** The importer priority. */
    protected int _priority = DEFAULT_PRIORITY;
    
    /** The allowed extensions. */
    protected Set<String> _extensions;
    
    /** The imported contents' types. */
    protected String[] _contentTypes;
    
    /** The imported contents' mixins. */
    protected String[] _mixins;
    
    /** The importer contents' language. */
    protected String _language;
    
    /** The importer contents' workflow name. */
    protected String _workflowName;
    
    /** The importer contents' initial action ID. */
    protected int _initialActionId;
    
    /** The importer contents' edition action ID. */
    protected int _editActionId;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _priority = configuration.getChild("priority").getValueAsInteger(DEFAULT_PRIORITY);
        
        configureExtensions(configuration.getChild("extensions"));
        
        configureContentCreation(configuration.getChild("content-creation"));
    }
    
    /**
     * Configure the allowed extensions.
     * @param configuration the extension configuration.
     * @throws ConfigurationException if an error occurs.
     */
    protected void configureExtensions(Configuration configuration) throws ConfigurationException
    {
        _extensions = new HashSet<>();
        
        String extensionsStr = configuration.getValue("");
        
        if (StringUtils.isBlank(extensionsStr))
        {
            _extensions.addAll(getDefaultExtensions());
        }
        else
        {
            for (String ext : StringUtils.split(extensionsStr, ", "))
            {
                String extension = ext.trim();
                if (extension.startsWith("."))
                {
                    extension = extension.substring(1);
                }
                
                _extensions.add(extension);
            }
        }
    }
    
    /**
     * Configure the content creation parameters.
     * @param configuration the content creation configuration.
     * @throws ConfigurationException if an error occurs.
     */
    protected void configureContentCreation(Configuration configuration) throws ConfigurationException
    {
        String typesStr = configuration.getChild("content-types").getValue();
        _contentTypes = StringUtils.split(typesStr, ", ");
        
        String mixins = configuration.getChild("mixins").getValue("");  // mixins can be empty
        _mixins = StringUtils.split(mixins, ", ");
        
        _language = configuration.getChild("language").getValue();
        
        configureWorkflow(configuration);
    }
    
    /**
     * Configure the content workflow.
     * @param configuration the content creation configuration.
     * @throws ConfigurationException if an error occurs.
     */
    protected void configureWorkflow(Configuration configuration) throws ConfigurationException
    {
        Configuration wfConf = configuration.getChild("workflow");
        
        _workflowName = wfConf.getAttribute("name");
        
        _initialActionId = wfConf.getAttributeAsInteger("createActionId");
        _editActionId = wfConf.getAttributeAsInteger("editActionId");
    }
    
    @Override
    public int getPriority()
    {
        return _priority;
    }
    
    /**
     * Get the default allowed extensions.
     * @return the default allowed extensions, without leading dots. Cannot be null.
     */
    protected Collection<String> getDefaultExtensions()
    {
        return Collections.emptySet();
    }
    
    /**
     * Test if the given filename has a supported extension.
     * @param name the name, can't be null.
     * @return true if the extension is supported, false otherwise.
     * @throws IOException if an error occurs.
     */
    protected boolean isExtensionValid(String name) throws IOException
    {
        return _extensions.isEmpty() || _extensions.contains(FilenameUtils.getExtension(name));
    }
    
    /**
     * The content types of a created content.
     * @param params the import parameters.
     * @return the content types of a created content.
     */
    protected String[] getContentTypes(Map<String, Object> params)
    {
        return _contentTypes;
    }
    
    /**
     * The mixins of a created content.
     * @param params the import parameters.
     * @return The mixins of a created content.
     */
    protected String[] getMixins(Map<String, Object> params)
    {
        return _mixins;
    }
    
    /**
     * The language of a created content.
     * @param params the import parameters.
     * @return The language of a created content.
     */
    protected String getLanguage(Map<String, Object> params)
    {
        return _language;
    }
    
    /**
     * The workflow name of a created content.
     * @param params the import parameters.
     * @return The workflow name of a created content.
     */
    protected String getWorkflowName(Map<String, Object> params)
    {
        return _workflowName;
    }
    
    /**
     * The workflow creation action ID of a created content.
     * @param params the import parameters.
     * @return The workflow creation action ID of a created content.
     */
    protected int getInitialActionId(Map<String, Object> params)
    {
        return _initialActionId;
    }
    
    /**
     * The workflow action ID used to edit a content.
     * @param params the import parameters.
     * @return The workflow action ID used to edit a content.
     */
    protected int getEditActionId(Map<String, Object> params)
    {
        return _editActionId;
    }
    
    /**
     * Get the map used to store the mapping from "local" ID (defined in the import file)
     * to the AmetysObject ID of the contents, when actually imported.
     * @param params the import parameters.
     * @return the content "local to repository" ID map.
     */
    protected Map<String, String> getContentIdMap(Map<String, Object> params)
    {
        // Get or create the map in the global parameters.
        @SuppressWarnings("unchecked")
        Map<String, String> contentIdMap = (Map<String, String>) params.get(_CONTENT_ID_MAP_KEY);
        if (contentIdMap == null)
        {
            contentIdMap = new HashMap<>();
            params.put(_CONTENT_ID_MAP_KEY, contentIdMap);
        }
        
        return contentIdMap;
    }
    
    /**
     * Get the map used to store the content references.
     * The Map is shaped like: referencing content -&gt; local metadata path -&gt; content references.
     * @param params the import parameters.
     * @return the content reference map.
     */
    protected Map<Content, Map<String, Object>> getContentRefMap(Map<String, Object> params)
    {
        // Get or create the map in the global parameters.
        @SuppressWarnings("unchecked")
        Map<Content, Map<String, Object>> contentRefMap = (Map<Content, Map<String, Object>>) params.get(_CONTENT_LINK_MAP_KEY);
        if (contentRefMap == null)
        {
            contentRefMap = new HashMap<>();
            params.put(_CONTENT_LINK_MAP_KEY, contentRefMap);
        }
        
        return contentRefMap;
    }
    
    /**
     * Add a content reference to the map.
     * @param content The referencing content.
     * @param metadataPath The path of the metadata which holds the content references.
     * @param reference The content reference.
     * @param params The import parameters.
     */
    protected void addContentReference(Content content, String metadataPath, ContentReference reference, Map<String, Object> params)
    {
        addContentReference(getContentRefMap(params), content, metadataPath, reference);
    }
    
    /**
     * Add a content reference to the map.
     * @param contentRefMap The content reference map.
     * @param content The referencing content.
     * @param metadataPath The path of the metadata which holds the content references.
     * @param reference The content reference.
     */
    protected void addContentReference(Map<Content, Map<String, Object>> contentRefMap, Content content, String metadataPath, ContentReference reference)
    {
        Map<String, Object> contentReferences;
        if (contentRefMap.containsKey(content))
        {
            contentReferences = contentRefMap.get(content);
        }
        else
        {
            contentReferences = new HashMap<>();
            contentRefMap.put(content, contentReferences);
        }
        
        contentReferences.put(metadataPath, reference);
    }
    
    /**
     * Add content references to the map.
     * @param contentRefMap The content reference map.
     * @param content The referencing content.
     * @param metadataPath The path of the metadata which holds the content references.
     * @param references the content reference list.
     */
    protected void addContentReferences(Map<Content, Map<String, Object>> contentRefMap, Content content, String metadataPath, List<ContentReference> references)
    {
        Map<String, Object> contentReferences;
        if (contentRefMap.containsKey(content))
        {
            contentReferences = contentRefMap.get(content);
        }
        else
        {
            contentReferences = new HashMap<>();
            contentRefMap.put(content, contentReferences);
        }
        
        contentReferences.put(metadataPath, references);
    }
    
    /**
     * Get the map used to store the repeater sizes.
     * The Map is shaped like: referencing content -&gt; local metadata path -&gt; content references.
     * @param params the import parameters.
     * @return the content reference map.
     */
    protected Map<Content, Map<String, Integer>> getContentRepeaterSizeMap(Map<String, Object> params)
    {
        // Get or create the map in the global parameters.
        @SuppressWarnings("unchecked")
        Map<Content, Map<String, Integer>> contentRepeaterSizeMap = (Map<Content, Map<String, Integer>>) params.get(_CONTENT_REPEATER_SIZE_MAP);
        if (contentRepeaterSizeMap == null)
        {
            contentRepeaterSizeMap = new HashMap<>();
            params.put(_CONTENT_REPEATER_SIZE_MAP, contentRepeaterSizeMap);
        }
        
        return contentRepeaterSizeMap;
    }
    
    /**
     * Set a repeater size in the map (needed to execute the edit content function).
     * @param content The content containing the repeater.
     * @param metadataPath The repeater metadata path.
     * @param repeaterSize The repeater size.
     * @param params The import parameters.
     */
    protected void setRepeaterSize(Content content, String metadataPath, int repeaterSize, Map<String, Object> params)
    {
        Map<Content, Map<String, Integer>> contentRepeaterSizeMap = getContentRepeaterSizeMap(params);
        
        Map<String, Integer> repeaters;
        if (contentRepeaterSizeMap.containsKey(content))
        {
            repeaters = contentRepeaterSizeMap.get(content);
        }
        else
        {
            repeaters = new HashMap<>();
            contentRepeaterSizeMap.put(content, repeaters);
        }
        
        repeaters.put(metadataPath, repeaterSize);
    }
    
    /**
     * Create a content.
     * @param title the content title.
     * @param params the import parameters.
     * @return the created content.
     * @throws WorkflowException if an error occurs.
     */
    protected Content createContent(String title, Map<String, Object> params) throws WorkflowException
    {
        String[] contentTypes = getContentTypes(params);
        String[] mixins = getMixins(params);
        String language = getLanguage(params);
        String workflowName = getWorkflowName(params);
        int initialActionId = getInitialActionId(params);
        
        return createContent(title, contentTypes, mixins, language, workflowName, initialActionId, params);
    }
    
    /**
     * Create a content.
     * @param title the content title.
     * @param contentTypes the content types.
     * @param mixins the content mixins.
     * @param language the content language.
     * @param params the import parameters.
     * @return the created content.
     * @throws WorkflowException if an error occurs.
     */
    protected Content createContent(String title, String[] contentTypes, String[] mixins, String language, Map<String, Object> params) throws WorkflowException
    {
        String workflowName = getWorkflowName(params);
        int initialActionId = getInitialActionId(params);
        
        return createContent(title, contentTypes, mixins, language, workflowName, initialActionId, params);
    }
    
    /**
     * Create a content.
     * @param title the content title.
     * @param contentTypes the content types.
     * @param mixins the content mixins.
     * @param language the content language.
     * @param workflowName the content workflow name.
     * @param initialActionId the content create action ID.
     * @param params the import parameters.
     * @return the created content.
     * @throws WorkflowException if an error occurs.
     */
    protected Content createContent(String title, String[] contentTypes, String[] mixins, String language, String workflowName, int initialActionId, Map<String, Object> params) throws WorkflowException
    {
        String name;
        try
        {
            name = NameHelper.filterName(title);
        }
        catch (Exception e)
        {
            // Ignore the exception, just provide a valid start.
            name = "content-" + title;
        }
        
        Map<String, Object> result = _contentWorkflowHelper.createContent(workflowName, initialActionId, name, title, contentTypes, mixins, language);
        
        return (Content) result.get(AbstractContentWorkflowComponent.CONTENT_KEY);
    }
    
    /**
     * Restore content references.
     * @param params The import parameters.
     */
    protected void restoreContentReferences(Map<String, Object> params)
    {
        Map<Content, Map<String, Object>> contentRefMap = getContentRefMap(params);
        Map<Content, Map<String, Integer>> contentRepeaterSizeMap = getContentRepeaterSizeMap(params);
        int editActionId = getEditActionId(params);
        
        for (Content content : contentRefMap.keySet())
        {
            if (content instanceof WorkflowAwareContent)
            {
                Map<String, Object> contentReferences = contentRefMap.get(content);
                Map<String, Integer> repeaters = contentRepeaterSizeMap.get(content);
                
                Map<String, Object> values = new HashMap<>();
                
                // Fill the value map with the content references.
                setReferenceMetadatas(contentReferences, values, repeaters, params);
                
                try
                {
                    if (!values.isEmpty())
                    {
                        Map<String, Object> contextParameters = new HashMap<>();
                        contextParameters.put(EditContentFunction.QUIT, true);
                        contextParameters.put(EditContentFunction.FORM_RAW_VALUES, values);
                        
                        Map<String, Object> inputs = new HashMap<>();
                        inputs.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, contextParameters);
                        inputs.put(AbstractContentWorkflowComponent.CONTENT_KEY, content);
                        
                        _contentWorkflowHelper.doAction((WorkflowAwareContent) content, editActionId, inputs);
                    }
                }
                catch (WorkflowException e)
                {
                    // TODO Throw exception?
                    getLogger().warn("An error occurred restoring content references in content {}", content, e);
                }
            }
        }
    }
    
    /**
     * Fill the value map with the content references.
     * @param contentReferences The list of content references indexed by metadata path.
     * @param values The value map passed to the EditContentFunction class.
     * @param repeaters The repeater sizes for this content.
     * @param params The import parameters.
     */
    protected void setReferenceMetadatas(Map<String, Object> contentReferences, Map<String, Object> values, Map<String, Integer> repeaters, Map<String, Object> params)
    {
        for (String metadataPath : contentReferences.keySet())
        {
            Object value = contentReferences.get(metadataPath);
            String metaKey = EditContentFunction.FORM_ELEMENTS_PREFIX + metadataPath;
            
            if (value instanceof List<?>)
            {
                // Multiple value
                @SuppressWarnings("unchecked")
                List<ContentReference> references = (List<ContentReference>) value;
                List<String> contentIds = new ArrayList<>(references.size());
                for (ContentReference reference : references)
                {
                    String refContentId = getReferencedContentId(reference, params);
                    if (refContentId != null)
                    {
                        contentIds.add(refContentId);
                    }
                }
                
                if (!contentIds.isEmpty())
                {
                    values.put(metaKey, contentIds);
                }
            }
            else if (value instanceof ContentReference)
            {
                // Single value.
                String refContentId = getReferencedContentId((ContentReference) value, params);
                if (refContentId != null)
                {
                    values.put(metaKey, refContentId);
                }
            }
        }
        
        if (repeaters != null)
        {
            for (String repeaterPath : repeaters.keySet())
            {
                Integer size = repeaters.get(repeaterPath);
                if (size > 0)
                {
                    String sizeKey = EditContentFunction.INTERNAL_FORM_ELEMENTS_PREFIX + repeaterPath + "/size";
                    values.put(sizeKey, repeaters.get(repeaterPath).toString());
                }
            }
        }
    }
    
    /**
     * Get the content ID from a content reference.
     * @param contentRef The content reference.
     * @param params The import parameters.
     * @return the content ID if it was found, or null otherwise.
     */
    protected String getReferencedContentId(ContentReference contentRef, Map<String, Object> params)
    {
        int refType = contentRef.getType();
        if (refType == ContentReference.TYPE_LOCAL_ID)
        {
            String localId = (String) contentRef.getValue();
            String contentId = getContentIdMap(params).get(localId);
            if (StringUtils.isNotEmpty(contentId) && _resolver.hasAmetysObjectForId(contentId))
            {
                return contentId;
            }
        }
        else if (refType == ContentReference.TYPE_CONTENT_ID)
        {
            String contentId = (String) contentRef.getValue();
            if (StringUtils.isNotEmpty(contentId) && _resolver.hasAmetysObjectForId(contentId))
            {
                return contentId;
            }
        }
        else if (refType == ContentReference.TYPE_CONTENT_VALUES)
        {
            @SuppressWarnings("unchecked")
            Map<String, String> values = (Map<String, String>) contentRef.getValue();
            Content content = getContentFromProperties(values);
            if (content != null)
            {
                return content.getId();
            }
        }
        
        return null;
    }
    
    /**
     * Search a content from a map of its metadata values.
     * @param propertyValues The metadata values.
     * @return The Content if found, null otherwise.
     */
    protected Content getContentFromProperties(Map<String, String> propertyValues)
    {
        Content content = null;
        
        AndExpression expression = new AndExpression();
        for (String property : propertyValues.keySet())
        {
            String value = propertyValues.get(property);
            expression.add(new StringExpression(property, Operator.EQ, value));
        }
        
        String query = ContentQueryHelper.getContentXPathQuery(expression);
        
        AmetysObjectIterable<Content> contents = _resolver.query(query);
        Iterator<Content> it = contents.iterator();
        
        if (it.hasNext())
        {
            content = it.next();
            
            if (it.hasNext())
            {
                content = null;
            }
        }
        
        return content;
    }
    
    /**
     * Class representing a reference to a content in an import file.
     */
    public class ContentReference
    {
        /**
         * The referenced content doesn't exist in the repository, it's in the import file.
         * The reference value is the content ID in the import file.
         */
        public static final int TYPE_LOCAL_ID = 1;
        
        /**
         * The referenced content exits in the repository and its ID is known.
         * The reference value is the content ID in the repository (AmetysObject ID).
         */
        public static final int TYPE_CONTENT_ID = 2;
        
        /**
         * The referenced content exits in the repository. Its ID is not known,
         * but it can be identified by one or several of its metadata.
         * The reference value is a Map of metadata name -&gt; value.
         */
        public static final int TYPE_CONTENT_VALUES = 3;
        
        /** The reference type. */
        private int _type;
        
        /** The reference value, depends on the reference type. */
        private Object _value;
        
        /**
         * Build a content reference.
         * @param type the reference type.
         * @param value the reference value.
         */
        public ContentReference(int type, Object value)
        {
            this._type = type;
            this._value = value;
        }

        /**
         * Get the type.
         * @return the type
         */
        public int getType()
        {
            return _type;
        }

        /**
         * Set the type.
         * @param type the type to set
         */
        public void setType(int type)
        {
            this._type = type;
        }

        /**
         * Get the value.
         * @return the value
         */
        public Object getValue()
        {
            return _value;
        }

        /**
         * Set the value.
         * @param value the value to set
         */
        public void setValue(Object value)
        {
            this._value = value;
        }
    }
}
