/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.function.Function;
import java.util.stream.Stream;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.LifecycleHelper;
import org.apache.cocoon.util.log.SLF4JLoggerAdapter;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.xml.serializer.OutputPropertiesFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.languages.LanguagesManager;
import org.ametys.core.datasource.AbstractDataSourceManager.DataSourceDefinition;
import org.ametys.core.datasource.LDAPDataSourceManager;
import org.ametys.core.datasource.SQLDataSourceManager;
import org.ametys.core.ui.Callable;
import org.ametys.plugins.contentio.synchronize.impl.DefaultSynchronizingContentOperator;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.DefinitionContext;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.checker.ItemCheckerTestFailureException;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.parameter.Validator;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.LogEnabled;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * DAO for accessing {@link SynchronizableContentsCollection}
 */
public class SynchronizableContentsCollectionDAO extends AbstractLogEnabled implements Component, Serviceable, Initializable, Contextualizable, Disposable
{
    /** Avalon Role */
    public static final String ROLE = SynchronizableContentsCollectionDAO.class.getName();
    
    /** Separator for parameters with model id as prefix */
    public static final String SCC_PARAMETERS_SEPARATOR = "$";
    
    private static File __CONFIGURATION_FILE;
    
    private Map<String, SynchronizableContentsCollection> _synchronizableCollections;
    private long _lastFileReading;

    private SynchronizeContentsCollectionModelExtensionPoint _syncCollectionModelEP;
    private ContentTypeExtensionPoint _contentTypeEP;
    private WorkflowHelper _workflowHelper;
    private SynchronizingContentOperatorExtensionPoint _synchronizingContentOperatorEP;
    private LanguagesManager _languagesManager;
    
    private ServiceManager _smanager;
    private Context _context;

    private SQLDataSourceManager _sqlDataSourceManager;
    private LDAPDataSourceManager _ldapDataSourceManager;
    
    @Override
    public void initialize() throws Exception
    {
        __CONFIGURATION_FILE = new File(AmetysHomeHelper.getAmetysHome(), "config" + File.separator + "synchronizable-collections.xml");
        _synchronizableCollections = new HashMap<>();
        _lastFileReading = 0;
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _smanager = smanager;
        _syncCollectionModelEP = (SynchronizeContentsCollectionModelExtensionPoint) smanager.lookup(SynchronizeContentsCollectionModelExtensionPoint.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _workflowHelper = (WorkflowHelper) smanager.lookup(WorkflowHelper.ROLE);
        _synchronizingContentOperatorEP = (SynchronizingContentOperatorExtensionPoint) smanager.lookup(SynchronizingContentOperatorExtensionPoint.ROLE);
        _languagesManager = (LanguagesManager) smanager.lookup(LanguagesManager.ROLE);
    }
    
    private SQLDataSourceManager _getSQLDataSourceManager()
    {
        if (_sqlDataSourceManager == null)
        {
            try
            {
                _sqlDataSourceManager = (SQLDataSourceManager) _smanager.lookup(SQLDataSourceManager.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException(e);
            }
        }
        
        return _sqlDataSourceManager;
    }
    
    private LDAPDataSourceManager _getLDAPDataSourceManager()
    {
        if (_ldapDataSourceManager == null)
        {
            try
            {
                _ldapDataSourceManager = (LDAPDataSourceManager) _smanager.lookup(LDAPDataSourceManager.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException(e);
            }
        }
        
        return _ldapDataSourceManager;
    }
    
    /**
     * Gets a synchronizable contents collection to JSON format
     * @param collectionId The id of the synchronizable contents collection to get
     * @return An object representing a {@link SynchronizableContentsCollection}
     */
    @Callable (rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public Map<String, Object> getSynchronizableContentsCollectionAsJson(String collectionId)
    {
        return getSynchronizableContentsCollectionAsJson(getSynchronizableContentsCollection(collectionId));
    }
    
    /**
     * Gets a synchronizable contents collection to JSON format
     * @param collection The synchronizable contents collection to get
     * @return An object representing a {@link SynchronizableContentsCollection}
     */
    public Map<String, Object> getSynchronizableContentsCollectionAsJson(SynchronizableContentsCollection collection)
    {
        Map<String, Object> result = new LinkedHashMap<>();
        result.put("id", collection.getId());
        result.put("label", collection.getLabel());
        
        String cTypeId = collection.getContentType();
        result.put("contentTypeId", cTypeId);
        
        ContentType cType = _contentTypeEP.getExtension(cTypeId);
        result.put("contentType", cType != null ? cType.getLabel() : cTypeId);
        
        String modelId = collection.getSynchronizeCollectionModelId();
        result.put("modelId", modelId);
        SynchronizableContentsCollectionModel model = _syncCollectionModelEP.getExtension(modelId);
        result.put("model", model.getLabel());
        
        result.put("isValid", _isValid(collection));
        
        return result;
    }
    
    /**
     * Get the synchronizable contents collections
     * @return the synchronizable contents collections
     */
    public List<SynchronizableContentsCollection> getSynchronizableContentsCollections()
    {
        getLogger().debug("Calling #getSynchronizableContentsCollections()");
        _readFile(false);
        ArrayList<SynchronizableContentsCollection> cols = new ArrayList<>(_synchronizableCollections.values());
        getLogger().debug("#getSynchronizableContentsCollections() returns '{}'", cols);
        return cols;
    }
    
    /**
     * Get a synchronizable contents collection by its id
     * @param collectionId The id of collection
     * @return the synchronizable contents collection or <code>null</code> if not found
     */
    public SynchronizableContentsCollection getSynchronizableContentsCollection(String collectionId)
    {
        getLogger().debug("Calling #getSynchronizableContentsCollection(String collectionId) with collectionId '{}'", collectionId);
        _readFile(false);
        SynchronizableContentsCollection col = _synchronizableCollections.get(collectionId);
        getLogger().debug("#getSynchronizableContentsCollection(String collectionId) with collectionId '{}' returns '{}'", collectionId, col);
        return col;
    }
    
    private void _readFile(boolean forceRead)
    {
        try
        {
            if (!__CONFIGURATION_FILE.exists())
            {
                getLogger().debug("=> SCC file does not exist, it will be created.");
                _createFile(__CONFIGURATION_FILE);
            }
            else
            {
                // In Linux file systems, the precision of java.io.File.lastModified() is the second, so we need here to always have
                // this (bad!) precision by doing the truncation to second precision (/1000 * 1000) on the millis time value.
                // Therefore, the boolean outdated is computed with '>=' operator, and not '>', which will lead to sometimes (but rarely) unnecessarily re-read the file.
                long cfgFileLastModified = (__CONFIGURATION_FILE.lastModified() / 1000) * 1000;
                boolean outdated = cfgFileLastModified >= _lastFileReading;
                getLogger().debug("=> forceRead: {}", forceRead);
                getLogger().debug("=> The configuration was last modified in (long value): {}", cfgFileLastModified);
                getLogger().debug("=> The '_lastFileReading' fields is equal to (long value): {}", _lastFileReading);
                if (forceRead || outdated)
                {
                    getLogger().debug(forceRead ? "=> SCC file will be read (force)" : "=> SCC file was (most likely) updated since the last time it was read ({} >= {}). It will be re-read...", cfgFileLastModified, _lastFileReading);
                    getLogger().debug("==> '_synchronizableCollections' map before calling #_readFile(): '{}'", _synchronizableCollections);
                    _lastFileReading = Instant.now().truncatedTo(ChronoUnit.SECONDS).toEpochMilli();
                    _synchronizableCollections = new LinkedHashMap<>();
                    
                    Configuration cfg = new DefaultConfigurationBuilder().buildFromFile(__CONFIGURATION_FILE);
                    for (Configuration collectionConfig : cfg.getChildren("collection"))
                    {
                        SynchronizableContentsCollection syncCollection = _createSynchronizableCollection(collectionConfig);
                        _synchronizableCollections.put(syncCollection.getId(), syncCollection);
                    }
                    getLogger().debug("==> '_synchronizableCollections' map after calling #_readFile(): '{}'", _synchronizableCollections);
                }
                else
                {
                    getLogger().debug("=> SCC file will not be re-read, the internal representation is up-to-date.");
                }
            }
        }
        catch (Exception e)
        {
            getLogger().error("Failed to retrieve synchronizable contents collections from the configuration file {}", __CONFIGURATION_FILE, e);
        }
    }
    
    private void _createFile(File file) throws IOException, TransformerConfigurationException, SAXException
    {
        file.createNewFile();
        try (OutputStream os = new FileOutputStream(file))
        {
            TransformerHandler th = _getTransformerHandler(os);
            
            th.startDocument();
            XMLUtils.createElement(th, "collections");
            th.endDocument();
        }
    }
    
    private SynchronizableContentsCollection _createSynchronizableCollection(Configuration collectionConfig) throws ConfigurationException
    {
        String modelId = collectionConfig.getChild("model").getAttribute("id");
        
        if (_syncCollectionModelEP.hasExtension(modelId))
        {
            SynchronizableContentsCollectionModel model = _syncCollectionModelEP.getExtension(modelId);
            Class<SynchronizableContentsCollection> synchronizableCollectionClass = model.getSynchronizableCollectionClass();
            
            SynchronizableContentsCollection synchronizableCollection = null;
            try
            {
                synchronizableCollection = synchronizableCollectionClass.getDeclaredConstructor().newInstance();
            }
            catch (Exception e)
            {
                throw new IllegalArgumentException("Cannot instanciate the class " + synchronizableCollectionClass.getCanonicalName() + ". Check that there is a public constructor with no arguments.");
            }
            
            Logger logger = LoggerFactory.getLogger(synchronizableCollectionClass);
            try
            {
                if (synchronizableCollection instanceof LogEnabled)
                {
                    ((LogEnabled) synchronizableCollection).setLogger(logger);
                }
                
                LifecycleHelper.setupComponent(synchronizableCollection, new SLF4JLoggerAdapter(logger), _context, _smanager, collectionConfig);
            }
            catch (Exception e)
            {
                throw new ConfigurationException("The model id '" + modelId + "' is not a valid", e);
            }
            
            return synchronizableCollection;
        }
        
        throw new ConfigurationException("The model id '" + modelId + "' is not a valid model for collection '" + collectionConfig.getChild("id") + "'", collectionConfig);
    }
    
    /**
     * Gets the configuration for creating/editing a collection of synchronizable contents.
     * @return A map containing information about what is needed to create/edit a collection of synchronizable contents
     * @throws Exception If an error occurs.
     */
    @Callable (rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public Map<String, Object> getEditionConfiguration() throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        
        // MODELS
        List<Object> collectionModels = new ArrayList<>();
        for (String modelId : _syncCollectionModelEP.getExtensionsIds())
        {
            SynchronizableContentsCollectionModel model = _syncCollectionModelEP.getExtension(modelId);
            Map<String, Object> modelMap = new HashMap<>();
            modelMap.put("id", modelId);
            modelMap.put("label", model.getLabel());
            modelMap.put("description", model.getDescription());
            
            Map<String, Object> params = new LinkedHashMap<>();
            for (ModelItem param : model.getModelItems())
            {
                // prefix in case of two parameters from two different models have the same id which can lead to some errors in client-side
                params.put(modelId + SCC_PARAMETERS_SEPARATOR + param.getPath(), param.toJSON(DefinitionContext.newInstance().withEdition(true)));
            }
            modelMap.put("parameters", params);
            
            collectionModels.add(modelMap);
        }
        result.put("models", collectionModels);
        
        // CONTENT TYPES
        result.put(
            "contentTypes",
            _transformToJSONEnumerator(
                _contentTypeEP.getExtensionsIds().stream().map(_contentTypeEP::getExtension).filter(this::_isValidContentType),
                ContentType::getId,
                ContentType::getLabel
            )
        );
        
        // LANGUAGES
        result.put(
            "languages",
            _transformToJSONEnumerator(
                _languagesManager.getAvailableLanguages().entrySet().stream(),
                entry -> entry.getKey(),
                entry -> entry.getValue().getLabel()
            )
        );
        
        // WORKFLOWS
        result.put(
            "workflows",
            _transformToJSONEnumerator(
                Stream.of(_workflowHelper.getWorkflowNames()),
                Function.identity(),
                id -> _workflowHelper.getWorkflowLabel(id)
            )
        );
        
        // SYNCHRONIZING CONTENT OPERATORS
        result.put(
            "contentOperators",
            _transformToJSONEnumerator(
                _synchronizingContentOperatorEP.getExtensionsIds().stream(),
                Function.identity(),
                id -> _synchronizingContentOperatorEP.getExtension(id).getLabel()
            )
        );
        result.put("defaultContentOperator", DefaultSynchronizingContentOperator.class.getName());
        
        // EXISTING SCC
        result.put(
            "existingSCC",
            _transformToJSONEnumerator(
                getSynchronizableContentsCollections().stream(),
                SynchronizableContentsCollection::getId,
                SynchronizableContentsCollection::getLabel
            )
        );
        
        return result;
    }
    
    private <T> List<Map<String, Object>> _transformToJSONEnumerator(Stream<T> values, Function<T, String> valueFunction, Function<T, I18nizableText> labelFunction)
    {
        return values.map(value ->
                Map.of(
                    "value", valueFunction.apply(value),
                    "label", labelFunction.apply(value)
                )
            )
            .toList();
    }
    
    private boolean _isValidContentType (ContentType cType)
    {
        return !cType.isReferenceTable() && !cType.isAbstract() && !cType.isMixin();
    }
    
    /**
     * Gets the values of the parameters of the given collection
     * @param collectionId The id of the collection
     * @return The values of the parameters
     */
    @Callable (rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public Map<String, Object> getCollectionParameterValues(String collectionId)
    {
        Map<String, Object> result = new LinkedHashMap<>();
        
        SynchronizableContentsCollection collection = getSynchronizableContentsCollection(collectionId);
        if (collection == null)
        {
            getLogger().error("The collection of id '{}' does not exist.", collectionId);
            result.put("error", "unknown");
            return result;
        }
        
        result.put("id", collectionId);
        result.put("label", collection.getLabel());
        String modelId = collection.getSynchronizeCollectionModelId();
        result.put("modelId", modelId);
        
        result.put("contentType", collection.getContentType());
        result.put("contentPrefix", collection.getContentPrefix());
        result.put("restrictedField", collection.getRestrictedField());
        result.put("synchronizeExistingContentsOnly", collection.synchronizeExistingContentsOnly());
        result.put("removalSync", collection.removalSync());
        result.put("ignoreRestrictions", collection.ignoreRestrictions());
        result.put("checkCollection", collection.checkCollection());
        result.put("compatibleSCC", collection.getCompatibleSCC(false));
        result.put("validateAfterImport", collection.validateAfterImport());
        
        result.put("workflowName", collection.getWorkflowName());
        result.put("initialActionId", collection.getInitialActionId());
        result.put("synchronizeActionId", collection.getSynchronizeActionId());
        result.put("validateActionId", collection.getValidateActionId());
        
        result.put("contentOperator", collection.getSynchronizingContentOperator());
        result.put("reportMails", collection.getReportMails());
        
        result.put("languages", collection.getLanguages());
        
        Map<String, Object> values = collection.getParameterValues();
        for (String key : values.keySet())
        {
            result.put(modelId + SCC_PARAMETERS_SEPARATOR + key, values.get(key));
        }
        
        return result;
    }
    
    private boolean _writeFile()
    {
        File backup = _createBackup();
        boolean errorOccured = false;
        
        // Do writing
        try (OutputStream os = new FileOutputStream(__CONFIGURATION_FILE))
        {
            TransformerHandler th = _getTransformerHandler(os);

            // sax the config
            try
            {
                th.startDocument();
                XMLUtils.startElement(th, "collections");
                
                _toSAX(th);
                XMLUtils.endElement(th, "collections");
                th.endDocument();
            }
            catch (Exception e)
            {
                getLogger().error("Error when saxing the collections", e);
                errorOccured = true;
            }
        }
        catch (IOException | TransformerConfigurationException | TransformerFactoryConfigurationError e)
        {
            if (getLogger().isErrorEnabled())
            {
                getLogger().error("Error when trying to modify the group directories with the configuration file {}", __CONFIGURATION_FILE, e);
            }
        }
        
        _restoreBackup(backup, errorOccured);
        
        return errorOccured;
    }
    
    private File _createBackup()
    {
        File backup = new File(__CONFIGURATION_FILE.getPath() + ".tmp");
        
        // Create a backup file
        try
        {
            Files.copy(__CONFIGURATION_FILE.toPath(), backup.toPath());
        }
        catch (IOException e)
        {
            getLogger().error("Error when creating backup '{}' file", __CONFIGURATION_FILE.toPath(), e);
        }
        
        return backup;
    }
    
    private void _restoreBackup(File backup, boolean errorOccured)
    {
        // Restore the file if an error previously occured
        try
        {
            if (errorOccured)
            {
                // An error occured, restore the original
                Files.copy(backup.toPath(), __CONFIGURATION_FILE.toPath(), StandardCopyOption.REPLACE_EXISTING);
                // Force to reread the file
                _readFile(true);
            }
            Files.deleteIfExists(backup.toPath());
        }
        catch (IOException e)
        {
            if (getLogger().isErrorEnabled())
            {
                getLogger().error("Error when restoring backup '{}' file", __CONFIGURATION_FILE, e);
            }
        }
    }
    
    /**
     * Add a new {@link SynchronizableContentsCollection}
     * @param values The parameters' values
     * @return The id of new created collection or null in case of error
     * @throws ProcessingException if creation failed
     */
    @Callable (rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public String addCollection (Map<String, Object> values) throws ProcessingException
    {
        getLogger().debug("Add new Collection with values '{}'", values);
        _readFile(false);
        
        String id = _generateUniqueId((String) values.get("label"));
        
        try
        {
            _addCollection(id, values);
            return id;
        }
        catch (Exception e)
        {
            throw new ProcessingException("Failed to add new collection'" + id + "'", e);
        }
    }
    
    /**
     * Edit a {@link SynchronizableContentsCollection}
     * @param id The id of collection to edit
     * @param values The parameters' values
     * @return The id of new created collection or null in case of error
     * @throws ProcessingException if edition failed
     */
    @Callable (rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public Map<String, Object> editCollection (String id, Map<String, Object> values) throws ProcessingException
    {
        getLogger().debug("Edit Collection with id '{}' and values '{}'", id, values);
        Map<String, Object> result = new LinkedHashMap<>();
        
        SynchronizableContentsCollection collection = _synchronizableCollections.get(id);
        if (collection == null)
        {
            getLogger().error("The collection with id '{}' does not exist, it cannot be edited.", id);
            result.put("error", "unknown");
            return result;
        }
        else
        {
            _synchronizableCollections.remove(id);
        }
        
        try
        {
            _addCollection(id, values);
            result.put("id", id);
            return result;
        }
        catch (Exception e)
        {
            throw new ProcessingException("Failed to edit collection of id '" + id + "'", e);
        }
    }
    
    private boolean _isValid(SynchronizableContentsCollection collection)
    {
        // Check validation of a data source on its parameters
        
        SynchronizableContentsCollectionModel model = _syncCollectionModelEP.getExtension(collection.getSynchronizeCollectionModelId());
        if (model != null)
        {
            for (ModelItem param : model.getModelItems())
            {
                if (!_validateParameter(param, collection))
                {
                    // At least one parameter is invalid
                    return false;
                }
                
                if (ModelItemTypeConstants.DATASOURCE_ELEMENT_TYPE_ID.equals(param.getType().getId()))
                {
                    String dataSourceId = (String) collection.getParameterValues().get(param.getPath());
                    
                    if (!_checkDataSource(dataSourceId))
                    {
                        // At least one data source is not valid
                        return false;
                    }
                    
                }
            }
            
            return true;
        }
        
        return false; // no model found
    }
    
    private boolean _validateParameter(ModelItem modelItem, SynchronizableContentsCollection collection)
    {
        if (modelItem instanceof ElementDefinition)
        {
            ElementDefinition param = (ElementDefinition) modelItem;
            Validator validator = param.getValidator();
            if (validator != null)
            {
                Object value = collection.getParameterValues().get(param.getPath());
                
                return !validator.validate(value).hasErrors();
            }
        }
        
        return true;
    }
    
    private boolean _checkDataSource(String dataSourceId)
    {
        if (dataSourceId != null)
        {
            try
            {
                DataSourceDefinition def = _getSQLDataSourceManager().getDataSourceDefinition(dataSourceId);
                
                if (def != null)
                {
                    _getSQLDataSourceManager().checkParameters(def.getParameters());
                }
                else
                {
                    def = _getLDAPDataSourceManager().getDataSourceDefinition(dataSourceId);
                    if (def != null)
                    {
                        _getLDAPDataSourceManager().getDataSourceDefinition(dataSourceId);
                    }
                    else
                    {
                        // The data source was not found
                        return false;
                    }
                }
            }
            catch (ItemCheckerTestFailureException e)
            {
                // Connection to the SQL data source failed
                return false;
            }
        }
        
        return true;
    }
    
    private boolean _addCollection(String id, Map<String, Object> values) throws FileNotFoundException, IOException, TransformerConfigurationException, SAXException
    {
        File backup = _createBackup();
        boolean success = false;
        
        // Do writing
        try (OutputStream os = new FileOutputStream(__CONFIGURATION_FILE))
        {
            TransformerHandler th = _getTransformerHandler(os);

            // sax the config
            th.startDocument();
            XMLUtils.startElement(th, "collections");
            
            // SAX already existing collections
            _toSAX(th);
            
            // SAX the new collection
            _saxCollection(th, id, values);
            
            XMLUtils.endElement(th, "collections");
            th.endDocument();
            
            success = true;
        }
        
        _restoreBackup(backup, !success);
        
        _readFile(false);
        
        return success;
    }
    
    private TransformerHandler _getTransformerHandler(OutputStream os) throws TransformerConfigurationException
    {
        // create a transformer for saving sax into a file
        TransformerHandler th = ((SAXTransformerFactory) TransformerFactory.newInstance()).newTransformerHandler();
        
        StreamResult result = new StreamResult(os);
        th.setResult(result);

        // create the format of result
        Properties format = new Properties();
        format.put(OutputKeys.METHOD, "xml");
        format.put(OutputKeys.INDENT, "yes");
        format.put(OutputKeys.ENCODING, "UTF-8");
        format.put(OutputPropertiesFactory.S_KEY_INDENT_AMOUNT, "4");
        th.getTransformer().setOutputProperties(format);
        
        return th;
    }
    
    /**
     * Removes the given collection
     * @param id The id of the collection to remove
     * @return A map containing the id of the removed collection, or an error
     */
    @Callable (rights = "Runtime_Rights_Admin_Access", context = "/admin")
    public Map<String, Object> removeCollection(String id)
    {
        getLogger().debug("Remove Collection with id '{}'", id);
        Map<String, Object> result = new LinkedHashMap<>();
        
        _readFile(false);
        if (_synchronizableCollections.remove(id) == null)
        {
            getLogger().error("The synchronizable collection with id '{}' does not exist, it cannot be removed.", id);
            result.put("error", "unknown");
            return result;
        }
        
        if (_writeFile())
        {
            return null;
        }
        
        result.put("id", id);
        return result;
    }
    
    private String _generateUniqueId(String label)
    {
        // Id generated from name lowercased, trimmed, and spaces and underscores replaced by dashes
        String value = label.toLowerCase().trim().replaceAll("[\\W_]", "-").replaceAll("-+", "-").replaceAll("^-", "");
        int i = 2;
        String suffixedValue = value;
        while (_synchronizableCollections.get(suffixedValue) != null)
        {
            suffixedValue = value + i;
            i++;
        }
        
        return suffixedValue;
    }
    
    private void _toSAX(TransformerHandler handler) throws SAXException
    {
        for (SynchronizableContentsCollection collection : _synchronizableCollections.values())
        {
            _saxCollection(handler, collection);
        }
    }
    
    private void _saxCollection(ContentHandler handler, String id, Map<String, Object> parameters) throws SAXException
    {
        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("id", id);
        
        XMLUtils.startElement(handler, "collection", atts);
        
        String label = (String) parameters.get("label");
        if (label != null)
        {
            new I18nizableText(label).toSAX(handler, "label");
        }
        
        _saxNonNullValue(handler, "contentType", parameters.get("contentType"));
        _saxNonNullValue(handler, "contentPrefix", parameters.get("contentPrefix"));
        _saxNonNullValue(handler, "restrictedField", parameters.get("restrictedField"));
        _saxNonNullValue(handler, "synchronizeExistingContentsOnly", parameters.get("synchronizeExistingContentsOnly"));
        _saxNonNullValue(handler, "removalSync", parameters.get("removalSync"));
        _saxNonNullValue(handler, "ignoreRestrictions", parameters.get("ignoreRestrictions"));
        _saxNonNullValue(handler, "checkCollection", parameters.get("checkCollection"));
        _saxMultipleValues(handler, "compatibleSCC", parameters.get("compatibleSCC"));
        
        _saxNonNullValue(handler, "workflowName", parameters.get("workflowName"));
        _saxNonNullValue(handler, "initialActionId", parameters.get("initialActionId"));
        _saxNonNullValue(handler, "synchronizeActionId", parameters.get("synchronizeActionId"));
        _saxNonNullValue(handler, "validateActionId", parameters.get("validateActionId"));
        _saxNonNullValue(handler, "validateAfterImport", parameters.get("validateAfterImport"));
        
        _saxNonNullValue(handler, "reportMails", parameters.get("reportMails"));
        _saxNonNullValue(handler, "contentOperator", parameters.get("contentOperator"));
        
        _saxMultipleValues(handler, "languages", parameters.get("languages"));
        
        String modelId = (String) parameters.get("modelId");
        _saxModel(handler, modelId, parameters, true);
        
        XMLUtils.endElement(handler, "collection");
    }
    
    @SuppressWarnings("unchecked")
    private void _saxMultipleValues(ContentHandler handler, String tagName, Object values) throws SAXException
    {
        if (values != null)
        {
            XMLUtils.startElement(handler, tagName);
            for (String lang : (List<String>) values)
            {
                XMLUtils.createElement(handler, "value", lang);
            }
            XMLUtils.endElement(handler, tagName);
        }
    }

    private void _saxCollection(ContentHandler handler, SynchronizableContentsCollection collection) throws SAXException
    {
        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("id", collection.getId());
        XMLUtils.startElement(handler, "collection", atts);
        
        collection.getLabel().toSAX(handler, "label");
        
        _saxNonNullValue(handler, "contentType", collection.getContentType());
        _saxNonNullValue(handler, "contentPrefix", collection.getContentPrefix());
        _saxNonNullValue(handler, "restrictedField", collection.getRestrictedField());
        
        _saxNonNullValue(handler, "workflowName", collection.getWorkflowName());
        _saxNonNullValue(handler, "initialActionId", collection.getInitialActionId());
        _saxNonNullValue(handler, "synchronizeActionId", collection.getSynchronizeActionId());
        _saxNonNullValue(handler, "validateActionId", collection.getValidateActionId());
        _saxNonNullValue(handler, "validateAfterImport", collection.validateAfterImport());

        _saxNonNullValue(handler, "reportMails", collection.getReportMails());
        _saxNonNullValue(handler, "contentOperator", collection.getSynchronizingContentOperator());
        _saxNonNullValue(handler, "synchronizeExistingContentsOnly", collection.synchronizeExistingContentsOnly());
        _saxNonNullValue(handler, "removalSync", collection.removalSync());
        _saxNonNullValue(handler, "ignoreRestrictions", collection.ignoreRestrictions());
        _saxNonNullValue(handler, "checkCollection", collection.checkCollection());
        _saxMultipleValues(handler, "compatibleSCC", collection.getCompatibleSCC(false));
        
        _saxMultipleValues(handler, "languages", collection.getLanguages());
        
        _saxModel(handler, collection.getSynchronizeCollectionModelId(), collection.getParameterValues(), false);
        
        XMLUtils.endElement(handler, "collection");
    }
    
    private void _saxNonNullValue(ContentHandler handler, String tagName, Object value) throws SAXException
    {
        if (value != null)
        {
            XMLUtils.createElement(handler, tagName, value.toString());
        }
    }
    
    private void _saxModel(ContentHandler handler, String modelId, Map<String, Object> paramValues, boolean withPrefix) throws SAXException
    {
        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("id", modelId);
        XMLUtils.startElement(handler, "model", atts);
        
        SynchronizableContentsCollectionModel model = _syncCollectionModelEP.getExtension(modelId);
        String prefix = withPrefix ? modelId + SCC_PARAMETERS_SEPARATOR : StringUtils.EMPTY;
        for (ModelItem parameter : model.getModelItems())
        {
            String paramFieldName = prefix + parameter.getPath();
            Object value = paramValues.get(paramFieldName);
            if (value != null)
            {
                atts.clear();
                atts.addCDATAAttribute("name", parameter.getPath());
                XMLUtils.createElement(handler, "param", atts, ((ElementDefinition) parameter).getType().toString(value));
            }
        }
        
        XMLUtils.endElement(handler, "model");
    }
    
    @Override
    public void dispose()
    {
        _synchronizableCollections.clear();
        _lastFileReading = 0;
    }
}
