/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.clientsideelement;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.clientsideelement.SmartContentClientSideElement;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.core.ui.Callable;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionHelper;
import org.ametys.plugins.core.ui.log.LogManager;
import org.ametys.runtime.authentication.AccessDeniedException;

/**
 * Smart content client side element for SCC, the SCC model ID configured in plugin.xml is used.
 */
public class SCCSmartContentClientSideElement extends SmartContentClientSideElement
{
    /** The log manager */
    protected LogManager _logManager;
    /** SCC helper */
    protected SynchronizableContentsCollectionHelper _sccHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _logManager = (LogManager) smanager.lookup(LogManager.ROLE);
        _sccHelper = (SynchronizableContentsCollectionHelper) smanager.lookup(SynchronizableContentsCollectionHelper.ROLE);
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        List<Script> clonedScripts = new ArrayList<>();
        
        List<Script> scripts = super.getScripts(ignoreRights, contextParameters);
        
        for (Script script : scripts)
        {
            if (script.getParameters().containsKey("sccModelId"))
            {
                String sccModelId = (String) script.getParameters().get("sccModelId");
                SynchronizableContentsCollection collection = _sccHelper.getSCCFromModelId(sccModelId);
                if (collection != null)
                {
                    clonedScripts.add(_getScriptFromCollection(script, collection));
                }
            }
        }
        
        return clonedScripts;
    }

    /**
     * Get a cloned script by updating configuration with collection values.
     * @param script The script to clone
     * @param collection The collection
     * @return A cloned script
     */
    protected Script _getScriptFromCollection(Script script, SynchronizableContentsCollection collection)
    {
        Script clonedScript = new Script(script);
        clonedScript.getParameters().put("collectionId", collection.getId());
        return clonedScript;
    }
    
    /**
     * Synchronize the content on the given collection with the given synchronization code.
     * @param collectionId Collection ID
     * @param contentId Content ID
     * @param syncCode Synchronization code
     * @return true if an error occurred
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public boolean synchronizeContent(String collectionId, String contentId, String syncCode)
    {
        ModifiableDefaultContent content = _resolver.resolveById(contentId);
        if (!_hasRight(content))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to synchronize content '" + contentId + "' without sufficient right");
        }
        
        return _sccHelper.synchronizeContent(collectionId, contentId, syncCode);
    }
    
    /**
     * Get the value of the synchronization field.
     * @param collectionId Collection ID
     * @param contentId Content ID
     * @return The value of the synchronization field
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public String getSyncCode(String contentId, String collectionId)
    {
        ModifiableDefaultContent content = _resolver.resolveById(contentId);
        if (!_hasRight(content))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to get synchronization code of content '" + contentId + "' without sufficient right");
        }
        
        return _sccHelper.getSyncCode(contentId, collectionId);
    }
    
    /**
     * Get event logs
     * @param timestamp Events after this timestamp will be retrieved
     * @param categories Events will be filtered by these categories. If empty, all categories configured by the client side are accepted.
     * @return the target types
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> getEvents (Long timestamp, List<String> categories)
    {
        if (!hasRight(getRights(Map.of())))
        {
            throw new AccessDeniedException("The user '" + _currentUserProvider.getUser() + "' try to get logs without sufficient rights");
        }
        
        return _logManager.getEvents(timestamp, List.of("org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionHelper"));
    }
}
