/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.in.xml;

import java.io.IOException;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.core.util.dom.DOMUtils;
import org.ametys.plugins.repository.data.extractor.xml.ModelAwareXMLValuesExtractor;

/**
 * Default implementation of an XML content importer.
 */
public class DefaultXmlContentImporter extends AbstractXmlContentImporter
{
    
    /** Default content types key in the import map. */
    protected static final String _DEFAULT_CONTENT_TYPES_KEY = DefaultXmlContentImporter.class.getName() + "$defaultContentTypes";
    /** Default content mixins key in the import map. */
    protected static final String _DEFAULT_CONTENT_MIXINS_KEY = DefaultXmlContentImporter.class.getName() + "$defaultMixins";
    /** Default content language key in the import map. */
    protected static final String _DEFAULT_CONTENT_LANG_KEY = DefaultXmlContentImporter.class.getName() + "$defaultLanguage";
    
    /** The content type helper. */
    protected ContentTypesHelper _cTypeHelper;
    
    /** The XPath expression to match. */
    protected String _matchPath;
    
    /** The XPath value to match. */
    protected String _matchValue;
    
    /** The XPath value to match. */
    protected Pattern _matchRegex;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _cTypeHelper = (ContentTypesHelper) serviceManager.lookup(ContentTypesHelper.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        
        configureXmlMatch(configuration.getChild("xml").getChild("match"));
    }
    
    /**
     * Configure the matching value.
     * @param configuration the matching configuration.
     * @throws ConfigurationException if an error occurs.
     */
    protected void configureXmlMatch(Configuration configuration) throws ConfigurationException
    {
        _matchPath = configuration.getAttribute("path");
        _matchValue = configuration.getAttribute("value", null);
        String regex = configuration.getAttribute("regex", null);
        if (StringUtils.isNotBlank(regex))
        {
            _matchRegex = Pattern.compile(regex);
        }
    }
    
    @Override
    public boolean supports(Document document) throws IOException
    {
        Node node = _xPathProcessor.selectSingleNode(document, _matchPath, getPrefixResolver());
        
        if (_matchValue == null && _matchRegex == null)
        {
            return node != null;
        }
        else if (_matchRegex != null)
        {
            String text = getTextContent(node, null, true);
            return text != null ? _matchRegex.matcher(text).matches() : false;
        }
        else
        {
            return _matchValue.equals(getTextContent(node, null, true));
        }
    }
    
    @Override
    protected Set<String> importContents(Document node, Map<String, Object> params) throws IOException
    {
        Set<String> contentIds = new HashSet<>();
        
        Element root = node.getDocumentElement();
        
        if (root != null)
        {
            String[] defaultTypes = StringUtils.split(getAttributeValue(root, "default-types", ""), ", ");
            String[] defaultMixins = StringUtils.split(getAttributeValue(root, "default-mixins", ""), ", ");
            String defaultLang = getAttributeValue(root, "default-language", getLanguage(params));
            if (defaultTypes.length == 0)
            {
                defaultTypes = getContentTypes(params);
            }
            if (defaultMixins.length == 0)
            {
                defaultMixins = getMixins(params);
            }
            
            params.put(_DEFAULT_CONTENT_TYPES_KEY, defaultTypes);
            params.put(_DEFAULT_CONTENT_MIXINS_KEY, defaultMixins);
            params.put(_DEFAULT_CONTENT_LANG_KEY, defaultLang);
            
            // Import all the contents.
            NodeList contents = _xPathProcessor.selectNodeList(root, "content", getPrefixResolver());
            for (int i = 0; i < contents.getLength(); i++)
            {
                Element contentNode = (Element) contents.item(i);
                
                try
                {
                    Content content = importContent(contentNode, defaultTypes, defaultMixins, defaultLang, params);
                    
                    if (content != null)
                    {
                        contentIds.add(content.getId());
                    }
                }
                catch (Exception e)
                {
                    getLogger().error("Error importing a content.", e);
                }
            }
            
            params.remove(_DEFAULT_CONTENT_TYPES_KEY);
            params.remove(_DEFAULT_CONTENT_MIXINS_KEY);
            params.remove(_DEFAULT_CONTENT_LANG_KEY);
        }
        
        return contentIds;
    }
    
    /**
     * Import a content from a XML node.
     * @param contentElement the content XML node.
     * @param defaultTypes the default content types.
     * @param defaultMixins the default mixins.
     * @param defaultLang the default content language.
     * @param params the import parameters.
     * @return the Content or null if not created.
     * @throws Exception if an error occurs creating the Content.
     */
    protected Content importContent(Element contentElement, String[] defaultTypes, String[] defaultMixins, String defaultLang, Map<String, Object> params) throws Exception
    {
        String localId = getAttributeValue(contentElement, "id", "");
        
        String cTypesStr = getAttributeValue(contentElement, "types", "");
        String mixinsStr = getAttributeValue(contentElement, "mixins", "");
        
        String[] contentTypes = StringUtils.isEmpty(cTypesStr) ? defaultTypes : StringUtils.split(cTypesStr, ", ");
        String[] contentMixins = StringUtils.isEmpty(mixinsStr) ? defaultMixins : StringUtils.split(mixinsStr, ", ");
        String language = getAttributeValue(contentElement, "language", defaultLang);
        
        String title = _xPathProcessor.evaluateAsString(contentElement, "@title", getPrefixResolver());
        
        WorkflowAwareContent content = (WorkflowAwareContent) createContent(title, contentTypes, contentMixins, language, params);
        
        ModelAwareXMLValuesExtractor extractor = new ModelAwareXMLValuesExtractor(DOMUtils.getChildElementByTagName(contentElement, "attributes"), (path, type) -> Optional.empty(), content.getModel());
        Map<String, Object> values = extractor.extractValues();
        _contentWorkflowHelper.editContent(content, values, getEditActionId(params));
        
        // If the content contains a "local" ID (local to the import file), remember it.
        if (StringUtils.isNotBlank(localId))
        {
            Map<String, String> contentIdMap = getContentIdMap(params);
            contentIdMap.put(localId, content.getId());
        }
        
        return content;
    }
}
