/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.impl;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;

import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;

/**
 * Implementation of {@link SynchronizableContentsCollection} to be synchronized with a LDAP data source
 */
public class SQLSynchronizableContentsCollection extends AbstractDataSourceSynchronizableContentsCollection implements Component
{
    private static final String __PARAM_SQL_TABLE = "tableName";
    
    /** The SQL collection DAO */
    protected SQLCollectionDAO _sqlCollectionDAO;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _sqlCollectionDAO = (SQLCollectionDAO) smanager.lookup(SQLCollectionDAO.ROLE);
    }
    
    /**
     * Get the name of SQL table
     * @return The SQL table name
     */
    public String getTableName()
    {
        return (String) getParameterValues().get(__PARAM_SQL_TABLE);
    }
    
    @Override
    protected Map<String, Map<String, Object>> internalSearch(Map<String, Object> searchParameters, int offset, int limit, List<Object> sort, Logger logger)
    {
        Map<String, Map<String, Object>> result = new LinkedHashMap<>();
        
        Map<String, List<String>> mapping = getMapping();
        String idField = getIdField();
        List<String> remoteKeys = mapping.get(idField);
        if (CollectionUtils.isNotEmpty(remoteKeys))
        {
            String remoteKey = remoteKeys.get(0);
            
            List<String> columns = mapping.values()
                                          .stream()
                                          .flatMap(List::stream)
                                          .collect(Collectors.toList());

            Map<String, Object> params = _getSearchParameters(searchParameters, offset, limit, sort, columns);
            List<Map<String, Object>> searchResults = _sqlCollectionDAO.search(params , getDataSourceId());
            result  = _normalizeSearchResult(remoteKey, columns, searchResults, logger);
        }
        else
        {
            _nbError++;
            logger.error("Missing SQL attribute in mapping for field '{}' holding the unique identifier", idField);
        }
        
        return result;
    }

    /**
     * Get the parameters map for mybatis search
     * @param parameters the filter parameter
     * @param offset the offset
     * @param limit the limit
     * @param sort the sort map
     * @param columns the list of columns
     * @return the parameter map
     */
    protected Map<String, Object> _getSearchParameters(Map<String, Object> parameters, int offset, int limit, List<Object> sort, List<String> columns)
    {
        Map<String, Object> params = new HashMap<>();
        params.put("table", getTableName());
        params.put("columns", columns);
        params.put("params", parameters);
        
        if (offset > 0)
        {
            params.put("offset", offset);
        }
        
        if (limit < Integer.MAX_VALUE)
        {
            params.put("limit", limit);
        }
        
        if (CollectionUtils.isNotEmpty(sort))
        {
            params.put("sorts", sort);
        }
        
        return params;
    }

    /**
     * We need to normalize the search result for each database type
     * For example, with Oracle, the returned SQL column names are upper-cased
     * so we need to retrieve for each returned column its real name for the mapping, 
     * @param remoteKey the remote key handling the id
     * @param columns the list of columns
     * @param searchResults the search result map
     * @param logger the logger
     * @return the normalize search result map
     */
    protected Map<String, Map<String, Object>> _normalizeSearchResult(String remoteKey, List<String> columns, List<Map<String, Object>> searchResults, Logger logger)
    {
        Map<String, Map<String, Object>> result = new LinkedHashMap<>();
        
        for (Map<String, Object> searchResult : searchResults)
        {
            Map<String, Object> newSearchResult = _getNormalizedSearchResult(columns, searchResult);
            
            Object idObjectValue = newSearchResult.get(remoteKey);
            if (_checkIdObjectValue(remoteKey, idObjectValue, logger))
            {
                newSearchResult.put(SCC_UNIQUE_ID, idObjectValue.toString());
                newSearchResult.put(remoteKey, idObjectValue.toString());
                result.put(idObjectValue.toString(), newSearchResult);
            }
        }
        
        return result;
    }
    
    /**
     * Check if the id object value is correct
     * @param remoteKey the remove key
     * @param idObjectValue the id object value
     * @param logger the logger
     * @return true if the id object value is correct
     */
    protected boolean _checkIdObjectValue(String remoteKey, Object idObjectValue, Logger logger)
    {
        if (idObjectValue == null)
        {
            logger.warn("The content identifier is mandatory but there is no value for the key : " + remoteKey);
            return false;
        }
        
        return true;
    }
    
    /**
     * Get normalized search result
     * Indeed we need to normalize the search result for each database type
     * For example, with Oracle, the returned SQL column names are uppercased
     * so we need to retrieve for each returned column its real name for the mapping, 
     * @param columns the columns list
     * @param searchResult the search result
     * @return the normalized search result
     */
    protected Map<String, Object> _getNormalizedSearchResult(List<String> columns, Map<String, Object> searchResult)
    {
        Map<String, Object> normalizedSearchResult = new HashMap<>();
        for (String key : searchResult.keySet())
        {
            //We search the initial column name
            List<String> filteredValue = columns.stream()
                                                .filter(c -> c.equalsIgnoreCase(key))
                                                .collect(Collectors.toList());
            if (!filteredValue.isEmpty()) //If we find no column, we ignore it (for example Oracle send 'Line' value) 
            {
                normalizedSearchResult.put(filteredValue.get(0), searchResult.get(key));
            }
        }
        return normalizedSearchResult;
    }
    
    @Override
    public int getTotalCount(Map<String, Object> parameters, Logger logger)
    {
        Map<String, Object> params = _getTotalCounParameters(parameters);
        return _sqlCollectionDAO.getTotalCount(params , getDataSourceId());
    }
    
    /**
     * Get the parameters map for mybatis total count
     * @param parameters the filter parameter
     * @return the parameter map
     */
    protected Map<String, Object> _getTotalCounParameters(Map<String, Object> parameters)
    {
        Map<String, Object> params = new HashMap<>();
        params.put("table", getTableName());
        params.put("params", _removeEmptyParameters(parameters));

        return params;
    }
    
}
