/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.search;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

import com.google.common.collect.ImmutableMap;

/**
 * Search model configuration for SCC import/search tool
 */
public class SCCSearchModelConfiguration
{
    /** Criteria to mask imported elements in SCCSearchTool */
    public static final String CRITERIA_MASK_IMPORTED = "maskImported";
    
    private static final Map<String, Object> __MASK_IMPORTED_CRITERION = ImmutableMap.of(
            "id", CRITERIA_MASK_IMPORTED, 
            "label", new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_UITOOL_SEARCH_MASK_IMPORTED"), 
            "type", "BOOLEAN"
        );
    
    private boolean _displayMaskImported = true;
    private Map<String, Map<String, Object>> _criteria = new LinkedHashMap<>();
    private List<Map<String, Object>> _columns = new LinkedList<>();

    /**
     * Set to <code>false</code> if you don't want to display the maskImported criteria.
     * @param displayMaskImported <code>false</code> if you want to disable the maskImportedCriteria
     */
    public void displayMaskImported(boolean displayMaskImported)
    {
        _displayMaskImported = displayMaskImported;
    }
    
    /**
     * Add a criterion to the model.
     * @param id ID
     */
    public void addCriterion(String id)
    {
        addCriterion(id, new I18nizableText(id));
    }
    
    /**
     * Add a criterion to the model.
     * @param id ID
     * @param label Internationalized label
     */
    public void addCriterion(String id, I18nizableText label)
    {
        addCriterion(id, label, "STRING");
    }

    /**
     * Add a criterion to the model.
     * @param id ID
     * @param label Internationalized label
     * @param type Type (string, boolean, etc.)
     */
    public void addCriterion(String id, I18nizableText label, String type)
    {
        addCriterion(id, label, type, null);
    }

    /**
     * Add a criterion to the model.
     * @param id ID
     * @param label Internationalized label
     * @param type Type (string, boolean, etc.)
     * @param widget The widget to display
     */
    public void addCriterion(String id, I18nizableText label, String type, String widget)
    {
        Map<String, Object> criterion = new HashMap<>();
        criterion.put("id", id);
        criterion.put("label", label);
        criterion.put("type", type);
        criterion.put("widget", widget);
        _criteria.put(id, criterion);
    }

    /**
     * Add a column to the model.
     * @param id ID
     */
    public void addColumn(String id)
    {
        addColumn(id, new I18nizableText(id));
    }
    
    /**
     * Add a column to the model.
     * @param id ID
     * @param label Internationalized label
     */
    public void addColumn(String id, I18nizableText label)
    {
        addColumn(id, label, 150);
    }
    
    /**
     * Add a column to the model.
     * @param id ID
     * @param label Internationalized label
     * @param width Width of the column
     */
    public void addColumn(String id, I18nizableText label, int width)
    {
        addColumn(id, label, width, true);
    }

    /**
     * Add a column to the model.
     * @param id ID
     * @param label Internationalized label
     * @param sortable true if sortable
     */
    public void addColumn(String id, I18nizableText label, boolean sortable)
    {
        addColumn(id, label, 150, sortable);
    }
    
    /**
     * Add a column to the model.
     * @param id ID
     * @param label Internationalized label
     * @param width Width of the column
     * @param sortable true if sortable
     */
    public void addColumn(String id, I18nizableText label, int width, boolean sortable)
    {
        addColumn(id, label, width, sortable, ModelItemTypeConstants.STRING_TYPE_ID);
    }

    /**
     * Add a column to the model.
     * @param id ID
     * @param label Internationalized label
     * @param width Width of the column
     * @param sortable true if sortable
     * @param type Type (string, boolean, etc.)
     */
    public void addColumn(String id, I18nizableText label, int width, boolean sortable, String type)
    {
        String renderer = null;
        if (_displayMaskImported && _columns.isEmpty())
        {
            renderer = "Ametys.plugins.contentio.search.SCCSearchTool.renderImported";
        }
        addColumn(id, label, width, sortable, type, renderer);
    }

    /**
     * Add a column to the model.
     * @param id ID
     * @param label Internationalized label
     * @param width Width of the column
     * @param sortable true if sortable
     * @param type Type (string, boolean, etc.)
     * @param renderer The render JS function
     */
    public void addColumn(String id, I18nizableText label, int width, boolean sortable, String type, String renderer)
    {
        Map<String, Object> column = new HashMap<>();
        column.put("path", id);
        column.put("label", label);
        column.put("sortable", sortable);
        column.put("type", type);
        column.put("width", width);
        column.put("renderer", renderer);
        
        _columns.add(column);
    }

    /**
     * Convert criteria and columns to JSON Map and add searchUrlPlugin, searchUrl and pageSize to JSON results.
     * @return A Map to be convert as JSON
     */
    public Map<String, Object> toJSON()
    {
        Map<String, Object> json = new HashMap<>();
        Map<String, Map<String, Object>> displayedCriteria = new LinkedHashMap<>(_criteria);
        if (_displayMaskImported)
        {
            displayedCriteria.put(CRITERIA_MASK_IMPORTED, __MASK_IMPORTED_CRITERION);
        }
        
        json.put("criteria", ImmutableMap.of(UUID.randomUUID().toString(), ImmutableMap.of("elements", displayedCriteria, "role", "fieldset")));
        json.put("columns", _columns);
        json.put("searchUrlPlugin", "contentio");
        json.put("searchUrl", "scc-search/list.json");
        json.put("pageSize", 50);
        return json;
    }
}
