/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize;

import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.data.external.ExternalizableDataProvider;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * {@link ExternalizableDataProvider} based on {@link SynchronizableContentsCollection}
 */
public class SynchronizableContentsCollectionDataProvider extends AbstractLogEnabled implements ExternalizableDataProvider, Component, Serviceable
{
    /** Constant for storing the scc identifier into the externalizable data context */
    public static final String SCC_ID_CONTEXT_KEY = "sccId";
    
    /** The DAO for synchronizable contents collections */
    protected SynchronizableContentsCollectionDAO _synchronizableContentsCollectionDAO;
    /** SCC helper */
    protected SynchronizableContentsCollectionHelper _sccHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _synchronizableContentsCollectionDAO = (SynchronizableContentsCollectionDAO) manager.lookup(SynchronizableContentsCollectionDAO.ROLE);
        _sccHelper = (SynchronizableContentsCollectionHelper) manager.lookup(SynchronizableContentsCollectionHelper.ROLE);
    }

    public Set<String> getExternalizableDataPaths(ModelAwareDataHolder dataHolder)
    {
        if (dataHolder instanceof Content)
        {
            Content content = (Content) dataHolder;
            return _sccHelper.getSynchronizableCollectionIds(content)
                    .stream()
                    .map(_synchronizableContentsCollectionDAO::getSynchronizableContentsCollection)
                    .filter(collection -> collection != null)
                    .map(collection -> _getExternalizableDataPaths(dataHolder, collection))
                    .flatMap(Collection::stream)
                    .collect(Collectors.toSet());
        }
        else
        {
            return Set.of();
        }
    }
    
    public boolean isDataExternalizable(ModelAwareDataHolder dataHolder, ModelItem modelItem, Map<String, Object> context)
    {
        Set<String> externalizableDataPaths = Optional.of(context)
                .map(ctx -> ctx.get(SCC_ID_CONTEXT_KEY))
                .filter(String.class::isInstance)
                .map(String.class::cast)
                .map(_synchronizableContentsCollectionDAO::getSynchronizableContentsCollection)
                .map(col -> _getExternalizableDataPaths(dataHolder, col))
                .orElseGet(() -> getExternalizableDataPaths(dataHolder));
        
        return externalizableDataPaths.contains(modelItem.getPath());
    }
    
    private Set<String> _getExternalizableDataPaths(ModelAwareDataHolder dataHolder, SynchronizableContentsCollection collection)
    {
        return dataHolder instanceof Content ? collection.getLocalAndExternalFields(buildParametersMap((Content) dataHolder)) : Set.of();
    }
    
    private Map<String, Object> buildParametersMap(Content content)
    {
        return Map.of("contentTypes", Arrays.asList(content.getTypes()));
    }

}
