/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.rights;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.Content;
import org.ametys.core.right.AbstractStaticRightAssignmentContext;
import org.ametys.core.right.RightAssignmentContext;
import org.ametys.core.ui.ClientSideElementHelper;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDAO;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;

/**
 * {@link RightAssignmentContext} for assign rights to a {@link Content} or a jcr node root holding the contents
 */
public class SynchronizeContentRightAssignmentContext extends AbstractStaticRightAssignmentContext
{
    /** The prefix for rights on the root of a collection */
    public static final String ROOT_CONTEXT_PREFIX = "/synchronized-contents/";

    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The synchronize collection DAO */
    protected SynchronizableContentsCollectionDAO _collectionsDAO;
    /** The i18n utils */
    protected I18nUtils _i18nUtils;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _collectionsDAO = (SynchronizableContentsCollectionDAO) smanager.lookup(SynchronizableContentsCollectionDAO.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    public Object convertJSContext(Object context)
    {
        if (context instanceof String)
        {
            String stringContext = (String) context;
            if (stringContext.startsWith(ROOT_CONTEXT_PREFIX))
            {
                return context;
            }
            else
            {
                return _resolver.resolveById((String) context);
            }
        }

        return null;
    }
    
    @Override
    public String getContextIdentifier(Object context)
    {
        if (context instanceof Content)
        {
            return ((AmetysObject) context).getId();
        }
        else
        {
            return (String) context;
        }
    }
    
    @Override
    public Set<Object> getParentContexts(Object context)
    {
        if (context instanceof Content)
        {
            String[] collectionIds = ((Content) context).getInternalDataHolder().getValue(SynchronizableContentsCollection.COLLECTION_ID_DATA_NAME);
            return Set.of(ROOT_CONTEXT_PREFIX + collectionIds[0]);
        }
        
        return null;
    }
    
    @Override
    public List<Object> getRootContexts(Map<String, Object> contextParameters)
    {
        List<Object> rootContexts = new ArrayList<>();
        if (matchWorkspace(contextParameters))
        {
            for (SynchronizableContentsCollection synchronizableContentsCollection : _collectionsDAO.getSynchronizableContentsCollections())
            {
                if (synchronizableContentsCollection.handleRightAssignmentContext())
                {
                    rootContexts.add(ROOT_CONTEXT_PREFIX + synchronizableContentsCollection.getId());
                }
            }
        }
        return rootContexts;
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        List<Script> scripts = new ArrayList<>();
        
        List<Script> superScripts = super.getScripts(ignoreRights, contextParameters);
        if (superScripts != null && superScripts.size() == 1)
        {
            for (SynchronizableContentsCollection synchronizableContentsCollection : _collectionsDAO.getSynchronizableContentsCollections())
            {
                if (synchronizableContentsCollection.handleRightAssignmentContext())
                {
                    // First duplicate the script
                    Script script = ClientSideElementHelper.cloneScript(superScripts.get(0));
                    
                    script.getParameters().put("root-context", ROOT_CONTEXT_PREFIX + synchronizableContentsCollection.getId());
                    script.getParameters().put("scc", synchronizableContentsCollection.getId());
                    
                    _parametrizeValue(script, "label", synchronizableContentsCollection.getLabel());
                    _parametrizeValue(script, "radio-option-all-label", synchronizableContentsCollection.getLabel());
                    _parametrizeValue(script, "hint-all-contents", synchronizableContentsCollection.getLabel());
                    _parametrizeValue(script, "result-grid-mask-message", synchronizableContentsCollection.getLabel());
                    
                    scripts.add(script);
                }
            }
        }

        return scripts;
    }
    
    private void _parametrizeValue(Script script, String key, I18nizableText i18nKey)
    {
        Object label = script.getParameters().get(key);
        if (label == null)
        {
            script.getParameters().put(key, i18nKey);
        }
        else if (label instanceof String)
        {
            script.getParameters().put(key, label + " " + _i18nUtils.translate(i18nKey));
        }
        else if (label instanceof I18nizableText)
        {
            I18nizableText i18nLabel = (I18nizableText) label;
            if (i18nLabel.isI18n())
            {
                if (i18nLabel.getParameters() != null)
                {
                    List<String> parameters = new ArrayList<>(i18nLabel.getParameters());
                    parameters.add(_i18nUtils.translate(i18nKey));
                    script.getParameters().put(key, new I18nizableText(i18nLabel.getCatalogue(), i18nLabel.getKey(), parameters));
                }
                else
                {
                    Map<String, I18nizableTextParameter> parametersMap = i18nLabel.getParameterMap() != null ? new HashMap<>(i18nLabel.getParameterMap()) : new HashMap<>();
                    parametersMap.put("collection", i18nKey);
                    script.getParameters().put(key, new I18nizableText(i18nLabel.getCatalogue(), i18nLabel.getKey(), parametersMap));
                }
            }
            else
            {
                script.getParameters().put(key, i18nLabel.getLabel() + " " + _i18nUtils.translate(i18nKey));
            }
        }
    }
}
