/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.in;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.servlet.multipart.Part;
import org.apache.cocoon.servlet.multipart.PartOnDisk;
import org.apache.cocoon.servlet.multipart.RejectedPart;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;

import org.ametys.core.util.HttpUtils;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.contentio.in.ContentImportManager.ImportResult;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

import jakarta.mail.MessagingException;

/**
 * Import contents from an uploaded file.
 */
public class ImportFileAction extends ServiceableAction
{
    
    /** The content import manager. */
    protected ContentImportManager _importManager;
    
    /** The source resolver. */
    protected org.apache.excalibur.source.SourceResolver _sourceResolver;
    
    /** The i18n utils. */
    protected I18nUtils _i18nUtils;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _importManager = (ContentImportManager) serviceManager.lookup(ContentImportManager.ROLE);
        _sourceResolver = (org.apache.excalibur.source.SourceResolver) serviceManager.lookup(org.apache.excalibur.source.SourceResolver.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        boolean sendMail = parameters.getParameterAsBoolean("send-mail", false);
        
        boolean success = false;
        
        Map<String, String> result = new HashMap<>();
        
        ImportResult importResult = null;
        
        Part part = (Part) request.get("file");
        if (part instanceof RejectedPart || part == null)
        {
            result.put("error", "rejected");
        }
        else
        {
            PartOnDisk uploadedFilePart = (PartOnDisk) part;
            File uploadedFile = uploadedFilePart.getFile();
            
            try
            {
                importResult = _importManager.importContents(uploadedFile);
                
                if (!importResult.isImporterFound())
                {
                    result.put("error", "no-importer");
                }
                else
                {
                    Set<String> contentIds = importResult.getImportedContentIds();
                    success = true;
                    
                    result.put("importedCount", Integer.toString(contentIds.size()));
                    result.put("importedIds", StringUtils.join(contentIds, '|'));
                }
            }
            catch (ContentImportException e)
            {
                ByteArrayOutputStream os = new ByteArrayOutputStream();
                e.printStackTrace(new PrintStream(os));
                getLogger().error("Exception importing contents from file: " + uploadedFile.getName(), e);
                result.put("message", os.toString());
            }
        }
        
        result.put("success", String.valueOf(success));
        
        if (sendMail)
        {
            sendMail(success, importResult);
        }
        
        return result;
    }
    
    /**
     * Send an import report as e-mail.
     * @param success true if the import was successful, false otherwise.
     * @param importResult the import result.
     */
    protected void sendMail(boolean success, ImportResult importResult)
    {
        try
        {
            String sysadminMail = Config.getInstance().getValue("smtp.mail.sysadminto");
            String baseUrl = HttpUtils.sanitize(Config.getInstance().getValue("cms.url"));
            
            Map<String, Object> mailParams = new HashMap<>();
            
            if (success)
            {
                mailParams.put("success", true);
                mailParams.put("baseUrl", baseUrl);
                mailParams.put("contentIds", importResult.getImportedContentIds());
            }
            else
            {
                mailParams.put("success", false);
                if (importResult != null && !importResult.isImporterFound())
                {
                    mailParams.put("error", "no-importer");
                }
            }
            
            Source bodySource = _sourceResolver.resolveURI("cocoon://_plugins/contentio/mail/body", null, mailParams);
            String body = IOUtils.toString(bodySource.getInputStream(), StandardCharsets.UTF_8);
            
            I18nizableText i18nSubject = new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_IMPORTFILE_MAIL_SUBJECT");
            String subject = _i18nUtils.translate(i18nSubject);
            
            if (StringUtils.isNotBlank(sysadminMail))
            {
                SendMailHelper.newMail()
                              .withSubject(subject)
                              .withTextBody(body)
                              .withRecipient(sysadminMail)
                              .sendMail();
            }
        }
        catch (MessagingException | IOException e)
        {
            getLogger().warn("Error sending the import report e-mail.", e);
        }
    }
    
}
