/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.schedule.progression.ProgressionTrackerFactory;
import org.ametys.core.schedule.progression.SimpleProgressionTracker;
import org.ametys.core.util.MapUtils;
import org.ametys.plugins.contentio.synchronize.workflow.EditSynchronizedContentFunction;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.data.external.ExternalizableDataProvider.ExternalizableDataStatus;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.data.holder.values.SynchronizationContext;
import org.ametys.plugins.repository.data.holder.values.ValueContext;
import org.ametys.plugins.repository.version.VersionableAmetysObject;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.View;

import com.opensymphony.workflow.WorkflowException;

/**
 * Abstract implementation of {@link SynchronizableContentsCollection}.
 */
public abstract class AbstractSimpleSynchronizableContentsCollection extends AbstractSynchronizableContentsCollection
{
    /** The extension point for Synchronizing Content Operators */
    protected SynchronizingContentOperatorExtensionPoint _synchronizingContentOperatorEP;
    
    /** The content helper */
    protected ContentHelper _contentHelper;
    
    private List<String> _handledContents;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _contentHelper = (ContentHelper) manager.lookup(ContentHelper.ROLE);
        _synchronizingContentOperatorEP = (SynchronizingContentOperatorExtensionPoint) manager.lookup(SynchronizingContentOperatorExtensionPoint.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        _handledContents = new ArrayList<>();
    }

    @Override
    public List<ModifiableContent> populate(Logger logger, ContainerProgressionTracker progressionTracker)
    {
        _handledContents.clear();
        List<ModifiableContent> populatedContents = super.populate(logger, progressionTracker);
        _handledContents.clear();
        return populatedContents;
    }
    
    @Override
    protected List<ModifiableContent> _internalPopulate(Logger logger, ContainerProgressionTracker progressionTracker)
    {
        return _importOrSynchronizeContents(new HashMap<>(), false, logger, progressionTracker);
    }
    
    /**
     * Adds the given content as handled (i.e. will not be removed if _removalSync is true)
     * @param id The id of the content
     */
    protected void _handleContent(String id)
    {
        _handledContents.add(id);
    }
    
    /**
     * Returns true if the given content is handled
     * @param id The content to test
     * @return true if the given content is handled
     */
    protected boolean _isHandled(String id)
    {
        return _handledContents.contains(id);
    }
    
    /**
     * Imports or synchronizes a content for each available language
     * @param idValue The unique identifier of the content
     * @param remoteValues The remote values
     * @param forceImport To force import and ignoring the synchronize existing contents only option
     * @param logger The logger
     * @return The list of synchronized or imported contents
     */
    protected List<ModifiableContent> _importOrSynchronizeContent(String idValue, Map<String, List<Object>> remoteValues, boolean forceImport, Logger logger)
    {
        List<ModifiableContent> contents = new ArrayList<>();
        
        for (String lang : getLanguages())
        {
            _importOrSynchronizeContent(idValue, lang, remoteValues, forceImport, logger)
                .ifPresent(contents::add);
        }
        
        return contents;
    }
    
    /**
     * Imports or synchronizes a content for a given language
     * @param idValue The unique identifier of the content
     * @param lang The language of content to import or synchronize
     * @param remoteValues The remote values
     * @param forceImport To force import and ignoring the synchronize existing contents only option
     * @param logger The logger
     * @return The imported or synchronized content
     */
    protected Optional<ModifiableContent> _importOrSynchronizeContent(String idValue, String lang, Map<String, List<Object>> remoteValues, boolean forceImport, Logger logger)
    {
        try
        {
            ModifiableContent content = getContent(lang, idValue, false);
            if (content != null)
            {
                return Optional.of(_synchronizeContent(content, remoteValues, logger));
            }
            else if (forceImport || !synchronizeExistingContentsOnly())
            {
                return Optional.ofNullable(_importContent(idValue, null, lang, remoteValues, logger));
            }
        }
        catch (Exception e)
        {
            _nbError++;
            logger.error("An error occurred while importing or synchronizing content", e);
        }
        
        return Optional.empty();
    }

    @Override
    public void synchronizeContent(ModifiableContent content, Logger logger) throws Exception
    {
        String idValue = content.getValue(getIdField());
        
        Map<String, Object> searchParameters = putIdParameter(idValue);
        Map<String, Map<String, List<Object>>> results = getTransformedRemoteValues(searchParameters, logger);
        if (!results.isEmpty())
        {
            try
            {
                _synchronizeContent(content, results.get(idValue), logger);
            }
            catch (Exception e)
            {
                _nbError++;
                logger.error("An error occurred while importing or synchronizing content", e);
                throw e;
            }
        }
        else
        {
            logger.warn("The content {} ({}) with synchronization code '{}' doesn't exist anymore in the datasource from SCC '{}'", content.getTitle(), content.getId(), idValue, getId());
        }
    }
    /**
     * Synchronize a content with remove values.
     * @param content The content to synchronize
     * @param remoteValues Values to synchronize
     * @param logger The logger
     * @return The synchronized content
     * @throws Exception if an error occurs
     */
    protected ModifiableContent _synchronizeContent(ModifiableContent content, Map<String, List<Object>> remoteValues, Logger logger) throws Exception
    {
        long startTime = System.currentTimeMillis();
        
        String contentTitle = content.getTitle();
        String lang = content.getLanguage();
        
        // Update content
        logger.info("Start synchronizing content '{}' for language {}", contentTitle, lang);

        boolean fromNewSCC = !_sccHelper.getSynchronizableCollectionIds(content).contains(getId());
        if (fromNewSCC)
        {
            _sccHelper.updateSCCProperty(content, getId());
        }
        _sccHelper.updateLastSynchronizationProperties(content);
        content.saveChanges();
        ensureTitleIsPresent(content, remoteValues, logger);
        
        boolean hasChanged = _fillContent(remoteValues, content, Map.of(), fromNewSCC, logger);
        if (hasChanged)
        {
            _nbSynchronizedContents++;
            logger.info("Some changes were detected for content '{}' and language {}", contentTitle, lang);
        }
        else
        {
            _nbNotChangedContents++;
            logger.info("No changes detected for content '{}' and language {}", contentTitle, lang);
        }
        
        // Do additional operation on the content
        SynchronizingContentOperator synchronizingContentOperator = _synchronizingContentOperatorEP.getExtension(getSynchronizingContentOperator());
        if (synchronizingContentOperator != null)
        {
            synchronizingContentOperator.additionalOperation(content, remoteValues, logger);
        }
        else
        {
            logger.warn("Cannot find synchronizing content operator with id '{}'. No additional operation has been done.", getSynchronizingContentOperator());
        }
        long endTime = System.currentTimeMillis();
        logger.info("End synchronization of content '{}' for language {} in {} ms", contentTitle, lang, endTime - startTime);
        
        return content;
    }
    
    @Override
    public List<ModifiableContent> importContent(String idValue, Map<String, Object> additionalParameters, Logger logger) throws Exception
    {
        List<ModifiableContent> createdContents = new ArrayList<>();
        
        Map<String, Object> searchParameters = putIdParameter(idValue);
        Map<String, Map<String, List<Object>>> results = getTransformedRemoteValues(searchParameters, logger);
        if (!results.isEmpty())
        {
            for (String lang : getLanguages())
            {
                ModifiableContent existingContent = getContent(lang, idValue, false);
                
                // Content does not exists whatever its configuration, import it
                if (existingContent == null)
                {
                    try
                    {
                        createdContents.add(_importContent(idValue, additionalParameters, lang, results.get(idValue), logger));
                    }
                    catch (Exception e)
                    {
                        _nbError++;
                        logger.error("An error occurred while importing or synchronizing content", e);
                    }
                }
                // The content exists but not in the current collection, synchronize it
                else if (getContent(lang, idValue, true) == null)
                {
                    _synchronizeContent(existingContent, results.get(idValue), logger);
                }
                // The content exists in the current collection, ignore it
                else if (logger.isWarnEnabled())
                {
                    logger.warn("The content of SCC '{}' identified by the synchronization code '{}' and language '{}' already exists, it has not been imported twice.", getId(), idValue, lang);
                }
            }
        }
        
        return createdContents;
    }
    
    /**
     * Set search parameters for the ID value.
     * @param idValue Value to search
     * @return Map with the search parameters
     */
    protected abstract Map<String, Object> putIdParameter(String idValue);

    /**
     * Import a content from remote values.
     * @param idValue Id (for import/synchronization) of the content to import
     * @param additionalParameters Specific parameters for import
     * @param lang Lang of the content
     * @param remoteValues Values of the content
     * @param logger The logger
     * @return The content created by import, or null
     * @throws Exception if an error occurs.
     */
    protected ModifiableContent _importContent(String idValue, Map<String, Object> additionalParameters, String lang, Map<String, List<Object>> remoteValues, Logger logger) throws Exception
    {
        long startTime = System.currentTimeMillis();

        // Calculate contentTitle
        String contentTitle = Optional.ofNullable(remoteValues.get(Content.ATTRIBUTE_TITLE))
                .map(Collection::stream)
                .orElseGet(Stream::empty)
                .filter(String.class::isInstance)
                .map(String.class::cast)
                .filter(StringUtils::isNotBlank)
                .findFirst()
                .orElse(idValue);

        // Create new content
        logger.info("Start importing content '{}' for language {}", contentTitle, lang);

        ModifiableContent content = createContentAction(lang, contentTitle, logger);
        if (content != null)
        {
            _sccHelper.updateSCCProperty(content, getId());
            _sccHelper.updateLastSynchronizationProperties(content);
            
            // Force syncCode as soon as possible
            ValueContext context = ValueContext.newInstance();
            if (getLocalAndExternalFields(Map.of("contentTypes", Arrays.asList(content.getTypes()))).contains(getIdField()))
            {
                context.withStatus(ExternalizableDataStatus.EXTERNAL);
            }
            DataHolderHelper.setValue(content, getIdField(), idValue, context, true);
            content.saveChanges();
            
            // Fill the content with the other values
            _fillContent(remoteValues, content, additionalParameters, true, logger);

            if (content instanceof WorkflowAwareContent)
            {
                // Validate content if allowed
                validateContent((WorkflowAwareContent) content, logger);
            }

            _nbCreatedContents++;

            // Do additional operation on the content
            SynchronizingContentOperator synchronizingContentOperator = _synchronizingContentOperatorEP.getExtension(getSynchronizingContentOperator());
            synchronizingContentOperator.additionalOperation(content, remoteValues, logger);

            long endTime = System.currentTimeMillis();
            logger.info("End import of content '{}' for language {} in {} ms", content.getId(), lang, endTime - startTime);
        }
        
        return content;
    }
    
    /**
     * Ensure that the title is present and set it to the current title if it is not present.
     * @param content The content to synchronize
     * @param remoteValues The remote values that would contain the title (to check)
     * @param logger The logger to warn if the title is not present
     */
    protected void ensureTitleIsPresent(Content content, Map<String, List<Object>> remoteValues, Logger logger)
    {
        if (remoteValues.containsKey(Content.ATTRIBUTE_TITLE))
        {
            boolean atLeastOneTitle =  remoteValues.get(Content.ATTRIBUTE_TITLE)
                                                   .stream()
                                                   .filter(String.class::isInstance)
                                                   .map(String.class::cast)
                                                   .anyMatch(StringUtils::isNotBlank);
            if (atLeastOneTitle)
            {
                return;
            }
        }
        
        // Force to current title
        logger.warn("The remote value of '{}' is empty for the content {}. The '{}' attribute is mandatory, the current title will remain.", Content.ATTRIBUTE_TITLE, content, Content.ATTRIBUTE_TITLE);
        remoteValues.put(Content.ATTRIBUTE_TITLE, List.of(content.getTitle()));
    }

    @Override
    public ModifiableContent getContent(String lang, String idValue, boolean forceStrictCheck)
    {
        String query = _getContentPathQuery(lang, idValue, getContentType(), forceStrictCheck);
        AmetysObjectIterable<ModifiableContent> contents = _resolver.query(query);
        return contents.stream().findFirst().orElse(null);
    }
    
    /**
     * Creates content action with result from request
     * @param lang The language
     * @param contentTitle The content title
     * @param logger The logger
     * @return The content id, or null of a workflow error occurs
     */
    protected ModifiableContent createContentAction(String lang, String contentTitle, Logger logger)
    {
        return createContentAction(getContentType(), getWorkflowName(), getInitialActionId(), lang, contentTitle, logger);
    }
    
    /**
     * Fill the content with remote values.
     * @param remoteValues The remote values
     * @param content The content to synchronize
     * @param additionalParameters Additional parameters
     * @param create <code>true</code> if content is creating, false if it is updated
     * @param logger The logger
     * @return <code>true</code> if the content has been modified, <code>false</code> otherwise
     * @throws Exception if an error occurs
     */
    protected boolean _fillContent(Map<String, List<Object>> remoteValues, ModifiableContent content, Map<String, Object> additionalParameters, boolean create, Logger logger) throws Exception
    {
        if (content instanceof WorkflowAwareContent)
        {
            // Transform remote value to get values with the cardinality corresponding to the model
            Map<String, Object> contentValues = _transformRemoteValuesCardinality(remoteValues, getContentType());
            // Remove the id field (if present) because it should be already set before calling this method
            contentValues.remove(getIdField());
            
            // Add additional values
            contentValues.putAll(getAdditionalAttributeValues(content.getValue(getIdField()), content, additionalParameters, create, logger));

            // Remove title from values if it is empty
            if (StringUtils.isEmpty((String) contentValues.get(Content.ATTRIBUTE_TITLE)))
            {
                contentValues.remove(Content.ATTRIBUTE_TITLE);
            }
            
            Set<String> notSynchronizedContentIds = getNotSynchronizedRelatedContentIds(content, contentValues, additionalParameters, content.getLanguage(), logger);

            // Get nested values supported by the EditContentFunction
            Map<String, Object> nestedValues = _getNestedValues(contentValues);
            return _editContent((WorkflowAwareContent) content, Optional.empty(), nestedValues, additionalParameters, create, notSynchronizedContentIds, logger);
        }
        
        return false;
    }
    
    /**
     * Synchronize the content with given values.
     * @param content The content to synchronize
     * @param view the view containing the item to edit
     * @param values the values
     * @param additionalParameters Additional parameters
     * @param create <code>true</code> if content is creating, <code>false</code> if it is updated
     * @param notSynchronizedContentIds the ids of the contents related to the given content but that are not part of the synchronization
     * @param logger The logger
     * @return <code>true</code> if the content has been modified, <code>false</code> otherwise
     * @throws WorkflowException if an error occurs
     */
    protected boolean _editContent(WorkflowAwareContent content, Optional<View> view, Map<String, Object> values, Map<String, Object> additionalParameters, boolean create, Set<String> notSynchronizedContentIds, Logger logger) throws WorkflowException
    {
        SynchronizationContext synchronizationContext = SynchronizationContext.newInstance()
                                                                              .withStatus(ExternalizableDataStatus.EXTERNAL)
                                                                              .withExternalizableDataContextEntry(SynchronizableContentsCollectionDataProvider.SCC_ID_CONTEXT_KEY, getId())
                                                                              .withIncompatibleValuesIgnored(true);
        
        if (view.map(v -> content.hasDifferences(v, values, synchronizationContext))
                .orElseGet(() -> content.hasDifferences(values, synchronizationContext)))
        {
            Map<String, Object> inputs = _getEditInputs(content, view, values, additionalParameters, create, notSynchronizedContentIds, logger);
            Map<String, Object> actionResult = _contentWorkflowHelper.doAction(content, getSynchronizeActionId(), inputs);
            return (boolean) actionResult.getOrDefault(AbstractContentWorkflowComponent.HAS_CHANGED_KEY, false);
        }
        else
        {
            return false;
        }
    }
    
    @SuppressWarnings("unchecked")
    private static Map<String, Object> _getNestedValues(Map<String, Object> values)
    {
        Map<String, Object> nestedValues = new HashMap<>();
        for (String key : values.keySet())
        {
            nestedValues = (Map<String, Object>) MapUtils.deepMerge(nestedValues, _getNestedValue(key, values.get(key)));
        }
        return nestedValues;
    }
    
    @SuppressWarnings("unchecked")
    private static Map<String, Object> _getNestedValue(String currentPath, Object currentValue)
    {
        Map<String, Object> nestedValues = new HashMap<>();
        int separatorIndex = currentPath.indexOf('/');
        if (separatorIndex < 0)
        {
            if (currentValue instanceof Map)
            {
                nestedValues.put(currentPath, _getNestedValues((Map<String, Object>) currentValue));
            }
            else
            {
                nestedValues.put(currentPath, currentValue);
            }
        }
        else
        {
            nestedValues.put(currentPath.substring(0, separatorIndex), _getNestedValue(currentPath.substring(separatorIndex + 1), currentValue));
        }
        return nestedValues;
    }
    
    /**
     * Get the inputs for edit content function.
     * @param content The content to synchronize
     * @param view the view containing the item to edit
     * @param values the values
     * @param additionalParameters Additional parameters
     * @param create <code>true</code> if content is creating, <code>false</code> if it is updated
     * @param notSynchronizedContentIds the ids of the contents related to the given content but that are not part of the synchronization
     * @param logger The logger
     * @return the input parameters
     */
    protected Map<String, Object> _getEditInputs(WorkflowAwareContent content, Optional<View> view, Map<String, Object> values, Map<String, Object> additionalParameters, boolean create, Set<String> notSynchronizedContentIds, Logger logger)
    {
        Map<String, Object> inputs = new HashMap<>();
        _addEditInputsForSCC(inputs, content, logger);
        inputs.put(EditSynchronizedContentFunction.ADDITIONAL_PARAMS_KEY, additionalParameters);
        inputs.put(EditSynchronizedContentFunction.SYNCHRO_INVERT_EDIT_ACTION_ID_KEY, getSynchronizeActionId());
        inputs.put(EditSynchronizedContentFunction.NOT_SYNCHRONIZED_RELATED_CONTENT_IDS_KEY, notSynchronizedContentIds);
        inputs.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, _getEditInputsContextParameters(view, values, create));
        return inputs;
    }

    /**
     * Add the inputs specific for the SCC to the inputs for edit content function.
     * @param inputs The inputs to complete
     * @param content The content to synchronize
     * @param logger The logger
     */
    protected void _addEditInputsForSCC(Map<String, Object> inputs, WorkflowAwareContent content, Logger logger)
    {
        inputs.put(EditSynchronizedContentFunction.SCC_KEY, this);
        inputs.put(EditSynchronizedContentFunction.SCC_LOGGER_KEY, logger);
    }
    
    /**
     * Get the context parameters to add to inputs for edit content function
     * @param view the view containing the item to edit
     * @param values the values
     * @param create <code>true</code> if content is creating, <code>false</code> if it is updated
     * @return the context parameters
     */
    protected Map<String, Object> _getEditInputsContextParameters(Optional<View> view, Map<String, Object> values, boolean create)
    {
        Map<String, Object> parameters = new HashMap<>();
        parameters.put(EditContentFunction.VALUES_KEY, values);
        view.ifPresent(v -> parameters.put(EditContentFunction.VIEW, v));
        parameters.put(EditContentFunction.QUIT, true);
        parameters.put(EditSynchronizedContentFunction.IMPORT, create);
        return parameters;
    }
    
    /**
     * Validates a content after import
     * @param content The content to validate
     * @param logger The logger
     */
    protected void validateContent(WorkflowAwareContent content, Logger logger)
    {
        if (validateAfterImport())
        {
            validateContent(content, getValidateActionId(), logger);
        }
    }
    
    @Override
    public Map<String, Map<String, Object>> search(Map<String, Object> searchParameters, int offset, int limit, List<Object> sort, Logger logger)
    {
        // Search
        Map<String, Map<String, Object>> results = internalSearch(_removeEmptyParameters(searchParameters), offset, limit, sort, logger);
        
        return results;
    }

    /**
     * Search values and return the result without any treatment.
     * @param searchParameters Search parameters to restrict the search
     * @param offset Begin of the search
     * @param limit Number of results
     * @param sort Sort of results (ignored for LDAP results)
     * @param logger The logger
     * @return Map of results without any treatment.
     */
    protected abstract Map<String, Map<String, Object>> internalSearch(Map<String, Object> searchParameters, int offset, int limit, List<Object> sort, Logger logger);

    /**
     * Search values and return the result organized by attributes and transformed by the {@link SynchronizingContentOperator} if exists.
     * @param searchParameters Search parameters to restrict the search
     * @param logger The logger
     * @return Map of results organized by attributes.
     */
    protected Map<String, Map<String, List<Object>>> getTransformedRemoteValues(Map<String, Object> searchParameters, Logger logger)
    {
        Map<String, Map<String, List<Object>>> remoteValues = getRemoteValues(searchParameters, logger);
        return _transformRemoteValues(remoteValues, logger);
    }
    
    /**
     * Search values and return the result organized by attributes
     * @param searchParameters Search parameters to restrict the search
     * @param logger The logger
     * @return Map of results organized by attributes.
     */
    protected abstract Map<String, Map<String, List<Object>>> getRemoteValues(Map<String, Object> searchParameters, Logger logger);

    /**
     * Transform the given remote values by the {@link SynchronizingContentOperator} if exists.
     * @param remoteValues The remote values
     * @param logger The logger
     * @return the transformed values
     */
    protected Map<String, Map<String, List<Object>>> _transformRemoteValues(Map<String, Map<String, List<Object>>> remoteValues, Logger logger)
    {
        SynchronizingContentOperator synchronizingContentOperator = _synchronizingContentOperatorEP.getExtension(getSynchronizingContentOperator());
        if (synchronizingContentOperator != null)
        {
            Map<String, Map<String, List<Object>>> transformedRemoteValues = new LinkedHashMap<>();
            ContentType contentType = _contentTypeEP.getExtension(getContentType());
            
            for (String key : remoteValues.keySet())
            {
                transformedRemoteValues.put(key, synchronizingContentOperator.transform(contentType, remoteValues.get(key), logger));
            }
            
            return transformedRemoteValues;
        }
        else
        {
            logger.warn("Cannot find synchronizing content operator with id '{}'. No transformation has applied on remote values", getSynchronizingContentOperator());
            return remoteValues; // no transformation
        }
    }
    
    /**
     * Retrieves additional values to synchronize for a content
     * @param idValue id value of the content
     * @param content The content
     * @param additionalParameters Additional parameters
     * @param create <code>true</code> if the content has been newly created, <code>false</code> otherwise
     * @param logger The logger
     * @return the values to add
     */
    protected Map<String, Object> getAdditionalAttributeValues(String idValue, Content content, Map<String, Object> additionalParameters, boolean create, Logger logger)
    {
        // No additional values by default
        return new LinkedHashMap<>();
    }
    
    /**
     * Retrieves the ids of the contents related to the given content but that are not part of the synchronization
     * @param content content
     * @param contentValues the content values that will be set
     * @param additionalParameters Additional parameters
     * @param lang Language of the content
     * @param logger The logger
     * @return the ids of the contents that are not part of the synchronization
     */
    protected Set<String> getNotSynchronizedRelatedContentIds(Content content, Map<String, Object> contentValues, Map<String, Object> additionalParameters, String lang, Logger logger)
    {
        // All contents are synchronized by default
        return new HashSet<>();
    }
    
    @Override
    public void updateSyncInformations(ModifiableContent content, String syncCode, Logger logger) throws Exception
    {
        if (StringUtils.isBlank(syncCode))
        {
            _sccHelper.removeSCCProperty(content, getId());
            content.removeValue(getIdField());
        }
        else
        {
            _sccHelper.updateSCCProperty(content, getId());
            content.setValue(getIdField(), syncCode);
        }
        
        if (content.needsSave())
        {
            content.saveChanges();
            
            if (content instanceof VersionableAmetysObject)
            {
                ((VersionableAmetysObject) content).checkpoint();
            }
        }
    }

    @Override
    public int getTotalCount(Map<String, Object> searchParameters, Logger logger)
    {
        return search(searchParameters, 0, Integer.MAX_VALUE, null, logger).size();
    }
    
    /**
     * Import or synchronize several contents from search params.
     * @param searchParameters Search parameters
     * @param forceImport To force import and ignoring the synchronize existing contents only option
     * @param logger The logger
     * @return The {@link List} of imported or synchronized {@link ModifiableContent}
     */
    protected final List<ModifiableContent> _importOrSynchronizeContents(Map<String, Object> searchParameters, boolean forceImport, Logger logger)
    {
        return _importOrSynchronizeContents(searchParameters, forceImport, logger, ProgressionTrackerFactory.createContainerProgressionTracker("Import or synchronize contents", logger));
    }
    
    /**
     * Import or synchronize several contents from search params.
     * @param searchParameters Search parameters
     * @param forceImport To force import and ignoring the synchronize existing contents only option
     * @param logger The logger
     * @param progressionTracker The progression tracker
     * @return The {@link List} of imported or synchronized {@link ModifiableContent}
     */
    protected List<ModifiableContent> _importOrSynchronizeContents(Map<String, Object> searchParameters, boolean forceImport, Logger logger, ContainerProgressionTracker progressionTracker)
    {
        SimpleProgressionTracker progressionTrackerForTransformRemoteValues = progressionTracker.addSimpleStep("transformremotevalues", new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_SCHEDULER_SYNCHRONIZE_COLLECTION_TRANSFORM_STEP_LABEL"));
        SimpleProgressionTracker progressionTrackerForSynchro = progressionTracker.addSimpleStep("synchronizetransformedremotevalues", new I18nizableText("plugin.contentio", "PLUGINS_CONTENTIO_SCHEDULER_SYNCHRONIZE_COLLECTION_SYNCHRONIZE_STEP_LABEL"));
        
        List<ModifiableContent> contents = new ArrayList<>();
        
        Map<String, Map<String, List<Object>>> remoteValuesByContent = getTransformedRemoteValues(searchParameters, logger);
        progressionTrackerForTransformRemoteValues.increment();
        
        progressionTrackerForSynchro.setSize(remoteValuesByContent.size());
        for (String idValue : remoteValuesByContent.keySet())
        {
            Map<String, List<Object>> remoteValues = remoteValuesByContent.get(idValue);
            _handleContent(idValue);
            contents.addAll(_importOrSynchronizeContent(idValue, remoteValues, forceImport, logger));
            progressionTrackerForSynchro.increment();
        }
        
        return contents;
    }
    
    @Override
    protected List<ModifiableContent> _getContentsToRemove(AmetysObjectIterable<ModifiableContent> contents)
    {
        return contents.stream()
            .filter(content -> !_isHandled(content.getValue(getIdField())))
            .collect(Collectors.toList());
    }
}
