/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.datafiller;

import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import org.ametys.cms.content.indexing.solr.observation.ObserverHelper;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.explorer.ExplorerResourcesDAO;
import org.ametys.cms.indexing.IndexingException;
import org.ametys.cms.indexing.WorkspaceIndexer;
import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.core.datasource.SQLDataSourceManager;
import org.ametys.core.schedule.progression.ProgressionTrackerFactory;
import org.ametys.core.script.SQLScriptHelper;
import org.ametys.core.ui.Callable;
import org.ametys.core.util.dom.DOMUtils;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDAO;
import org.ametys.plugins.contentio.synchronize.impl.DefaultSynchronizingContentOperator;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.explorer.resources.actions.AddOrUpdateResourceHelper;
import org.ametys.plugins.explorer.resources.actions.AddOrUpdateResourceHelper.ResourceOperationMode;
import org.ametys.plugins.linkdirectory.link.LinkDAO;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * Manage the creation of sitemap, contents and services related to the creation of test pages
 */
public class CreateTestPageManager extends AbstractLogEnabled implements Component, Serviceable, PluginAware
{
    /** Avalon Role */
    public static final String ROLE = CreateTestPageManager.class.getName();

    private static final String __WEBINF_PARAM_DATAFILLER_SOURCE_DIRECTORY = "context:/WEB-INF/param/datafill/";
    private static final String __SITEMAP_SOURCE_URI = __WEBINF_PARAM_DATAFILLER_SOURCE_DIRECTORY + "sitemap.xml";
    private static final String __LINKDIRECTORY_SOURCE_URI = __WEBINF_PARAM_DATAFILLER_SOURCE_DIRECTORY + "linkdirectory.xml";
    private static final Logger __SCC_LOGGER = LoggerFactory.getLogger(SynchronizableContentsCollection.class);

    /** The content type extension point */
    private ContentTypeExtensionPoint _cTypeEP;
    /** Generic content  creation manager */
    private GenericContentCreationManager _genericContentCreationManager;
    /** Generic service  creation manager */
    private GenericServiceCreationManager _genericServiceCreationManager;
    /** Generic service creation manager */
    private GenericFormCreationManager _genericFormCreationManager;
    /** Classified ads contents creation manager */
    private ClassifiedAdsContentCreationManager _classifiedAdsContentCreationManager;
    /** The DAO for Links */
    private LinkDAO _linkDAO;
    /** The ametys resolver */
    private AmetysObjectResolver _resolver;
    /** The resource Helper */
    private AddOrUpdateResourceHelper _resourceHelper;
    /** The explorer resources DAO */
    private ExplorerResourcesDAO _resourcesDAO;
    /** The SCC DAO */
    private SynchronizableContentsCollectionDAO _sccDAO;
    /** The site manager */
    private SiteManager _siteManager;
    /** The Sitemap creation Manager */
    private SitemapPopulator _sitemapPopulator;
    /** The source resolver */
    private SourceResolver _sourceResolver;
    /** The Workspace indexer */
    private WorkspaceIndexer _workspaceIndexer;

    private String _fillingDataDirectory;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _genericContentCreationManager = (GenericContentCreationManager) manager.lookup(GenericContentCreationManager.ROLE);
        _genericServiceCreationManager = (GenericServiceCreationManager) manager.lookup(GenericServiceCreationManager.ROLE);
        _genericFormCreationManager = (GenericFormCreationManager) manager.lookup(GenericFormCreationManager.ROLE);
        if (manager.hasService(ClassifiedAdsContentCreationManager.ROLE))
        {
            _classifiedAdsContentCreationManager = (ClassifiedAdsContentCreationManager) manager.lookup(ClassifiedAdsContentCreationManager.ROLE);
        }
        _linkDAO = (LinkDAO) manager.lookup(LinkDAO.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _resourceHelper = (AddOrUpdateResourceHelper) manager.lookup(AddOrUpdateResourceHelper.ROLE);
        _resourcesDAO = (ExplorerResourcesDAO) manager.lookup(org.ametys.plugins.explorer.resources.actions.ExplorerResourcesDAO.ROLE);
        _sccDAO = (SynchronizableContentsCollectionDAO) manager.lookup(SynchronizableContentsCollectionDAO.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _sitemapPopulator = (SitemapPopulator) manager.lookup(SitemapPopulator.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _workspaceIndexer = (WorkspaceIndexer) manager.lookup(WorkspaceIndexer.ROLE);
    }

    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _fillingDataDirectory = "plugin:" + pluginName + "://filling-data/";
    }

    /**
     * Generate multiple pages to check quality of templates.
     * First, a sitemap of blank pages is created according to a sitemap.xml file located at _FILE_SITEMAP_URI.
     * This sitemap is intended to be customised to fit project needs.
     * 
     * Generic contents and services are then created and integrated into pages
     * to test the different view and zone allowed by each template
     * @param siteName the site name
     * @param language the language of the sitemap where the test page will be added
     * @return A map containing two key:
     *          - success: true if no error happened during generation
     *          - pageIds: a list of all the Ids of page created by the function
     */
    @Callable(rights = "Data_Filler_Right_Create_Test_Page_Button")
    public Map<String, Object> createTestPages(String siteName, String language)
    {
        long startTime = System.currentTimeMillis();
        getLogger().info("Starting creating contents, services and pages for tests");
        Map<String, Object> result = new HashMap<>();
        boolean success = true;
        try
        {
            ObserverHelper.suspendObservationForIndexation();

            Site site = _siteManager.getSite(siteName);
            
            Sitemap sitemap = site.getSitemap(language);
            success = _createSitemaps(sitemap) && success;

            success = _createTestContentPages(sitemap) && success;
            if (_classifiedAdsContentCreationManager != null)
            {
                success = _createTestClassifiedAdsPages(sitemap) && success;
            }

            Set<String> pagesWithAttachment = new HashSet<>();
            success = _createTestServicePages(sitemap, pagesWithAttachment) && success;
            success = _createTestForm(siteName) && success;

            success = _createLinks(siteName, language) && success;

            success = _createUserAndOrgUnit() && success;
            
            success = _createExplorerResources(pagesWithAttachment) && success;
        }
        finally
        {
            getLogger().info("Reindexing Workspaces...");
            ObserverHelper.restartObservationForIndexation();
            try
            {
                _workspaceIndexer.indexAllWorkspaces();
            }
            catch (IndexingException e)
            {
                success = false;
                getLogger().error("An error occurred indexing all workspaces", e);
            }
        }

        long endTime = System.currentTimeMillis();
        getLogger().info("End of the creation of contents, services and pages for tests in {} ms",  endTime - startTime);

        result.put("success", success);
        Collection<String> createdPages = _sitemapPopulator.getCreatedPages().values();
        result.put("pageIds", createdPages);
        return result;
    }
    
    private boolean _createSitemaps(Sitemap sitemap)
    {
        Source sourceFile = null;
        try
        {
            sourceFile = _sourceResolver.resolveURI(__SITEMAP_SOURCE_URI);
            if (sourceFile.exists())
            {
                getLogger().info("Creating sitemap {}...", sitemap.getSitemapName());
                _sitemapPopulator.createSitemap(sitemap, sourceFile);
                getLogger().info("End of the sitemap creation");
            }
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred while creating the sitemap...", e);
            return false;
        }
        finally
        {
            _sourceResolver.release(sourceFile);
        }
        return true;
    }
    
    private boolean _createTestContentPages(Sitemap sitemap)
    {
        getLogger().info("Creating generic content pages");
        try
        {
            _genericContentCreationManager.createGenericContentPages(sitemap);
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred: failed to create content and add them to test pages", e);
            return false;
        }
        
        getLogger().info("End of generic content pages creation");
        return true;
    }
    
    private boolean _createTestClassifiedAdsPages(Sitemap sitemap)
    {
        getLogger().info("Creating classified ads pages");
        try
        {
            _classifiedAdsContentCreationManager.createClassifiedAdsPages(sitemap);
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred: failed to create classified ads root page and create ads", e);
            return false;
        }
    
        getLogger().info("End of classified ads pages creation");
        return true;
    }
    
    private boolean _createTestServicePages(Sitemap sitemap, Set<String> pageWithAttachmentID)
    {
        getLogger().info("Creating service pages");
        try
        {
            pageWithAttachmentID.addAll(_genericServiceCreationManager.createGenericServicePages(sitemap));
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred: failed to create services", e);
            return false;
        }
        getLogger().info("End of service pages creation");
        return true;
    }
    
    private boolean _createTestForm(String sitename)
    {
        getLogger().info("Creating form");
        try
        {
            _genericFormCreationManager.createGenericForms(sitename);
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred: failed to create form", e);
            return false;
        }
        getLogger().info("End of form creation");
        return true;
    }

    // FIXME DATAFILLER-12 Add check for the creation of link
    // FIXME DATAFILLER-13 Improve management of theme for Link
    /**
     * Add links in the link directory according the configuration provided by the user in
     * WEB-INF/param/datafill/linkdirectory.xml
     * @param siteName the sitename where the links will be added
     * @param lang the lang of the link
     */
    private boolean _createLinks(String siteName, String lang)
    {
        Source sourceFile = null;
        try
        {
            sourceFile = _sourceResolver.resolveURI(__LINKDIRECTORY_SOURCE_URI);
            if (sourceFile.exists())
            {
                getLogger().info("Add link in Link Directory");

                DocumentBuilderFactory docFactory = DocumentBuilderFactory.newInstance();
                DocumentBuilder docBuilder = docFactory.newDocumentBuilder();

                try (InputStream is = sourceFile.getInputStream())
                {
                    Document doc = docBuilder.parse(is);
                    Element root = doc.getDocumentElement();
                    List<Element> linkNodeList = DOMUtils.getChildElementsByTagName(root, "link");

                    for (Element linkElement : linkNodeList)
                    {
                        Map<String, Object> parameters = new HashMap<>();
                        parameters.put("siteName", siteName);
                        parameters.put("lang", lang);

                        List<Element> parameterElementList = DOMUtils.getChildElements(linkElement);
                        for (Element parameterElement : parameterElementList)
                        {
                            String name = parameterElement.getNodeName();
                            String value = parameterElement.getTextContent();
                            parameters.put(name, value);
                        }
                        
                        // Here we compute the pageId of the page requested by the user
                        if (parameters.get("url-type").equals("PAGE"))
                        {
                            String pageURL = (String) parameters.get("url");
                            String pageId = _sitemapPopulator.getCreatedPages().get(lang + "#" + pageURL);
                            parameters.put("url", pageId);
                        }
                        try
                        {
                            _linkDAO.createLink(parameters);
                        }
                        catch (IllegalArgumentException e)
                        {
                            getLogger().error("Failed to create link {}", parameters.get("title"), e);
                        }
                    }
                    getLogger().info("End of link import");
                }
            }
        }
        catch (IOException e)
        {
            getLogger().error("Can't read source file at {}", __LINKDIRECTORY_SOURCE_URI, e);
            return false;
        }
        catch (ParserConfigurationException e)
        {
            getLogger().error("Failed to get document parser", e);
            return false;
        }
        catch (SAXException e)
        {
            getLogger().error("Failed to parse file at {}", __LINKDIRECTORY_SOURCE_URI, e);
            return false;
        }
        catch (Exception e)
        {
            getLogger().error("Failed to create Link in link directory", e);
            return false;
        }
        finally
        {
            _sourceResolver.release(sourceFile);
        }
        return true;
    }
    
    private boolean _createUserAndOrgUnit()
    {
        getLogger().info("Add user and OrgUnit...");
        
        // Try to find a User content type with an organizationChart
        // if a User content type is found, create user table and user content
        // if the content type has an organizationChart view, also create
        // entity table and entity content
        String contentTypeId = null;
        Iterator<String> i = _cTypeEP.getDirectSubTypes(UserDirectoryHelper.ABSTRACT_USER_CONTENT_TYPE).iterator();
        boolean hasOrgUnitView = false;
        while (i.hasNext() && !hasOrgUnitView)
        {
            contentTypeId = i.next();
            hasOrgUnitView = _cTypeEP.getExtension(contentTypeId).getViewNames().contains("organizationChart");
        }
        
        if (contentTypeId != null)
        {
            _addUserAndOrgunitToDatabase();
            try
            {
                _createContentsFromSCC(_getUserSCCParameters(contentTypeId));
                _createContentsFromSCC(_getOrgUnitSCCParameters());
            }
            catch (ProcessingException e)
            {
                getLogger().error("Failed to create user from database", e);
                return false;
            }
            catch (Exception e)
            {
                getLogger().error("Failure while trying to create user data", e);
                return false;
            }
        }
        else
        {
            getLogger().warn("No content type found for user content. No user content or entity has been generated");
        }
        getLogger().info("End of user and OrgUnit import");
        return true;
    }

    /**
     * Add two table in the internal database and fill them with user and orgunit description
     */
    private void _addUserAndOrgunitToDatabase()
    {
        Connection connection = null;
        try
        {
            String userScriptFile = _fillingDataDirectory + "scripts/addUsers.sql";
            String orgunitsScriptFile = _fillingDataDirectory + "scripts/addOrgunits.sql";
            connection = ConnectionHelper.getConnection(SQLDataSourceManager.AMETYS_INTERNAL_DATASOURCE_ID);
            boolean imported = SQLScriptHelper.createTableIfNotExists(connection, "datafiller_sql_users", userScriptFile, _sourceResolver);
            imported = SQLScriptHelper.createTableIfNotExists(connection, "datafiller_sql_orgunits", orgunitsScriptFile, _sourceResolver) && imported;

            if (imported)
            {
                getLogger().info("A new database was created for the data-filler user");
            }
        }
        catch (IOException | SQLException e)
        {
            getLogger().error("Failed to read import user script", e);
        }
        finally
        {
            ConnectionHelper.cleanup(connection);
        }
    }
    
    private void _createContentsFromSCC(Map<String, Object> sccParameters) throws ProcessingException
    {
     // To avoid creating the same SCC multiple time, we first check if an SCC with this label exist
        String sccLabel = (String) sccParameters.get("label");
        String sccId = sccLabel.toLowerCase().trim().replaceAll("[\\W_]", "-").replaceAll("-+", "-").replaceAll("^-", "");

        SynchronizableContentsCollection requestedSCC = _sccDAO.getSynchronizableContentsCollection(sccId);

        if (requestedSCC == null)
        {
            /* Create the SCC */
            String collectionId = _sccDAO.addCollection(sccParameters);
            requestedSCC = _sccDAO.getSynchronizableContentsCollection(collectionId);
        }

        // Finally, we create our users using the SCC.
        requestedSCC.populate(__SCC_LOGGER, ProgressionTrackerFactory.createContainerProgressionTracker("Populate", __SCC_LOGGER));
    }
    
    private Map<String, Object> _getUserSCCParameters(String contentTypeId)
    {
        String sccLabel = "Collection d'utilisateurs data-filler";

        Map<String, Object> sccParameters = new HashMap<>();
        sccParameters.put("label", sccLabel);
        sccParameters.put("contentType", contentTypeId);
        sccParameters.put("contentPrefix", "ud-");
        sccParameters.put("workflowName", "user");
        sccParameters.put("initialActionId", 11);
        sccParameters.put("synchronizeActionId", 800);
        sccParameters.put("validateActionId", 41);
        sccParameters.put("validateAfterImport", true);
        sccParameters.put("contentOperator", DefaultSynchronizingContentOperator.class.getName());
        sccParameters.put("languages", List.of("fr"));
        sccParameters.put("modelId", "org.ametys.plugins.contentio.synchronize.collection.sql");
        sccParameters.put("org.ametys.plugins.contentio.synchronize.collection.sql$datasourceId", SQLDataSourceManager.AMETYS_INTERNAL_DATASOURCE_ID);
        sccParameters.put("org.ametys.plugins.contentio.synchronize.collection.sql$tableName", "datafiller_sql_users");
        sccParameters.put("org.ametys.plugins.contentio.synchronize.collection.sql$idField", "uniqueId");
        sccParameters.put("org.ametys.plugins.contentio.synchronize.collection.sql$mapping", "["
                + "{\"metadata-ref\":\"firstname\",\"attribute\":\"prenom\",\"synchro\":false},"
                + "{\"metadata-ref\":\"lastname\",\"attribute\":\"nom\",\"synchro\":false},"
                + "{\"metadata-ref\":\"title\",\"attribute\":\"title\",\"synchro\":false},"
                + "{\"metadata-ref\":\"uniqueId\",\"attribute\":\"login\",\"synchro\":false}]");
        
        return sccParameters;
    }
    
    private Map<String, Object> _getOrgUnitSCCParameters()
    {
        String sccLabel = "Collection d'entités data-filler";
        
        Map<String, Object> sccParameters = new HashMap<>();
        sccParameters.put("label", sccLabel);
        sccParameters.put("contentType", "org.ametys.plugins.userdirectory.Content.udorgunit");
        sccParameters.put("contentPrefix", "ud-");
        sccParameters.put("workflowName", "udorgunit");
        sccParameters.put("initialActionId", 11);
        sccParameters.put("synchronizeActionId", 800);
        sccParameters.put("validateActionId", 41);
        sccParameters.put("validateAfterImport", true);
        sccParameters.put("contentOperator", DefaultSynchronizingContentOperator.class.getName());
        sccParameters.put("languages", List.of("fr"));
        sccParameters.put("modelId", "org.ametys.plugins.userdirectory.synchronize.orgunit.sql");
        sccParameters.put("org.ametys.plugins.userdirectory.synchronize.orgunit.sql$datasourceId", SQLDataSourceManager.AMETYS_INTERNAL_DATASOURCE_ID);
        sccParameters.put("org.ametys.plugins.userdirectory.synchronize.orgunit.sql$tableName", "datafiller_sql_orgunits");
        sccParameters.put("org.ametys.plugins.userdirectory.synchronize.orgunit.sql$idField", "orgUnitId");
        sccParameters.put("org.ametys.plugins.userdirectory.synchronize.orgunit.sql$orgunitRemoteIdColumnName", "id");
        sccParameters.put("org.ametys.plugins.userdirectory.synchronize.orgunit.sql$tableNameUser", "datafiller_sql_users");
        sccParameters.put("org.ametys.plugins.userdirectory.synchronize.orgunit.sql$orgUnitJoinColumnName", "orgunitId");
        sccParameters.put("org.ametys.plugins.userdirectory.synchronize.orgunit.sql$loginUser", "uniqueId");
        sccParameters.put("org.ametys.plugins.userdirectory.synchronize.orgunit.sql$loginColumnName", "login");
        sccParameters.put("org.ametys.plugins.userdirectory.synchronize.orgunit.sql$roleColumnName", "role");
        sccParameters.put("org.ametys.plugins.userdirectory.synchronize.orgunit.sql$mapping", "["
                + "{\"metadata-ref\":\"title\",\"attribute\":\"title\",\"synchro\":false},"
                + "{\"metadata-ref\":\"orgUnitId\",\"attribute\":\"id\",\"synchro\":false},"
                + "{\"metadata-ref\":\"parentOrgUnit\",\"attribute\":\"parentId\",\"synchro\":false},"
                + "{\"metadata-ref\":\"abstract\",\"attribute\":\"abstract\",\"synchro\":false}]");
        
        return sccParameters;
    }
    
    
    private boolean _createExplorerResources(Set<String> pageWithAttachment)
    {
        getLogger().info("Adding resources in explorer...");
        String sourceURI = _fillingDataDirectory + "resources/Dossier de remplissage.zip";
        Source zipSource = null;
        ResourceOperationMode mode = ResourceOperationMode.ADD_UNZIP;
        try
        {
            zipSource = _sourceResolver.resolveURI(sourceURI);
            String filename = StringUtils.substringAfterLast(zipSource.getURI(), "/");

            // Add the resource to the resource directory
            ModifiableResourceCollection root = (ModifiableResourceCollection) _resourcesDAO.getResourcesRootNodes().get(0);
            try (InputStream is = zipSource.getInputStream())
            {
                _resourceHelper.performResourceOperation(is, filename, root, mode);
            }
            
            // Add the resource to the page needing attachment
            for (String pageId : pageWithAttachment)
            {
                ModifiablePage attachmentPage = _resolver.resolveById(pageId);
                root = (ModifiableResourceCollection) attachmentPage.getRootAttachments();
                try (InputStream is = zipSource.getInputStream())
                {
                    _resourceHelper.performResourceOperation(is, filename, root, mode);
                }
            }
        }
        catch (IOException e)
        {
            getLogger().error("Failed to read source at {}", sourceURI, e);
            return false;
        }
        catch (Exception e)
        {
            getLogger().error("Something went wrong", e);
            return false;
        }
        finally
        {
            _sourceResolver.release(zipSource);
        }
        getLogger().info("Resources added in explorer");
        return true;
    }
}
