/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.datafiller;

import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import org.ametys.cms.repository.ContentDAO.TagMode;
import org.ametys.core.util.dom.DOMUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.MetadataAwareSitemapElement;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageDAO;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * Manage the generation of a sitemap for test purposes
 */
public class SitemapPopulator extends AbstractLogEnabled implements Component, Serviceable, Initializable
{
    /** Avalon Role */
    public static final String ROLE = SitemapPopulator.class.getName();

    /** The page DAO */
    private PageDAO _pageDAO;
    /** The ametys object resolver */
    private AmetysObjectResolver _resolver;

    private Map<String, String> _createdPageIds;

    @Override
    public void initialize() throws Exception
    {
        _createdPageIds = new HashMap<>();
    }

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _pageDAO = (PageDAO) manager.lookup(PageDAO.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }

    /**
     * Populate the sitemap according to the description found in the source
     * @param sitemap the sitemap where the pages will be added
     * @param source the XML source file that describe the sitemap to create
     * @throws Exception if an error occured during the sitemap creation
     */
    public void createSitemap(Sitemap sitemap, Source source) throws Exception
    {
        DocumentBuilderFactory docFactory = DocumentBuilderFactory.newInstance();
        DocumentBuilder docBuilder = docFactory.newDocumentBuilder();
        try (InputStream is = source.getInputStream())
        {
            Document doc = docBuilder.parse(is);
            Element root = doc.getDocumentElement();

            _createPagesForSitemap(root, sitemap);
        }
    }

    /**
     * get the cache of page created by {@link SitemapPopulator} and their corresponding key
     * @return a map containing for each sitemap the key of the created page and their corresponding Id
     */
    public Map<String, String> getCreatedPages()
    {
        return _createdPageIds;
    }

    /**
     * delete a page and remove it form the _createdPages
     * @param page the {@link Page} to be removed
     */
    public void deletePage(ModifiablePage page)
    {
        String parentPath = ((SitemapElement) page.getParent()).getPathInSitemap();
        String key = _getPageKey(page.getTitle(), parentPath, page.getSitemapName());
        _pageDAO.deletePage(page, true);
        _createdPageIds.remove(key);
    }

    /**
     * Create the sitemap
     * @param root the xml file root
     * @param sitemap the sitemap where the pages will be added
     */
    private void _createPagesForSitemap(Element root, Sitemap sitemap)
    {
        List<Element> pageElementList = DOMUtils.getChildElementsByTagName(root, "page");
        for (Element pageElement : pageElementList)
        {
            _createPageForSitemap(sitemap, pageElement, sitemap.getSitemapName());
        }
    }

    /**
     * Create blank page for the node NodeList and define it as a child of parentPage for the sitemap sitemapName
     * @param parentPage the parent page
     * @param pageElement the page node
     * @param sitemapName the sitemap name
     */
    private void _createPageForSitemap(MetadataAwareSitemapElement parentPage, Element pageElement, String sitemapName)
    {
        String title = pageElement.getAttribute("title");
        if (StringUtils.isBlank(title))
        {
            getLogger().error("Can't create page because no title is defined");
            return;
        }
        
        String titleLong = pageElement.getAttribute("longtitle");
        String name = pageElement.getAttribute("name");
        // Create pages
        // We have to use a hack in order to force the name of the page
        // If a page already exist, it will create a page with name= name-n
        ModifiablePage page = getOrCreatePage(parentPage, StringUtils.isNotBlank(name) ? name : title , titleLong);
        page.setTitle(title);
        page.saveChanges();

        // Create child page
        List<Element> subPageElementList = DOMUtils.getChildElementsByTagName(pageElement, "page");
        for (Element subPageElement : subPageElementList)
        {
            _createPageForSitemap(page, subPageElement, sitemapName);
        }

        // Set visibility
        boolean invisible = pageElement.getAttribute("invisible").equals("true");
        if (invisible)
        {
            _pageDAO.setVisibility(List.of(page.getId()), true);
        }

        // Tag pages
        String tags = pageElement.getAttribute("tags");
        if (StringUtils.isNotBlank(tags))
        {
            for (String tag : StringUtils.split(tags,  ","))
            {
                try
                {
                    _pageDAO.tag(List.of(page.getId()), List.of(tag), TagMode.INSERT, new HashMap<>(), true);
                }
                catch (Exception e)
                {
                    getLogger().warn("Failed to tag page '{}' with tags '{}'.", page.getPathInSitemap(), tags, e);
                }
            }
        }

        // Set template
        String template = pageElement.getAttribute("template");
        if (StringUtils.isNotBlank(template))
        {
            try
            {
                _pageDAO.setTemplate(List.of(page.getId()), template);
            }
            catch (IllegalStateException e)
            {
                getLogger().warn("Failed to set template for page {}", title, e);
            }
        }
    }

    
    /**
     * Get the children of the parent page with the given title.
     * If the page doesn't already exist, we create it
     * @param parentPage the parent page.
     * @param title the title of the requested page
     * @return the requested page
     */
    public ModifiablePage getOrCreatePage(MetadataAwareSitemapElement parentPage, String title)
    {
        return getOrCreatePage(parentPage, title, StringUtils.EMPTY);
    }
    
    /**
     * Get the children of the parent page with the given title.
     * @param parentPage the parent page
     * @param title the title of the requested page
     * @param titleLong the long title of the requested page
     * @return the requested page
     */
    public ModifiablePage getOrCreatePage(MetadataAwareSitemapElement parentPage, String title, String titleLong)
    {
        String key = _getPageKey(title, parentPage.getPathInSitemap(), parentPage.getSitemapName());
        
        ModifiablePage page = _getPage(key);
        
        if (page == null)
        {
            page = _createPage(parentPage, title, titleLong, key);
        }
        return page;
    }


    /**
     * Get the page with the given key previously created.
     * If the page doesn't exist, remove the entry from the map and return null
     * @param key the identifier of the page in the cache
     * @return the {@link ModifiablePage} requested
     */
    private ModifiablePage _getPage(String key)
    {
        if (_createdPageIds.containsKey(key))
        {
            try
            {
                return _resolver.resolveById(_createdPageIds.get(key));
            }
            catch (UnknownAmetysObjectException e)
            {
                // the page doesn't exist anymore
                _createdPageIds.remove(key);
            }
        }
        return null;
    }


    /**
     * Create a new page and save it in the SitemapPopulator cache
     * @param parentPage the parent page of the new page
     * @param title the title of the page
     * @param titleLong the long title of the page
     * @param key the identifier of the page in the cache
     * @return the {@link ModifiablePage} created
     */
    private ModifiablePage _createPage(MetadataAwareSitemapElement parentPage, String title, String titleLong, String key)
    {
        Map<String, Object> viewPageResult = _pageDAO.createPage(parentPage.getId(), title, titleLong);
        String pageId = (String) viewPageResult.get("id");
        _createdPageIds.put(key, pageId);
        return _resolver.resolveById(pageId);
    }


    /** 
     * Generate a key based on the title of the page and the path of its parent.
     * @param title the title of the page
     * @param parentPath the path of the parent in the sitemap
     * @return the key
     */
    private String _getPageKey(String title, String parentPath, String lang)
    {
        StringBuilder sb = new StringBuilder();
        sb.append(lang).append("#");
        if (StringUtils.isNotEmpty(parentPath))
        {
            sb.append("/").append(parentPath);
        }
        sb.append("/").append(title);
        return sb.toString();
    }
}
