/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.deploystarter;

import java.io.InputStream;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.ConfigurationUtil;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ResourceNotFoundException;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.MoveableSource;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceException;
import org.apache.excalibur.source.SourceResolver;
import org.w3c.dom.Element;

import org.ametys.core.right.Profile;
import org.ametys.core.right.RightProfilesDAO;
import org.ametys.core.schedule.Runnable;
import org.ametys.core.schedule.Runnable.FireProcess;
import org.ametys.core.schedule.Schedulable;
import org.ametys.core.schedule.SchedulableExtensionPoint;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.user.directory.ModifiableUserDirectory;
import org.ametys.core.user.directory.UserDirectory;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.core.impl.schedule.DefaultRunnable;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.plugin.PluginsManager;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * This class read a WEB-INF/param/start-actions.xml file to act
 */
public class Init extends AbstractLogEnabled implements org.ametys.runtime.plugin.Init, Serviceable
{
    /** The excalibur source resolver */
    protected SourceResolver _resolver;
    /** The user manager */
    protected UserManager _userManager;
    /** The profiles DAO */
    protected RightProfilesDAO _profilesDAO;
    /** The scheduler */
    protected Scheduler _scheduler;
    /** The scheduler EP */
    protected SchedulableExtensionPoint _schedulableEP;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        if (!PluginsManager.getInstance().isSafeMode())
        {
            _profilesDAO = (RightProfilesDAO) manager.lookup(RightProfilesDAO.ROLE);
            _scheduler = (Scheduler) manager.lookup(Scheduler.ROLE);
            _schedulableEP = (SchedulableExtensionPoint) manager.lookup(SchedulableExtensionPoint.ROLE);
        }
    }
    
    @Override
    public void init() throws Exception
    {
        getLogger().debug("'start actions' loading...");
        
        Source source = null;
        Source source2 = null;
        try
        {
            source = _resolver.resolveURI("context://WEB-INF/param/start-actions.xml");
            source2 = _resolver.resolveURI("context://WEB-INF/param/start-actions-" + DateUtils.dateToString(new Date()).replaceAll(":", "").replaceAll("\\+", "") + ".xml");
            if (source.exists())
            {
                Configuration configuration = null;
                try (InputStream is = source.getInputStream())
                {
                    configuration = new DefaultConfigurationBuilder().build(is);
                }

                // First move the source to ensure it would not apply a second time on next start
                ((MoveableSource) source).moveTo(source2);

                if (getLogger().isDebugEnabled())
                {
                    getLogger().debug("'start actions' file backed up to " + source2.getURI());
                }
                
                _act(configuration);
            }
            else
            {
                getLogger().debug("'start actions' no file found");
            }
        }
        catch (SourceException e)
        {
            getLogger().error("An error occurred before applying 'start actions'", e);
        }
        catch (ResourceNotFoundException e)
        {
            getLogger().debug("'start actions' no file found");
        }
        finally
        {
            _resolver.release(source);
            _resolver.release(source2);
        }

        getLogger().debug("'start actions' ends");
    }
    
    /**
     * Act
     * @param configuration The whole start actions configuration
     * @throws Exception If an error occurred
     */
    protected void _act(Configuration configuration) throws Exception
    {
        _actUsersPopulations(configuration.getChild("users-populations"));
        if (!PluginsManager.getInstance().isSafeMode())
        {
            _actRights(configuration.getChild("rights"));
            _actSchedulables(configuration.getChild("schedulables"));
        }
    }

    /**
     * Change properties on a given user
     * @param usersPopulationsConfiguration The sub configuration
     * @throws Exception If an error occurred
     */
    protected void _actUsersPopulations(Configuration usersPopulationsConfiguration) throws Exception
    {
        getLogger().debug("'start actions' acting on user populations...");

        for (Configuration usersPopulationConfiguration : usersPopulationsConfiguration.getChildren("users-population"))
        {
            String populationId = usersPopulationConfiguration.getAttribute("id");
            
            for (Configuration userConfiguration : usersPopulationConfiguration.getChildren("user"))
            {
                String login = userConfiguration.getAttribute("login");
                
                UserDirectory userDirectory = _userManager.getUserDirectory(populationId, login);
                if (populationId == null)
                {
                    throw new ConfigurationException("The user '" + UserIdentity.userIdentityToString(new UserIdentity(login, populationId)) + "' does not exists and thus cannot be modified", userConfiguration);
                }
                else if (!(userDirectory instanceof ModifiableUserDirectory))
                {
                    throw new ConfigurationException("The user '" + UserIdentity.userIdentityToString(new UserIdentity(login, populationId)) + "' is not modifiable", userConfiguration);
                }

                Map<String, String> modifications = new HashMap<>();
                modifications.put("login", login);
                
                for (Configuration propertyConfiguration : userConfiguration.getChildren())
                {
                    modifications.put(propertyConfiguration.getName(), propertyConfiguration.getValue(""));
                }
                
                getLogger().info("'start actions' updating user {}", UserIdentity.userIdentityToString(new UserIdentity(login, populationId)));
                ((ModifiableUserDirectory) userDirectory).update(modifications);
            }
        }
    }

    /**
     * Change a profile
     * @param rightsConfiguration The right part of the configuration
     * @throws Exception If an error occurred
     */
    protected void _actRights(Configuration rightsConfiguration) throws Exception
    {
        getLogger().debug("'start actions' acting on rights...");
        
        for (Configuration profileConfiguration : rightsConfiguration.getChild("profiles").getChildren("profile"))
        {
            String profileId = profileConfiguration.getAttribute("id");
            Profile profile = _profilesDAO.getProfile(profileId);
            if (profile == null)
            {
                throw new ConfigurationException("Cannot edit profile '" + profileId + "'... it does not exist", profileConfiguration);
            }
            
            List<String> rights = _profilesDAO.getRights(profileId);
            
            for (Configuration removeConfiguration : profileConfiguration.getChildren("remove"))
            {
                String rightId = removeConfiguration.getAttribute("right");
                rights.remove(rightId);
            }
            
            _profilesDAO.updateRights(profile, rights);
        }
    }
    
    /**
     * Launch schedulables on startup.
     * @param schedulablesConfiguration The schedulables part of the configuration
     * @throws Exception If an error occurred
     */
    protected void _actSchedulables(Configuration schedulablesConfiguration) throws Exception
    {
        getLogger().debug("'start actions' acting on schedulables...");

        for (Configuration schedulableConfiguration : schedulablesConfiguration.getChildren("schedulable"))
        {
            String schedulableId = schedulableConfiguration.getAttribute("id");
            Schedulable schedulable = _schedulableEP.getExtension(schedulableId);
            
            // Get the parameters of the job
            Map<String, Object> parameters = new HashMap<>();
            Map<String, ElementDefinition> schedulableParams = schedulable.getParameters();
            Configuration parametersConfiguration = schedulableConfiguration.getChild("params");
            Element parametersElt = ConfigurationUtil.toElement(parametersConfiguration);
            for (Configuration parameterConfiguration : schedulableConfiguration.getChild("params").getChildren())
            {
                String parameterName = parameterConfiguration.getName();
                if (schedulableParams.containsKey(parameterName))
                {
                    ElementDefinition definition = schedulableParams.get(parameterName);
                    String parameterType = parameterConfiguration.getAttribute("type", StringUtils.EMPTY);
                    Object parameterValue;
                    if (parameterType.equals("config"))
                    {
                        parameterValue = Config.getInstance().getValue(parameterConfiguration.getValue());
                    }
                    else
                    {
                        parameterValue = definition.getType().valueFromXML(parametersElt, parameterName, Optional.empty());
                    }
                    parameters.put(parameterName, parameterValue);
                }
            }
            
            // Run the schedulable job
            Runnable deployJob = new DefaultRunnable(
                    "deploy-starter-" + System.currentTimeMillis(),
                    new I18nizableText("plugin.deploy-starter", "PLUGINS_DEPLOY_STARTER_SCHEDULABLE"),
                    new I18nizableText(StringUtils.EMPTY),
                    FireProcess.NOW,
                    null,
                    schedulable.getId(),
                    true,
                    true,
                    false,
                    null,
                    false,
                    UserPopulationDAO.SYSTEM_USER_IDENTITY,
                    parameters
            );
            
            _scheduler.scheduleJob(deployJob);
        }
    }
}
