/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.cache;

import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.observation.Event;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.web.cache.CacheHelper;
import org.ametys.web.cache.pageelement.PageElementCache;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.synchronization.AbstractSynchronizeObserver;

/**
 * Invalidate the site cache and page element cache when 
 */
public class InvalidateCacheOnQueryChangesObserver extends AbstractSynchronizeObserver
{
    private PageElementCache _inputDataCache;
    private PageElementCache _zoneItemCache;
    private SiteManager _siteManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        
        _inputDataCache = (PageElementCache) manager.lookup(PageElementCache.ROLE + "/inputData");
        _zoneItemCache = (PageElementCache) manager.lookup(PageElementCache.ROLE + "/zoneItem");
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }

    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_QUERY_UPDATED) 
            || event.getId().equals(ObservationConstants.EVENT_QUERY_DELETED);
    }

    @Override
    protected void _internalObserve(Event event, Session liveSession) throws RepositoryException
    {
        if (getLogger().isInfoEnabled())
        {
            getLogger().info("Database changes: " + event + ", invalidating cache");
        }
        
        // clear the whole FO cache
        AmetysObjectIterable<Site> sites = _siteManager.getSites();
        for (Site site : sites)
        {
            try
            {
                CacheHelper.invalidateCache(site, getLogger());
            }
            catch (Exception e)
            {
                getLogger().error("Unable to clear cache of site " + site.getName(), e);
            }
        }
        
        // clear the page element cache
        _inputDataCache.clear();
        _zoneItemCache.clear();
    }

}
