/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.data.sql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.datasource.AbstractDataSourceManager.DataSourceDefinition;
import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.core.datasource.DataSourceClientInteraction.DataSourceType;
import org.ametys.core.datasource.SQLDataSourceManager;
import org.ametys.core.datasource.dbtype.SQLDatabaseTypeExtensionPoint;
import org.ametys.plugins.externaldata.data.DataInclusionException;
import org.ametys.plugins.externaldata.data.DataSourceFactory;
import org.ametys.plugins.externaldata.data.Query;
import org.ametys.plugins.externaldata.data.Query.ResultType;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * SQL Data Source &amp; Query Factory.
 * Must provide these additional configuration information for the Data Source :
 * - "driver" : JDBC driver.
 * - "url" : JDBC url.
 * - "user" : JDBC user.
 * - "password" : JDBC password.
 * Must provide these additional configuration information for the Query :
 * - "queryString" : the SQL query as a String, which can contain parameters in the form ${parameterName}.
 */
public class SqlDataSourceFactory implements DataSourceFactory<SqlQuery, SqlQueryResult>, PluginAware, Serviceable
{
    /** Query string configuration parameter. */
    public static final String QUERY_CONFIGURATION_QUERYSTRING = "queryString";
    
    /** Query configuration parameters. */
    public static final List<String> QUERY_CONFIGURATION_PARAMETERS = Arrays.asList(QUERY_CONFIGURATION_QUERYSTRING);

    private String _id;

    private SQLDatabaseTypeExtensionPoint _sqlDatabaseTypeExtensionPoint;
    private SQLDataSourceManager _sqlDataSourceManager;
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _sqlDataSourceManager = (SQLDataSourceManager) manager.lookup(SQLDataSourceManager.ROLE);
        _sqlDatabaseTypeExtensionPoint = (SQLDatabaseTypeExtensionPoint) manager.lookup(SQLDatabaseTypeExtensionPoint.ROLE);
    }
    
    @Override
    public Collection<DataSourceType> getHandledTypes()
    {
        return Collections.singleton(DataSourceType.SQL);
    }
    
    @Override
    public Collection<String> getQueryConfigurationParameters(String type)
    {
        return QUERY_CONFIGURATION_PARAMETERS;
    }
    
    @Override
    public SqlQuery buildQuery(String id, String type, String name, String description, ResultType resultType, String dataSourceId, Map<String, String> additionalConfiguration) throws DataInclusionException
    {
        String queryString = additionalConfiguration.get(QUERY_CONFIGURATION_QUERYSTRING);
        
        if (StringUtils.isBlank(queryString))
        {
            throw new DataInclusionException("Unable to build the SQL query : the query is required");
        }
        
        SqlQuery query = new SqlQuery();
        
        query.setFactory(this._id);
        query.setId(id);
        query.setName(name);
        query.setDescription(description);
        query.setQueryString(queryString);
        query.setResultType(resultType);
        query.setDataSourceId(dataSourceId);
        
        return query;
    }
    
    public SqlQueryResult execute(SqlQuery query, Map<String, String> parameterValues) throws DataInclusionException
    {
        return execute(query, parameterValues, 0, Integer.MAX_VALUE);
    }
    
    @Override
    public SqlQueryResult execute(SqlQuery sqlQuery, Map<String, String> parameterValues, int offset, int limit) throws DataInclusionException
    {
        String query = sqlQuery.getQueryString();
        for (String parameterName : sqlQuery.getParameters().keySet())
        {
            if (!parameterValues.containsKey(parameterName))
            {
                String regexp = "[^ =,\\s]*=[^$ ]*\\$\\{" + parameterName + "(\\[[^\\]]*\\])?\\}[^ ]*";
                query = query.replaceAll(regexp, "1=1");
            }
        }
        
        // Replace all the parameters by "?" placeholders.
        query = query.replaceAll(Query.PARAMETER_PATTERN, "?");
        
        if (offset != 0 || limit != Integer.MAX_VALUE)
        {
            String dataSourceId = sqlQuery.getDataSourceId();
            DataSourceDefinition sqlDefinition = _sqlDataSourceManager.getDataSourceDefinition(dataSourceId);
            String defUrl = (String) sqlDefinition.getParameters().get("url");
            String dbType = ConnectionHelper.getDatabaseType(defUrl);
            
            query = _sqlDatabaseTypeExtensionPoint.languageLimitQuery(dbType, query, Integer.toString(limit), Integer.toString(offset));
        }
        
        ResultSet resultSet = null;
        
        try (Connection connection = ConnectionHelper.getConnection(sqlQuery.getDataSourceId()); PreparedStatement stmt = connection.prepareStatement(query))
        {
            // Set the parameters.
            int paramIndex = 1;
            for (String parameterName : sqlQuery.getParameters().keySet())
            {
                if (parameterValues.containsKey(parameterName))
                {
                    String value = parameterValues.get(parameterName);
                    stmt.setString(paramIndex, value);
                    paramIndex++;
                }
            }
            
            resultSet = stmt.executeQuery();
            
            SqlQueryResult result = new SqlQueryResult(resultSet);
            
            // At this point, all the data is loaded into the result object, so
            // the resultSet and JDBC objects can safely be cleaned up.
            
            return result;
        }
        catch (SQLException e)
        {
            throw new DataInclusionException("Unable to execute the query.", e);
        }
        finally
        {
            ConnectionHelper.cleanup(resultSet);
        }
    }
}
