/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extrausermgt.authentication.cas;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletContext;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Session;
import org.apache.cocoon.environment.http.HttpEnvironment;
import org.apache.commons.lang3.StringUtils;
import org.jasig.cas.client.authentication.AuthenticationFilter;
import org.jasig.cas.client.util.AbstractCasFilter;
import org.jasig.cas.client.util.HttpServletRequestWrapperFilter;
import org.jasig.cas.client.validation.Assertion;

import org.ametys.core.authentication.AbstractCredentialProvider;
import org.ametys.core.authentication.AuthenticateAction;
import org.ametys.core.authentication.BlockingCredentialProvider;
import org.ametys.core.authentication.NonBlockingCredentialProvider;
import org.ametys.core.servletwrapper.filter.ServletFilterWrapper;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.URIUtils;

/**
 * This manager gets the credentials given by an authentication CAS filter.
 * <br>
 * The filter must set the 'remote user' header into the request. <br>
 * <br>
 * This manager can not get the password of the connected user: the user is
 * already authenticated. This manager should not be associated with a
 * <code>UsersManagerAuthentication</code>
 */
public class CASCredentialProvider extends AbstractCredentialProvider implements NonBlockingCredentialProvider, BlockingCredentialProvider, Contextualizable
{
    /** Parameter name for server url  */
    public static final String PARAM_SERVER_URL = "authentication.cas.serverUrl";
    
    /** Parameter name for "request proxy tickets" */
    private static final String __PARAM_REQUEST_PROXY_TICKETS = "authentication.cas.requestProxyTickets";
    
    /** Parameter name for "accept any proxy" */
    private static final String __PARAM_ACCEPT_ANY_PROXY = "authentication.cas.acceptAnyProxy";
    
    /** Parameter name for authorized proxy chains */
    private static final String __PARAM_AUTHORIZED_PROXY_CHAINS = "authentication.cas.authorizedProxyChain";
    
    /** Parameter name for the gateway mode */
    private static final String __PARAM_GATEWAY_ENABLED = "authentication.cas.enableGateway";
    
    /** Cas server URL with context (https://cas-server ou https://cas-server/cas) */
    protected String _serverUrl;
    
    private Context _context;

    /** Should the application request proxy tickets */
    private boolean _requestProxyTickets;
    /** Should the application accept any proxy */
    private boolean _acceptAnyProxy;
    /**
     * Authorized proxy chains, which is
     *  a newline-delimited list of acceptable proxy chains.
     *  A proxy chain includes a whitespace-delimited list of valid proxy URLs.
     *  Only one proxy chain needs to match for the login to be successful.
     */
    private String _authorizedProxyChains;
    /** Should the cas gateway mode be used */
    private boolean _gatewayModeEnabled;

    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void init(String id, String cpModelId, Map<String, Object> paramValues, String label) throws Exception
    {
        super.init(id, cpModelId, paramValues, label);
        _serverUrl = (String) paramValues.get(PARAM_SERVER_URL);
        _requestProxyTickets = (boolean) paramValues.get(__PARAM_REQUEST_PROXY_TICKETS);
        _acceptAnyProxy = (boolean) paramValues.get(__PARAM_ACCEPT_ANY_PROXY);
        _authorizedProxyChains = (String) paramValues.get(__PARAM_AUTHORIZED_PROXY_CHAINS);
        _gatewayModeEnabled = (boolean) paramValues.get(__PARAM_GATEWAY_ENABLED);
    }

    @Override
    public boolean blockingIsStillConnected(UserIdentity userIdentity, Redirector redirector) throws Exception
    {
        return StringUtils.equals(userIdentity.getLogin(), _getLoginFromFilter(false, redirector));
    }
    
    @Override
    public boolean nonBlockingIsStillConnected(UserIdentity userIdentity, Redirector redirector) throws Exception
    {
        return blockingIsStillConnected(userIdentity, redirector);
    }
    
    private String _getLoginFromFilter(boolean gateway, Redirector redirector) throws Exception
    {
        Map objectModel = ContextHelper.getObjectModel(_context);
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        if (request.getRequestURI().startsWith(request.getContextPath() + "/plugins/core/authenticate/") && "true".equals(request.getParameter("proxy")))
        {
            String redirectUrl = "cocoon://_plugins/extra-user-management/ametysCasProxy";
            getLogger().debug("Redirecting to '{}'", redirector);
            redirector.redirect(true, redirectUrl);
            return null;
        }
        
        StringBuffer serverName = new StringBuffer(request.getServerName());
        
        // Build an URI without :80 (http) and without :443 (https)
        if (request.isSecure())
        {
            if (request.getServerPort() != 443)
            {
                serverName.append(":");
                serverName.append(request.getServerPort());
            }
        }
        else
        {
            if (request.getServerPort() != 80)
            {
                serverName.append(":");
                serverName.append(request.getServerPort());
            }
        }
        
        String name = serverName.toString();
        
        // Create the filter chain.
        List<ServletFilterWrapper> runtimeFilters = new ArrayList<>();
        
        ServletContext servletContext = (ServletContext) objectModel.get(HttpEnvironment.HTTP_SERVLET_CONTEXT);
        Map<String, String> parameters = new HashMap<>();
        
        try
        {
            // Authentication filter.
            parameters.put("casServerLoginUrl", _serverUrl + "/login");
            parameters.put("serverName", name);
            parameters.put("gateway", String.valueOf(gateway));
            ServletFilterWrapper runtimeFilter = new ServletFilterWrapper(new AuthenticationFilter());
            runtimeFilter.init(parameters, servletContext);
            runtimeFilters.add(runtimeFilter);
            
            // Ticket validation filter.
            parameters.clear();
            parameters.put("casServerUrlPrefix", _serverUrl);
            parameters.put("serverName", name);
            if (_acceptAnyProxy)
            {
                parameters.put("acceptAnyProxy", "true");
            }
            else
            {
                parameters.put("allowedProxyChains", _authorizedProxyChains);
            }
            
            if (_requestProxyTickets && StringUtils.isNotEmpty(request.getParameter("ticket")))
            {
                String proxyCallbackUrl = "https://" + name + _getProxyCallbackRelativeUrl(request);
                getLogger().debug("The computed proxy callback url is: {}", proxyCallbackUrl);
                parameters.put("proxyCallbackUrl", proxyCallbackUrl);
                parameters.put("proxyGrantingTicketStorageClass", CasProxyGrantingTicketManager.class.getName());
                parameters.put("ticketValidatorClass", AmetysCas20ProxyTicketValidator.class.getName());
            }
            
            runtimeFilter = new ServletFilterWrapper(new AmetysCas20ProxyReceivingTicketValidationFilter());
            runtimeFilter.init(parameters, servletContext);
            runtimeFilters.add(runtimeFilter);
            
            // Ticket validation filter.
            parameters.clear();
            runtimeFilter = new ServletFilterWrapper(new HttpServletRequestWrapperFilter());
            runtimeFilter.init(parameters, servletContext);
            runtimeFilters.add(runtimeFilter);
            
            getLogger().debug("Executing CAS filter chain...");
            
            // Execute the filter chain.
            for (ServletFilterWrapper filter : runtimeFilters)
            {
                filter.doFilter(objectModel, redirector);
            }
        }
        finally
        {
            getLogger().debug("Destroying CAS filter chain...");
            
            // Release filters
            for (ServletFilterWrapper filter : runtimeFilters)
            {
                filter.destroy();
            }
        }
        
        // If a redirect was sent, the getSession call won't work.
        if (!redirector.hasRedirected())
        {
            return _getLogin(request);
        }
        
        return null;
    }
    
    private String _getProxyCallbackRelativeUrl(Request request)
    {
        Map<String, String> params = new HashMap<>();
        params.put("proxy", "true");
        
        String userPopulationId = (String) request.getAttribute(AuthenticateAction.REQUEST_ATTRIBUTE_USER_POPULATION_ID);
        if (StringUtils.isNotEmpty(userPopulationId))
        {
            params.put(AuthenticateAction.REQUEST_PARAMETER_POPULATION_NAME, userPopulationId);
        }
        
        Integer cpIndex = _getRunningCpIndex(request);
        if (cpIndex.equals(-1))
        {
            // Force to authenticate over the first credential provider
            cpIndex = 0;
        }
        
        @SuppressWarnings("unchecked")
        List<String> contexts = (List<String>) request.getAttribute("Runtime:Contexts");
        
        String contextAsString = contexts != null ? StringUtils.join(contexts.toArray(), ',') : "";
        params.put("contexts", contextAsString);
        
        return URIUtils.buildURI(request.getContextPath() + "/plugins/core/authenticate/" + cpIndex.toString(), params);
    }
    
    private Integer _getRunningCpIndex(Request request)
    {
        Integer cpIndex = (Integer) request.getAttribute("Runtime:RequestCredentialProviderIndex");
        if (cpIndex != null)
        {
            return cpIndex;
        }
        
        Session session = request.getSession(false);
        if (session != null)
        {
            Integer formerRunningCredentialProviderIndex = (Integer) session.getAttribute("Runtime:ConnectingCredentialProviderIndexLastKnown");
            if (formerRunningCredentialProviderIndex != null)
            {
                return formerRunningCredentialProviderIndex;
            }
        }
        
        return -1;
    }

    @Override
    public boolean blockingGrantAnonymousRequest()
    {       
        return false;
    }
    
    @Override
    public boolean nonBlockingGrantAnonymousRequest()
    {
        return false;
    }

    @Override
    public UserIdentity blockingGetUserIdentity(Redirector redirector) throws Exception
    {
        String userLogin = _getLoginFromFilter(false, redirector);
        
        if (redirector.hasRedirected())
        {
            return null;
        }

        if (userLogin == null)
        {
            throw new IllegalStateException("CAS authentication needs a CAS filter.");
        }
        
        return new UserIdentity(userLogin, null);
    }
    
    @Override
    public UserIdentity nonBlockingGetUserIdentity(Redirector redirector) throws Exception
    {
        if (!_gatewayModeEnabled)
        {
            return null;
        }
        
        String userLogin = _getLoginFromFilter(true, redirector);
        if (userLogin == null)
        {
            return null;
        }
        
        return new UserIdentity(userLogin, null);
    }

    @Override
    public void blockingUserNotAllowed(Redirector redirector) throws Exception
    {
        // Nothing to do.
    }
    
    @Override
    public void nonBlockingUserNotAllowed(Redirector redirector) throws Exception
    {
        // Nothing to do.
    }

    @Override
    public void blockingUserAllowed(UserIdentity userIdentity, Redirector redirector)
    {
        // Empty method, nothing more to do.
    }
    
    @Override
    public void nonBlockingUserAllowed(UserIdentity userIdentity, Redirector redirector)
    {
        // Empty method, nothing more to do.
    }

    public boolean requiresNewWindow()
    {
        return true;
    }
    
    /**
     * Get the connected user login from the request or session.
     * @param request the request object.
     * @return the connected user login or null.
     */
    protected String _getLogin(Request request)
    {
        String userLogin = null;
        
        Session session = request.getSession(false);
        
        final Assertion assertion = (Assertion) (session == null ? request.getAttribute(AbstractCasFilter.CONST_CAS_ASSERTION) : session.getAttribute(AbstractCasFilter.CONST_CAS_ASSERTION));
        
        if (assertion != null)
        {
            userLogin = assertion.getPrincipal().getName();
        }
        return userLogin;
    }
}
