/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extrausermgt.graph;

import java.net.URI;
import java.util.Map;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.SessionAttributeProvider;
import org.ametys.plugins.extrausermgt.authentication.msal.AbstractMSALCredentialProvider;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.azure.identity.ClientSecretCredential;
import com.azure.identity.ClientSecretCredentialBuilder;
import com.microsoft.graph.serviceclient.GraphServiceClient;
import com.microsoft.graph.users.item.UserItemRequestBuilder;
import com.microsoft.kiota.authentication.AccessTokenProvider;
import com.microsoft.kiota.authentication.AllowedHostsValidator;
import com.microsoft.kiota.authentication.BaseBearerTokenAuthenticationProvider;

/**
 * Provide a Graph client for a user.
 * Depending on the configuration, the client can either be authenticated
 * using an application credentials or the current user Entra id token
 */
public class GraphClientProvider extends AbstractLogEnabled implements Initializable, Serviceable, Component
{
    /** The avalon role */
    public static final String ROLE = GraphClientProvider.class.getName();
    
    private static final String __SCOPE = "https://graph.microsoft.com/.default";

    private GraphServiceClient _graphClient;
    
    private CurrentUserProvider _currentUserProvider;
    private SessionAttributeProvider _sessionAttributeProvider;
    private UserManager _userManager;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _sessionAttributeProvider = (SessionAttributeProvider) manager.lookup(SessionAttributeProvider.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
    }
    
    public void initialize()
    {
        if ((boolean) Config.getInstance().getValue("org.ametys.plugins.extra-user-management.graph.useadmin"))
        {
            String appId = Config.getInstance().getValue("org.ametys.plugins.extra-user-management.graph.appid");
            String clientSecret = Config.getInstance().getValue("org.ametys.plugins.extra-user-management.graph.clientsecret");
            String tenant = Config.getInstance().getValue("org.ametys.plugins.extra-user-management.graph.tenant");
            
            ClientSecretCredential clientSecretCredential = new ClientSecretCredentialBuilder()
                    .clientId(appId)
                    .clientSecret(clientSecret)
                    .tenantId(tenant)
                    .build();
            
            _graphClient = new GraphServiceClient(clientSecretCredential, __SCOPE);
        }
    }

    /**
     * Get a graph client for the user.
     * No check is performed to ensure that the user is actually a valid graph user
     * @param userIdentity the user identity
     * @return the client
     * @throws GraphClientException when its not possible to retrieve a client for the user
     */
    public UserItemRequestBuilder getUserRequestBuilder(UserIdentity userIdentity) throws GraphClientException
    {
        if (_graphClient != null)
        {
            return _graphClient.users().byUserId(_getUserPrincipalName(userIdentity));
        }
        else
        {
            // This mode can only provide a client for the current user
            // check that the requested user is the current user before getting the client
            if (!userIdentity.equals(_currentUserProvider.getUser()))
            {
                throw new GraphClientException(userIdentity.toString() + " is not the current user. A graph client can only be retrieved for the current user");
            }
            return _sessionAttributeProvider.getSessionAttribute(AbstractMSALCredentialProvider.ACCESS_TOKEN_SESSION_ATTRIBUTE)
                .filter(String.class::isInstance)
                .map(String.class::cast)
                .map(this::_getClientFromToken)
                .map(GraphServiceClient::me)
                .orElseThrow(() -> new GraphClientException("The current user " + _currentUserProvider.getUser().toString() + "is not logged with Entra ID."));
        }
    }
    
    /**
     * Get an authenticated Graph client
     * @return a Graph client
     * @throws GraphClientException when its not possible to retrieve a client for the user
     */
    public GraphServiceClient getGraphClient() throws GraphClientException
    {
        if (_graphClient != null)
        {
            return _graphClient;
        }
        else
        {
            return _sessionAttributeProvider.getSessionAttribute(AbstractMSALCredentialProvider.ACCESS_TOKEN_SESSION_ATTRIBUTE)
                .filter(String.class::isInstance)
                .map(String.class::cast)
                .map(this::_getClientFromToken)
                .orElseThrow(() -> new GraphClientException("The current user " + _currentUserProvider.getUser().toString() + "is not logged with Entra ID."));
        }
    }
    
    private GraphServiceClient _getClientFromToken(String token)
    {
        AccessTokenAuthenticationProvider accessTokenAuthenticationProvider = new AccessTokenAuthenticationProvider(token);
        return new GraphServiceClient(new BaseBearerTokenAuthenticationProvider(accessTokenAuthenticationProvider));
    }
    
    private String _getUserPrincipalName(UserIdentity userIdentity) throws GraphClientException
    {
        if ("email".equals(Config.getInstance().getValue("org.ametys.plugins.extra-user-management.graph.authmethod")))
        {
            User user = _userManager.getUser(userIdentity);
            String email = user.getEmail();
            if (StringUtils.isBlank(email))
            {
                throw new GraphClientException("The user '" + userIdentity.toString() + "' has no email address set, thus exchange cannot be contacted using 'email' authentication method");
            }
            
            return email;
        }
        else
        {
            return userIdentity.getLogin();
        }
    }
    
    /**
     * Exception related to the Graph client
     */
    public static class GraphClientException extends Exception
    {
        /**
         * Constructs a new exception with the specified detail message.
         * @param message the detail message.
         */
        public GraphClientException(String message)
        {
            super(message);
        }
    }
    
    private static class AccessTokenAuthenticationProvider implements AccessTokenProvider
    {
        private String _accessToken;

        public AccessTokenAuthenticationProvider(String accessToken)
        {
            _accessToken = accessToken;
        }

        public String getAuthorizationToken(URI uri, Map<String, Object> additionalAuthenticationContext)
        {
            return _accessToken;
        }

        public AllowedHostsValidator getAllowedHostsValidator()
        {
            return null;
        }
    }
}
