/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extrausermgt.authentication.oidc;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Session;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.authentication.AbstractCredentialProvider;
import org.ametys.core.authentication.AuthenticateAction;
import org.ametys.core.authentication.BlockingCredentialProvider;
import org.ametys.core.authentication.NonBlockingCredentialProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.directory.NotUniqueUserException;
import org.ametys.core.user.directory.StoredUser;
import org.ametys.core.user.directory.UserDirectory;
import org.ametys.core.user.population.UserPopulation;
import org.ametys.plugins.extrausermgt.authentication.oidc.endofauthenticationprocess.EndOfAuthenticationProcess;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.workspaces.extrausermgt.authentication.oidc.OIDCCallbackAction;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.proc.BadJOSEException;
import com.nimbusds.jwt.JWT;
import com.nimbusds.oauth2.sdk.AuthorizationCode;
import com.nimbusds.oauth2.sdk.AuthorizationCodeGrant;
import com.nimbusds.oauth2.sdk.AuthorizationGrant;
import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.oauth2.sdk.RefreshTokenGrant;
import com.nimbusds.oauth2.sdk.ResponseType;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.oauth2.sdk.SerializeException;
import com.nimbusds.oauth2.sdk.TokenErrorResponse;
import com.nimbusds.oauth2.sdk.TokenRequest;
import com.nimbusds.oauth2.sdk.TokenResponse;
import com.nimbusds.oauth2.sdk.auth.ClientAuthentication;
import com.nimbusds.oauth2.sdk.auth.ClientSecretBasic;
import com.nimbusds.oauth2.sdk.auth.Secret;
import com.nimbusds.oauth2.sdk.http.HTTPResponse;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.oauth2.sdk.id.Issuer;
import com.nimbusds.oauth2.sdk.id.State;
import com.nimbusds.oauth2.sdk.token.AccessToken;
import com.nimbusds.oauth2.sdk.token.RefreshToken;
import com.nimbusds.openid.connect.sdk.AuthenticationRequest;
import com.nimbusds.openid.connect.sdk.OIDCTokenResponse;
import com.nimbusds.openid.connect.sdk.OIDCTokenResponseParser;
import com.nimbusds.openid.connect.sdk.Prompt;
import com.nimbusds.openid.connect.sdk.UserInfoRequest;
import com.nimbusds.openid.connect.sdk.UserInfoResponse;
import com.nimbusds.openid.connect.sdk.claims.IDTokenClaimsSet;
import com.nimbusds.openid.connect.sdk.claims.UserInfo;
import com.nimbusds.openid.connect.sdk.token.OIDCTokens;
import com.nimbusds.openid.connect.sdk.validators.IDTokenValidator;

/**
 * Sign in (through Google, facebook...) using the OpenId Connect (OIDC) protocol.
 */
public abstract class AbstractOIDCCredentialProvider extends AbstractCredentialProvider implements OIDCBasedCredentialProvider, BlockingCredentialProvider, NonBlockingCredentialProvider, Contextualizable, Serviceable
{
    /** Session attribute for OIDC */
    public static final String REDIRECT_URI_SESSION_ATTRIBUTE = "oidc_actualRedirectUri";
    /** Session attribute for OIDC*/
    public static final String TOKEN_SESSION_ATTRIBUTE = "oidc_token";
    /** Session date attribute for OIDC*/
    public static final String EXPDATE_SESSION_ATTRIBUTE = "oidc_expirationDate";
    /** Session attribute for OIDC*/
    public static final String REFRESH_TOKEN_SESSION_ATTRIBUTE = "oidc_refreshToken";
    /** Session attribute for OIDC*/
    public static final String STATE_SESSION_ATTRIBUTE = "oidc_state";

    private static final String __ATTRIBUTE_SILENT = "oidc_silent";
    
    /** Scope  for the authentication request */
    protected Scope _scope;
    
    /** URI for the authentication request */
    protected URI _authUri;
    
    /** URI for the token request */
    protected URI _tokenEndpointUri;
    
    /** URI for the user info request */
    protected URI _userInfoEndpoint;
    
    /** jwk URL for the validation of the token */
    protected URL _jwkSetURL;
    
    /** Issuer  for the validation of the token */
    protected Issuer _iss;
    
    /** Ametys context */
    protected Context _context;
    /** Client ID */
    protected ClientID _clientID;
    /** Client secret */
    protected Secret _clientSecret;
    
    /** If we should try to authenticate silently */
    protected boolean _silent;
    
    private EndOfAuthenticationProcess _endOfAuthenticationProcess;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _endOfAuthenticationProcess = (EndOfAuthenticationProcess) manager.lookup(EndOfAuthenticationProcess.ROLE);
    }
    
    @Override
    public void init(String id, String cpModelId, Map<String, Object> paramValues, String label) throws Exception
    {
        super.init(id, cpModelId, paramValues, label);
        _clientID = new ClientID(paramValues.get("authentication.oidc.idclient").toString());
        _clientSecret = new Secret(paramValues.get("authentication.oidc.clientsecret").toString());
        _silent = (boolean) paramValues.get("authentication.oidc.silent");

        initUrisScope();
    }
    
    public String getClientId()
    {
        return _clientID.getValue();
    }
    
    public String getIssuer()
    {
        return _iss.getValue();
    }
    
    public URL getJwkSetURL()
    {
        return _jwkSetURL;
    }
    
    /**
     * get the client authentication info for the token end point
     * @return the client authentication
     */
    protected ClientAuthentication getClientAuthentication()
    {
        return new ClientSecretBasic(_clientID, _clientSecret);
    }
    
    public boolean blockingGrantAnonymousRequest()
    {
        return false;
    }
    
    @Override
    public boolean nonBlockingGrantAnonymousRequest()
    {
        return false;
    }
    
    public boolean blockingIsStillConnected(UserIdentity userIdentity, Redirector redirector) throws Exception
    {
        Request request = ContextHelper.getRequest(_context);
        Session session = request.getSession(true);
   
        Date expDat = (Date) session.getAttribute(EXPDATE_SESSION_ATTRIBUTE);
        if (new Date().before(expDat))
        {
            return true;
        }
        
        RefreshToken refreshToken = (RefreshToken) session.getAttribute(REFRESH_TOKEN_SESSION_ATTRIBUTE);
        AuthorizationGrant refreshTokenGrant = new RefreshTokenGrant(refreshToken);
        
        // The credentials to authenticate the client at the token endpoint
        ClientAuthentication clientAuth = getClientAuthentication();
        
        // Make the token request
        OIDCTokens tokens = requestToken(clientAuth, refreshTokenGrant);

        // idToken to validate the token
        JWT idToken = tokens.getIDToken();
        // accessToken to be able to access the user info
        AccessToken accessToken = tokens.getAccessToken();
        IDTokenClaimsSet claims = validateIdToken(idToken);
        session.setAttribute(EXPDATE_SESSION_ATTRIBUTE, claims.getExpirationTime());
        session.setAttribute(TOKEN_SESSION_ATTRIBUTE, accessToken);
        
        return true;
    }
    
    @Override
    public boolean nonBlockingIsStillConnected(UserIdentity userIdentity, Redirector redirector) throws Exception
    {
        return blockingIsStillConnected(userIdentity, redirector);
    }
    
    private UserIdentity _login(boolean silent, Redirector redirector) throws Exception
    {
        Request request = ContextHelper.getRequest(_context);
        Session session = request.getSession(true);
        
        URI redirectUri = _buildRedirectUri();

        getLogger().debug("OIDCCredentialProvider callback URI: {}", redirectUri);
        
        boolean wasSilent = false;
        if (silent)
        {
            wasSilent = "true".equals(session.getAttribute(__ATTRIBUTE_SILENT));
        }
   
        String code = request.getParameter("code");
        // if the code is null, then this is the first time the user sign-in
        // if no state are stored in session, then the code belongs to a previous session. Restart
        if (code == null || session.getAttribute(STATE_SESSION_ATTRIBUTE) == null)
        {
            signIn(redirector, redirectUri, silent, wasSilent, session);
            return null;
        }

        // we got an authorization code
        // but first, check the state to prevent CSRF attacks
        checkState();
        AuthorizationCode authCode = new AuthorizationCode(code);
        // get the tokens (id token and access token)
        OIDCTokens tokens = requestToken(authCode, redirectUri);

        // idToken to validate the token
        JWT idToken = tokens.getIDToken();
        // accessToken to be able to access the user info
        AccessToken accessToken = tokens.getAccessToken();
        RefreshToken refreshToken = tokens.getRefreshToken();
        
        session.setAttribute(REFRESH_TOKEN_SESSION_ATTRIBUTE, refreshToken);
        
        // validate id token
        IDTokenClaimsSet claims = validateIdToken(idToken);

        // set expirationTime
        claims.getExpirationTime();
        session.setAttribute(EXPDATE_SESSION_ATTRIBUTE, claims.getExpirationTime());
        
        UserInfo userInfo = getUserInfo(accessToken);
        
        // then the user is finally logged in
        return getUserIdentity(userInfo, request, redirector);
    }

    public UserIdentity blockingGetUserIdentity(Redirector redirector) throws Exception
    {
        return _login(false, redirector);
    }
    
    public UserIdentity nonBlockingGetUserIdentity(Redirector redirector) throws Exception
    {
        if (!_silent)
        {
            return null;
        }
        
        return _login(true, redirector);
    }
    
    public void blockingUserNotAllowed(Redirector redirector) throws Exception
    {
        // Nothing to do.
    }

    @Override
    public void nonBlockingUserNotAllowed(Redirector redirector) throws Exception
    {
        // Nothing to do.
    }

    public void blockingUserAllowed(UserIdentity userIdentity, Redirector redirector) throws Exception
    {
        Request request = ContextHelper.getRequest(_context);
        Session session = request.getSession(true);
        String redirectUri = (String) session.getAttribute(AbstractOIDCCredentialProvider.REDIRECT_URI_SESSION_ATTRIBUTE);
        redirector.redirect(true, redirectUri);
    }
    
    @Override
    public void nonBlockingUserAllowed(UserIdentity userIdentity, Redirector redirector) throws Exception
    {
        blockingUserAllowed(userIdentity, redirector);
    }

    public boolean requiresNewWindow()
    {
        return true;
    }
    
    private UserPopulation _getPopulation(Request request)
    {
        @SuppressWarnings("unchecked")
        List<UserPopulation> userPopulations = (List<UserPopulation>) request.getAttribute(AuthenticateAction.REQUEST_ATTRIBUTE_AVAILABLE_USER_POPULATIONS_LIST);

        // If the list has only one element
        if (userPopulations.size() == 1)
        {
            return userPopulations.get(0);
        }

        // In this list a population was maybe chosen?
        final String chosenUserPopulationId = (String) request.getAttribute(AuthenticateAction.REQUEST_ATTRIBUTE_USER_POPULATION_ID);
        if (StringUtils.isNotBlank(chosenUserPopulationId))
        {
            return userPopulations.stream()
                    .filter(userPopulation -> StringUtils.equals(userPopulation.getId(), chosenUserPopulationId))
                    .findFirst()
                    .get();
        }

        // Cannot work here...
        throw new IllegalStateException("The " + this.getClass().getName() + " does not work when population is not known");
    }
    
    /**
     * Initialize the URIs
     * @throws AccessDeniedException If an error occurs
     */
    protected abstract void initUrisScope() throws AccessDeniedException;
    
    /**
     * Builds the redirect URI and the actual redirect URI
     * @return The redirect <code>URI</code> and saves the actual redirect <code>URI</code>
     * @throws URISyntaxException If an error occurs
     */
    private URI _buildRedirectUri() throws URISyntaxException
    {
        Request request = ContextHelper.getRequest(_context);
        
        // creation of the actual redirect URI (The one we actually want to go back to)
        StringBuilder actualRedirectUri = new StringBuilder(request.getRequestURI());
        String queryString = request.getQueryString();
        if (queryString != null)
        {
            // remove any existing code or state from the request param if any exist, they are outdated
            queryString = Stream.of(StringUtils.split(queryString, "&"))
                    .filter(str -> !StringUtils.startsWithAny(str, "code=", "state="))
                    .collect(Collectors.joining("&"));
            if (StringUtils.isNotEmpty(queryString))
            {
                actualRedirectUri.append("?");
                actualRedirectUri.append(queryString);
            }
        }
        
        // saving the actualRedirectUri to enable its use in "OIDCCallbackAction"
        Session session = request.getSession(true);
        session.setAttribute(REDIRECT_URI_SESSION_ATTRIBUTE, actualRedirectUri.toString());
        
        // creation of redirect URI (the issuer (google, facebook, etc.) is going to redirect to)
        return buildAbsoluteURI(request, OIDCCallbackAction.CALLBACK_URL);
    }

    /**
     * Computes the callback uri
     * @param request the current request
     * @param path the callback path
     * @return the callback uri
     */
    protected URI buildAbsoluteURI(Request request, String path)
    {
        StringBuilder uriBuilder = new StringBuilder()
            .append(request.getScheme())
            .append("://")
            .append(request.getServerName());
        
        if (request.isSecure())
        {
            if (request.getServerPort() != 443)
            {
                uriBuilder.append(":");
                uriBuilder.append(request.getServerPort());
            }
        }
        else
        {
            if (request.getServerPort() != 80)
            {
                uriBuilder.append(":");
                uriBuilder.append(request.getServerPort());
            }
        }

        uriBuilder.append(request.getContextPath());
        uriBuilder.append(path);
        
        return URI.create(uriBuilder.toString());
    }
    
    /**
     * Sign the user in by sending an authentication request to the issuer
     * @param redirector The redirector
     * @param redirectUri The redirect URI
     * @param silent if the user should be silently signed in 
     * @param wasSilent indicates that we already passed through this, to prevent infinite loops
     * @param session The current session
     * @throws ProcessingException If an error occurs
     * @throws IOException If an error occurs
     */
    protected void signIn(Redirector redirector, URI redirectUri, boolean silent, boolean wasSilent, Session session) throws ProcessingException, IOException
    {
        // sign-in request: redirect the client through the actual authentication process

        if (wasSilent)
        {
            // already passed through this, there should have been some error somewhere
            return;
        }
        
        if (silent)
        {
            session.setAttribute(__ATTRIBUTE_SILENT, "true");
        }
        
        // creation of the state used to secure the process
        State state = new State();
        session.setAttribute(STATE_SESSION_ATTRIBUTE, state);

        // compose the request
        AuthenticationRequest authenticationRequest = new AuthenticationRequest.Builder(new ResponseType(ResponseType.Value.CODE), _scope, _clientID, redirectUri)
                                                                               .endpointURI(_authUri)
                                                                               .state(state)
                                                                               .prompt(silent ? Prompt.Type.NONE : null)
                                                                               .build();
        
        String authReqURI = authenticationRequest.toURI().toString() + "&access_type=offline";
        
        redirector.redirect(false, authReqURI);
    }
    
    /**
     * Checks the State parameter of the request to prevent CSRF attacks
     * @throws AccessDeniedException If an error occurs
     */
    protected void checkState() throws AccessDeniedException
    {
        Request request = ContextHelper.getRequest(_context);
        Session session = request.getSession(true);
        String storedState = session.getAttribute(STATE_SESSION_ATTRIBUTE).toString();
        String stateRequest = request.getParameter("state");
        
        if (!storedState.equals(stateRequest))
        {
            getLogger().error("OIDC state mismatch. Method checkState of AbstractOIDCCredentialProvider");
            throw new AccessDeniedException("OIDC state mismatch");
        }
        
        session.setAttribute(STATE_SESSION_ATTRIBUTE, null);
    }
    
    /**
     * Request the tokens (ID token and Access token)
     * @param authCode The authorization code from the authentication request
     * @param redirectUri The redirect URI
     * @return The <code>OIDCTokens</code> that contains the access token and the id token
     * @throws AccessDeniedException If an error occurs
     */
    protected OIDCTokens requestToken(AuthorizationCode authCode, URI redirectUri) throws AccessDeniedException
    {
        // token request: checking if the user is known
        TokenRequest tokenReq = new TokenRequest(_tokenEndpointUri, getClientAuthentication(), new AuthorizationCodeGrant(authCode, redirectUri), null);
        // sending request
        HTTPResponse tokenHTTPResp = null;
        try
        {
            tokenHTTPResp = tokenReq.toHTTPRequest().send();
        }
        catch (SerializeException | IOException e)
        {
            getLogger().error("OIDC token request failed ", e);
            throw new AccessDeniedException("OIDC token request failed");
        }

        // cast the HTTPResponse to TokenResponse
        TokenResponse tokenResponse = null;
        try
        {
            tokenResponse = OIDCTokenResponseParser.parse(tokenHTTPResp);
        }
        catch (ParseException e)
        {
            getLogger().error("OIDC token request result invalid ", e);
            throw new AccessDeniedException("OIDC token request result invalid");
        }

        if (tokenResponse instanceof TokenErrorResponse)
        {
            getLogger().error("OIDC token request invalid token response instance of TokenErrorResponse in method requestToken from AbstractOIDCCredentialProvider");
            throw new AccessDeniedException("OIDC token request result invalid");
        }

        // get the tokens
        OIDCTokenResponse  accessTokenResponse = (OIDCTokenResponse) tokenResponse;
        
        return accessTokenResponse.getOIDCTokens();
    }
    
    /**
     * Request the tokens using a refresh token
     * @param clientAuth The client authentication
     * @param refreshTokenGrant The refreshtokenGrant
     * @return The <code>OIDCTokens</code> that contains the access token and the id token
     * @throws AccessDeniedException If an error occurs
     * @throws URISyntaxException If an error occurs
     */
    protected OIDCTokens requestToken(ClientAuthentication clientAuth, AuthorizationGrant refreshTokenGrant) throws AccessDeniedException, URISyntaxException
    {
        // token request: checking if the user is known
        TokenRequest tokenReq = new TokenRequest(_tokenEndpointUri, clientAuth, refreshTokenGrant, null);
        // sending request
       
        HTTPResponse tokenHTTPResp = null;
        try
        {
            tokenHTTPResp = tokenReq.toHTTPRequest().send();
        }
        catch (SerializeException | IOException e)
        {
            getLogger().error("OIDC token request failed ", e);
            throw new AccessDeniedException("OIDC token request failed");
        }

        // cast the HTTPResponse to TokenResponse
        TokenResponse tokenResponse = null;
        try
        {
            tokenResponse = OIDCTokenResponseParser.parse(tokenHTTPResp);
        }
        catch (ParseException e)
        {
            getLogger().error("OIDC token request result invalid ", e);
            throw new AccessDeniedException("OIDC token request result invalid");
        }

        if (tokenResponse instanceof TokenErrorResponse)
        {
            getLogger().error("OIDC token request result invalid: tokenResponse instance of TokenErrorResponse in method requestToken from AbstractOIDCCredentialProvider");
            throw new AccessDeniedException("OIDC token request result invalid");
        }

        // get the tokens
        OIDCTokenResponse  accessTokenResponse = (OIDCTokenResponse) tokenResponse;
        
        return accessTokenResponse.getOIDCTokens();
    }
    
    /**
     * Validate the id token from the token request
     * @param idToken The id token from the token request
     * @return The <code>IDTokenClaimsSet</code> that contains information on the connection such as the expiration time
     * @throws AccessDeniedException If an error occurs
     */
    protected IDTokenClaimsSet validateIdToken(JWT idToken) throws AccessDeniedException
    {
        JWSAlgorithm jwsAlg = JWSAlgorithm.RS256;
        // create validator for signed ID tokens
        IDTokenValidator validator = new IDTokenValidator(_iss, _clientID, jwsAlg, _jwkSetURL);
        IDTokenClaimsSet claims;
        
        try
        {
            claims = validator.validate(idToken, null);
        }
        catch (BadJOSEException e)
        {
            getLogger().error("OIDC invalid : issuer, clientId, jwsAlg or jwkSetURL", e);
            throw new AccessDeniedException("OIDC invalid signature issuer, clientId, jwsAlg or jwkSetURL");
        }
        catch (JOSEException e)
        {
            getLogger().error("OIDC error while validating token", e);
            throw new AccessDeniedException("OIDC error while validating token");
        }
        
        return claims;
    }
    
    /**
     * Request the userInfo using the user info end point and an access token
     * @param accessToken the access token to retrieve the user info
     * @return a representation of the user info from the scope requested with the token
     * @throws IOException if an error occurred while contacting the end point
     * @throws ParseException if an error occurred while parsing the end point answer
     */
    protected UserInfo getUserInfo(AccessToken accessToken) throws IOException, ParseException
    {
        HTTPResponse httpResponse = new UserInfoRequest(_userInfoEndpoint, accessToken).toHTTPRequest().send();
        UserInfoResponse userInfoResponse = UserInfoResponse.parse(httpResponse);
        
        if (userInfoResponse.indicatesSuccess())
        {
            return userInfoResponse.toSuccessResponse().getUserInfo();
        }
        else
        {
            String error = userInfoResponse.toErrorResponse().getErrorObject().toJSONObject().toJSONString();
            getLogger().error("Failed to retrieve the user info. The server indicate the following error :\n" + error);
            throw new AccessDeniedException("Failed to retrieve the user info. The server indicate the following error :\n" + error);
        }
    }

    /**
     * Compute a user identity based on the user info
     * @param userInfo the user info
     * @param request the original request
     * @param redirector the redirector to use if need be
     * @return the identified user info or null if no matching user were found
     * @throws NotUniqueUserException if multiple user matched
     */
    protected UserIdentity getUserIdentity(UserInfo userInfo, Request request, Redirector redirector) throws NotUniqueUserException
    {
        // get the user email
        String login = userInfo.getEmailAddress();
        if (login == null)
        {
            getLogger().error("Email not found, connection canceled ");
            throw new AccessDeniedException("Email not found, connection canceled");
        }
        
        // create a UserIdentity from the email
        UserPopulation userPopulation = _getPopulation(request);
        UserIdentity user = _getUserIdentity(login, userPopulation);
    
        // If we found a UserIdentity, we return it
        if (user != null)
        {
            return user;
        }
    
        // If not, we are going to pre-sign-up the user with its email, firstname and lastname
        String firstName = userInfo.getGivenName();
        String lastName = userInfo.getFamilyName();
        if (firstName == null || lastName == null)
        {
            getLogger().info("The fields could not be pre-filled");
        }
        
        // We call the temporarySignup method from the endOfSignupProcess, which will do nothing if it is in the CMS and temporary sign the user up if it is the site
        _endOfAuthenticationProcess.unexistingUser(login, firstName, lastName, userPopulation, redirector, request);
        
        return null;
    }

    private UserIdentity _getUserIdentity(String login, UserPopulation userPopulation) throws NotUniqueUserException
    {
        StoredUser storedUser = null;
        
        for (UserDirectory userDirectory : userPopulation.getUserDirectories())
        {
            storedUser = userDirectory.getStoredUser(login);

            if (storedUser == null)
            {
                // Try to get user by email
                storedUser = userDirectory.getStoredUserByEmail(login);
            }
            
            if (storedUser != null)
            {
                return userDirectory.getUserIdentity(storedUser);
            }
        }
        
        return null;
    }
}
