/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.edition;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.impl.FileSource;
import org.apache.xml.serializer.OutputPropertiesFactory;
import org.xml.sax.SAXException;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.execution.Extraction;
import org.ametys.plugins.extraction.execution.ExtractionDefinitionReader;

/**
 * Helper that manages the button that saves extraction's modifications
 */
public class SaveExtractionHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Avalon role name */
    public static final String ROLE = SaveExtractionHelper.class.getName();
    
    private static final String EXTRACT_EXTRA_DATA_REGEX = "\\(([-_a-zA-Z]+)\\)";
    
    private SourceResolver _sourceResolver;
    private ExtractionDefinitionReader _definitionReader;
    private UserHelper _userHelper;
    
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _sourceResolver = (SourceResolver) serviceManager.lookup(SourceResolver.ROLE);
        _definitionReader = (ExtractionDefinitionReader) serviceManager.lookup(ExtractionDefinitionReader.ROLE);
        _userHelper = (UserHelper) serviceManager.lookup(UserHelper.ROLE);
    }
    
    /**
     * Saves modifications on extraction. Creates the definition file if it doesn't exist
     * @param relativeDefinitionFilePath The extraction definition file path
     * @param extractionComponents A <code>Map</code> containing definition informations
     * @return <code>true</code> if extraction saving succeed, <code>false</code> otherwise
     * @throws Exception if an error occurs
     */
    public boolean saveExtraction(String relativeDefinitionFilePath, Map<String, Object> extractionComponents) throws Exception
    {
        boolean errorOccurred = false;
        
        String backupFilePath = ExtractionConstants.DEFINITIONS_DIR + relativeDefinitionFilePath + ".tmp";
        Source backupSrc = _sourceResolver.resolveURI(backupFilePath);
        File backupFile = ((FileSource) backupSrc).getFile();
        
        String definitionFilePath = ExtractionConstants.DEFINITIONS_DIR + relativeDefinitionFilePath;
        Source definitionSrc = _sourceResolver.resolveURI(definitionFilePath);
        File definitionFile = ((FileSource) definitionSrc).getFile();
        
        if (!definitionFile.exists())
        {
            throw new IllegalArgumentException("The file " + relativeDefinitionFilePath + " does not exist.");
        }
        
        Extraction extraction = _definitionReader.readExtractionDefinitionFile(definitionFile);

        // Create a backup file
        try
        {
            Files.copy(definitionFile.toPath(), backupFile.toPath());
        }
        catch (IOException e)
        {
            if (getLogger().isErrorEnabled())
            {
                getLogger().error("Error when creating backup '" + definitionFilePath + "' file", e);
            }
        }
        
        try (OutputStream os = Files.newOutputStream(Paths.get(definitionFile.getAbsolutePath())))
        {
            // Create a transformer for saving sax into a file
            TransformerHandler handler = ((SAXTransformerFactory) TransformerFactory.newInstance()).newTransformerHandler();
            
            StreamResult result = new StreamResult(os);
            handler.setResult(result);

            // create the format of result
            Properties format = new Properties();
            format.put(OutputKeys.METHOD, "xml");
            format.put(OutputKeys.INDENT, "yes");
            format.put(OutputKeys.ENCODING, "UTF-8");
            format.put(OutputPropertiesFactory.S_KEY_INDENT_AMOUNT, "4");
            handler.getTransformer().setOutputProperties(format);

            // sax the config
            try
            {
                _saxExtraction(extraction, extractionComponents, handler);
            }
            catch (Exception e)
            {
                if (getLogger().isErrorEnabled())
                {
                    getLogger().error("Error when saxing the extraction definition file '" + definitionFilePath + "'", e);
                }
                errorOccurred = true;
            }
        }
        catch (IOException | TransformerConfigurationException | TransformerFactoryConfigurationError e)
        {
            if (getLogger().isErrorEnabled())
            {
                getLogger().error("Error when trying to modify the extraction definition file '" + definitionFilePath + "'", e);
            }
        }
        
        try
        {
            // Restore the file if an error previously occurred and delete backup
            if (errorOccurred)
            {
                Files.copy(backupFile.toPath(), definitionFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
            Files.deleteIfExists(backupFile.toPath());
        }
        catch (IOException e)
        {
            if (getLogger().isErrorEnabled())
            {
                getLogger().error("Error when restoring backup '" + definitionFilePath + "' file", e);
            }
        }
        
        return !errorOccurred;
    }

    @SuppressWarnings("unchecked")
    private void _saxExtraction(Extraction extraction, Map<String, Object> extractionComponents, TransformerHandler handler) throws SAXException
    {
        handler.startDocument();

        // Set extraction's name attribute
        Map<String, Object> extractionData = (Map<String, Object>) extractionComponents.get("data");
        AttributesImpl attributes = new AttributesImpl();
        if (MapUtils.isNotEmpty(extractionData))
        {
            attributes.addCDATAAttribute("name", (String) extractionData.get("name"));
        }
        XMLUtils.startElement(handler, ExtractionConstants.EXTRACTION_TAG, attributes);
        
        // Set extraction's description
        String descriptionId = extraction.getDescriptionId();
        if (StringUtils.isNotEmpty(descriptionId))
        {
            AttributesImpl descriptionAttributes = new AttributesImpl();
            descriptionAttributes.addCDATAAttribute(ExtractionConstants.DESCRIPTION_IDENTIFIER_ATTRIBUTE_NAME, descriptionId);
            XMLUtils.createElement(handler, ExtractionConstants.DESCRIPTION_TAG, descriptionAttributes);
        }
        
        // Set extraction's author
        _saxAuthor(extraction, handler);

        List<Map<String, Object>> children = (List<Map<String, Object>>) extractionComponents.get("children");
        if (null != children)
        {
            for (Map<String, Object> child : children)
            {
                String tag = (String) child.get("tag");
                switch (tag)
                {
                    case ExtractionConstants.CLAUSES_VARIABLES_TAG:
                        _saxClausesVariables(child, handler);
                        break;
                    case ExtractionConstants.OPTIONAL_COLUMNS_TAG:
                        _saxOptionalColumns(child, handler);
                        break;
                    case ExtractionConstants.QUERY_COMPONENT_TAG:
                    case ExtractionConstants.THESAURUS_COMPONENT_TAG:
                    case ExtractionConstants.COUNT_COMPONENT_TAG:
                    case ExtractionConstants.MAPPING_QUERY_COMPONENT_TAG:
                        _saxExtractionComponent(child, handler);
                        break;
                    default:
                        break;
                }
            }
        }
        
        XMLUtils.endElement(handler, ExtractionConstants.EXTRACTION_TAG);
        handler.endDocument();
    }
    
    private void _saxAuthor(Extraction extraction, TransformerHandler handler) throws SAXException
    {
        // Author
        UserIdentity author = extraction.getAuthor();
        if (author != null)
        {
            _userHelper.saxUserIdentity(author, handler, ExtractionConstants.AUTHOR_TAG);
        }
    }

    @SuppressWarnings("unchecked")
    private void _saxClausesVariables(Map<String, Object> child, TransformerHandler handler) throws SAXException
    {
        Map<String, Object> data = (Map<String, Object>) child.get("data");
        
        List<Map<String, Object>> variables = (List<Map<String, Object>>) data.get("variables");
        if (!variables.isEmpty())
        {
            XMLUtils.startElement(handler, ExtractionConstants.CLAUSES_VARIABLES_TAG);
            for (Map<String, Object> variable : variables)
            {
                AttributesImpl attributes = new AttributesImpl();
                attributes.addCDATAAttribute("name", (String) variable.get("name"));
                attributes.addCDATAAttribute("type", (String) variable.get("type"));
                
                XMLUtils.startElement(handler, "variable", attributes);

                List<String> contentTypeIds = (List<String>) variable.get("contentTypeIds");
                if (contentTypeIds != null && !contentTypeIds.isEmpty())
                {
                    XMLUtils.startElement(handler, "content-types");
                    
                    for (String contentTypeId : contentTypeIds)
                    {
                        AttributesImpl contentTypeAttributes = new AttributesImpl();
                        contentTypeAttributes.addCDATAAttribute("id", contentTypeId);
                        XMLUtils.createElement(handler, "content-type", contentTypeAttributes);
                    }
                    
                    XMLUtils.endElement(handler, "content-types");
                }
                
                String searchModelId = (String) variable.get("searchModelId");
                if (StringUtils.isNotEmpty(searchModelId))
                {
                    XMLUtils.createElement(handler, "search-model-id", searchModelId);
                }
                
                String solrRequest = (String) variable.get("solrRequest");
                if (StringUtils.isNotEmpty(solrRequest))
                {
                    XMLUtils.createElement(handler, "solr-request", solrRequest);
                }
                
                XMLUtils.endElement(handler, "variable");
            }
            XMLUtils.endElement(handler, ExtractionConstants.CLAUSES_VARIABLES_TAG);
        }
    }

    @SuppressWarnings("unchecked")
    private void _saxOptionalColumns(Map<String, Object> child, TransformerHandler handler) throws SAXException
    {
        Map<String, Object> data = (Map<String, Object>) child.get("data");
        
        List<String> names = (List<String>) data.get("names");
        if (null != names && !names.isEmpty())
        {
            XMLUtils.startElement(handler, ExtractionConstants.OPTIONAL_COLUMNS_TAG);
            for (String name : names)
            {
                XMLUtils.startElement(handler, "name");
                XMLUtils.data(handler, name.trim());
                XMLUtils.endElement(handler, "name");
            }
            XMLUtils.endElement(handler, ExtractionConstants.OPTIONAL_COLUMNS_TAG);
        }
    }

    @SuppressWarnings("unchecked")
    private void _saxExtractionComponent(Map<String, Object> component, TransformerHandler handler) throws SAXException
    {
        Map<String, Object> data = (Map<String, Object>) component.get("data");
        
        String tag = (String) component.get("tag");
        AttributesImpl attributes = _getComponentAttibutes(data);
        XMLUtils.startElement(handler, tag, attributes);
        
        // sax component's elements
        _saxExtractionComponentClauses(data, handler);
        _saxExtractionComponentGroupingFields(data, handler);
        _saxExtractionComponentColumns(data, handler);
        _saxExtractionComponentSorts(data, handler);
        
        // process children
        if (component.get("children") != null)
        {
            List<Map<String, Object>> children = (List<Map<String, Object>>) component.get("children");
            for (Map<String, Object> child : children)
            {
                _saxExtractionComponent(child, handler);
            }
        }
        
        XMLUtils.endElement(handler, tag);
    }

    @SuppressWarnings("unchecked")
    private AttributesImpl _getComponentAttibutes(Map<String, Object> data)
    {
        AttributesImpl attributes = new AttributesImpl();
        
        Object componentTagName = data.get("componentTagName");
        if (null != componentTagName && !StringUtils.isEmpty((String) componentTagName))
        {
            attributes.addCDATAAttribute("tagName", (String) componentTagName);
        }
        
        List<String> contentTypes = (List<String>) data.get("contentTypes");
        if (null != contentTypes && !contentTypes.isEmpty())
        {
            attributes.addCDATAAttribute("contentTypes", String.join(ExtractionConstants.STRING_COLLECTIONS_INPUT_DELIMITER, contentTypes));
        }
        
        Object queryReferenceId = data.get("queryReferenceId");
        if (null != queryReferenceId && !StringUtils.isEmpty((String) queryReferenceId))
        {
            attributes.addCDATAAttribute("ref", (String) queryReferenceId);
        }
        
        Object microThesaurusId = data.get("microThesaurusId");
        if (null != microThesaurusId && !StringUtils.isEmpty((String) microThesaurusId))
        {
            attributes.addCDATAAttribute("microThesaurus", (String) microThesaurusId);
        }
        
        Object maxLevel = data.get("maxLevel");
        if (null != maxLevel && !StringUtils.isEmpty(String.valueOf(maxLevel)))
        {
            attributes.addCDATAAttribute("max-level", String.valueOf(maxLevel));
        }
        
        return attributes;
    }
    
    @SuppressWarnings("unchecked")
    private void _saxExtractionComponentClauses(Map<String, Object> data, TransformerHandler handler) throws SAXException
    {
        List<String> clauses = (List<String>) data.get("clauses");
        if (null != clauses && !clauses.isEmpty())
        {
            XMLUtils.startElement(handler, "clauses");
            for (String clause : clauses)
            {
                XMLUtils.startElement(handler, "clause");
                XMLUtils.data(handler, clause);
                XMLUtils.endElement(handler, "clause");
            }
            XMLUtils.endElement(handler, "clauses");
        }
    }

    private void _saxExtractionComponentGroupingFields(Map<String, Object> data, TransformerHandler handler) throws SAXException
    {
        Object groupingFields = data.get("groupingFields");
        if (null == groupingFields || StringUtils.isEmpty((String) groupingFields))
        {
            return;
        }
        XMLUtils.startElement(handler, "grouping-fields");
        XMLUtils.data(handler, (String) groupingFields);
        XMLUtils.endElement(handler, "grouping-fields");
    }

    private void _saxExtractionComponentColumns(Map<String, Object> data, TransformerHandler handler) throws SAXException
    {
        Object columnsObj = data.get("columns");
        if (null == columnsObj || StringUtils.isEmpty((String) columnsObj))
        {
            return;
        }
        
        Map<String, String> columns = _splitDataAndExtradataFromString((String) columnsObj);
        
        if (!columns.isEmpty())
        {
            AttributesImpl columnsAttributes = new AttributesImpl();
            Object overrideColumns = data.get("overrideColumns");
            if (null != overrideColumns && (Boolean) overrideColumns)
            {
                columnsAttributes.addCDATAAttribute("override", "true");
            }
            XMLUtils.startElement(handler, "columns", columnsAttributes);
            
            for (Map.Entry<String, String> column : columns.entrySet())
            {
                AttributesImpl columnAttributes = new AttributesImpl();
                if (column.getValue() != null)
                {
                    columnAttributes.addCDATAAttribute("optional", column.getValue());
                }
                XMLUtils.startElement(handler, "column", columnAttributes);
                XMLUtils.data(handler, column.getKey());
                XMLUtils.endElement(handler, "column");
            }
            
            XMLUtils.endElement(handler, "columns");
        }
    }

    private void _saxExtractionComponentSorts(Map<String, Object> data, TransformerHandler handler) throws SAXException
    {
        Object sortsObj = data.get("sorts");
        if (null == sortsObj || StringUtils.isEmpty((String) sortsObj))
        {
            return;
        }
        
        Map<String, String> sorts = _splitDataAndExtradataFromString((String) sortsObj);
        
        if (!sorts.isEmpty())
        {
            AttributesImpl sortsAttributes = new AttributesImpl();
            Object overrideSorts = data.get("overrideSorts");
            if (null != overrideSorts && (Boolean) overrideSorts)
            {
                sortsAttributes.addCDATAAttribute("override", "true");
            }
            XMLUtils.startElement(handler, "sorts", sortsAttributes);
            
            for (Map.Entry<String, String> sort : sorts.entrySet())
            {
                AttributesImpl sortAttributes = new AttributesImpl();
                if (sort.getValue() != null)
                {
                    sortAttributes.addCDATAAttribute("order", sort.getValue());
                }
                XMLUtils.startElement(handler, "sort", sortAttributes);
                XMLUtils.data(handler, sort.getKey());
                XMLUtils.endElement(handler, "sort");
            }
            
            XMLUtils.endElement(handler, "sorts");
        }
    }
    
    private Map<String, String> _splitDataAndExtradataFromString(String str)
    {
        Map<String, String> result = new LinkedHashMap<>();
        
        for (String data : str.split(ExtractionConstants.STRING_COLLECTIONS_INPUT_DELIMITER))
        {
            Pattern pattern = Pattern.compile(EXTRACT_EXTRA_DATA_REGEX);
            Matcher matcher = pattern.matcher(data);
            
            String extra = null;
            if (matcher.find())
            {
                extra = matcher.group(1);
            }
            
            String finalData = data;
            if (null != extra)
            {
                extra = extra.trim();
                int indexOfExtra = data.indexOf("(");
                finalData = data.substring(0, indexOfExtra);
            }
            
            result.put(finalData.trim(), extra);
        }
        
        return result;
    }
    
}
