/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.reading.ServiceableReader;
import org.xml.sax.SAXException;

import org.ametys.plugins.extraction.execution.pipeline.PipelineDescriptor;
import org.ametys.plugins.extraction.execution.pipeline.PipelineManager;

/**
 * Reader for copying the extraction result into the pipeline {@link OutputStream}
 */
public class ExtractionReader extends ServiceableReader
{
    private static final String __PIPELINE_PARAMETER = "pipeline";
    private static final String __LANG_PARAMETER = "lang";
    private static final String __DEFINITION_FILE_PARAMETER = "file";
    
    private PipelineManager _pipelineManager;
    private ExtractionExecutor _extractionExecutor;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _pipelineManager = (PipelineManager) manager.lookup(PipelineManager.ROLE);
        _extractionExecutor = (ExtractionExecutor) smanager.lookup(ExtractionExecutor.ROLE);
    }

    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String definitionFileName = request.getParameter(__DEFINITION_FILE_PARAMETER);
        if (definitionFileName == null)
        {
            throw new IllegalArgumentException("The '" + __DEFINITION_FILE_PARAMETER + "' parameter is mandatory, it must be present to select an extraction to execute.");
        }
        String lang = request.getParameter(__LANG_PARAMETER);
        Map<String, Object> params = _getParams(request);
        PipelineDescriptor pipeline = _getPipeline(request);
        
        try
        {
            _extractionExecutor.execute(definitionFileName, out, lang, params, pipeline);
        }
        catch (Exception e)
        {
            throw new ProcessingException(e);
        }
    }
    
    private PipelineDescriptor _getPipeline(Request request) throws IOException
    {
        String pipelineId = request.getParameter(__PIPELINE_PARAMETER);
        if (pipelineId == null)
        {
            // Use default pipeline
            pipelineId = _pipelineManager.getDefaultPipeline();
        }
        else if (!_pipelineManager.has(pipelineId))
        {
            // Pipeline does not exists
            throw new IllegalArgumentException("The pipeline '" + pipelineId + "' does not exists.");
        }
        PipelineDescriptor pipeline = _pipelineManager.get(pipelineId);
        return pipeline;
    }
    
    private static Map<String, Object> _getParams(Request request)
    {
        return Collections.list((Enumeration<String>) request.getParameterNames())
                .stream()
                .collect(Collectors.toMap(
                    Function.identity(), 
                    paramName -> request.getParameter(paramName)));
    }
}

