/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution.pipeline.impl;

import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.excalibur.source.SourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.core.util.SizeUtils.ExcludeFromSizeCalculation;
import org.ametys.plugins.extraction.execution.pipeline.ExtractionMatcher;
import org.ametys.plugins.extraction.execution.pipeline.Pipeline;
import org.ametys.plugins.extraction.execution.pipeline.PipelineDescriptor;
import org.ametys.plugins.extraction.execution.pipeline.PipelineSerializerModel;
import org.ametys.plugins.extraction.execution.pipeline.PipelineSerializerModelExtensionPoint;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Implementation of {@link PipelineDescriptor} which is {@link Configurable}.
 */
public class ConfigurablePipelineDescriptor implements PipelineDescriptor, Configurable
{
    @ExcludeFromSizeCalculation
    private static final Logger __LOGGER = LoggerFactory.getLogger(ConfigurablePipelineDescriptor.class);
    private static final String __DEFAULT_SERIALIZER = "xml";

    private String _id;
    private I18nizableText _label;
    private List<String> _extractions;
    private List<String> _xslts;
    private Map<String, String> _outputParams;
    @ExcludeFromSizeCalculation
    private PipelineSerializerModel _serializer;
    
    @ExcludeFromSizeCalculation
    private SourceResolver _resolver;
    @ExcludeFromSizeCalculation
    private PipelineSerializerModelExtensionPoint _pipelineSerializers;
    
    /**
     * Default constructor
     * @param id The id of the pipeline descriptor
     * @param resolver The source resolver
     * @param pipelineSerializers The extension point for pipeline serializers
     */
    public ConfigurablePipelineDescriptor(String id, SourceResolver resolver, PipelineSerializerModelExtensionPoint pipelineSerializers)
    {
        _id = id;
        _resolver = resolver;
        _pipelineSerializers = pipelineSerializers;
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _label = I18nizableText.parseI18nizableText(configuration.getChild("label"), "application");
        
        _extractions = new ArrayList<>();
        for (Configuration extractionConf : configuration.getChild("extractions").getChildren("extraction"))
        {
            String filePath = extractionConf.getValue();
            _extractions.add(filePath);
        }
        
        _xslts = new ArrayList<>();
        for (Configuration xsltConf : configuration.getChild("stylesheets").getChildren("xslt"))
        {
            String name = xsltConf.getAttribute("name");
            _xslts.add(name);
        }
        
        setOutputParameters(configuration.getChild("out", false));
        String serializerType = _outputParams.getOrDefault("type", __DEFAULT_SERIALIZER);
        _serializer = _pipelineSerializers.getExtension(serializerType);
        if (_serializer == null)
        {
            __LOGGER.warn("The serializer type '{}' in <out> tag of the pipeline descriptor '{}' is invalid.", serializerType, _id);
            _serializer = _pipelineSerializers.getExtension(__DEFAULT_SERIALIZER);
        }
    }

    @Override
    public I18nizableText getLabel()
    {
        return _label;
    }
    
    @Override
    public ExtractionMatcher getExtractionMatcher()
    {
        return !_extractions.isEmpty() ? new ListExtractionMatcher(_extractions) : new AllExtractionMatcher();
    }
    
    @Override
    public List<String> getStylesheets()
    {
        return _xslts;
    }
    
    @Override
    public PipelineSerializerModel getSerializerModel()
    {
        return _serializer;
    }
    
    @Override
    public String getResultSubfolder()
    {
        return _outputParams.getOrDefault("path", "" /* root folder */);
    }
    
    @Override
    public String getDefaultExtension()
    {
        return _outputParams.getOrDefault("extension", _serializer.getDefaultFileExtension());
    }
    
    @Override
    public Pipeline newPipeline(OutputStream out)
    {
        return new PipelineImpl(this, out, _resolver);
    }

    @Override
    public Map<String, String> getOutputParameters()
    {
        return _outputParams;
    }
    
    /**
     * Set the output parameters to get the extension, path, encoding, etc.
     * @param outputConf The output configuration
     * @throws ConfigurationException throws an exception if an error occurs
     */
    protected void setOutputParameters(Configuration outputConf) throws ConfigurationException
    {
        _outputParams = new HashMap<>();
        if (outputConf != null)
        {
            for (String name : outputConf.getAttributeNames())
            {
                _outputParams.put(name, outputConf.getAttribute(name));
            }
        }
    }
}
