/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.rights;

import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceException;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.impl.FileSource;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.right.AbstractRightBasedAccessController;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.execution.Extraction;
import org.ametys.plugins.extraction.execution.ExtractionDAO;
import org.ametys.runtime.i18n.I18nizableText;


/**
 * {@link AccessController} to allow read access and handle for author of a extraction file
 *
 */
public class ExtractionAuthorAccessController extends AbstractRightBasedAccessController implements Serviceable, Initializable
{
    private static final List<String> __AUTHOR_RIGHTS = List.of(ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID, "Workflow_Rights_Edition_Online");
    
    private SourceResolver _srcResolver;
    private String _rootPath;
    private ExtractionDAO _extractionDAO;
    private ContentTypesHelper _contentTypesHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _srcResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _extractionDAO = (ExtractionDAO) manager.lookup(ExtractionDAO.ROLE);
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
    }
    
    public void initialize() throws Exception
    {
        FileSource rootDir = (FileSource) _srcResolver.resolveURI(ExtractionConstants.DEFINITIONS_DIR);
        // use the URI. The path is not available if the definitions folder is not created at start time
        _rootPath = rootDir.getURI();
    }
    
    public boolean supports(Object object)
    {
        return object instanceof Extraction
                // a fileSource that don't exist is not a collection.
                // not checking that it exists leads to the root being supported if not created
                || object instanceof FileSource fileSource && fileSource.exists() && !fileSource.isCollection() && fileSource.getURI().startsWith(_rootPath)
                || object instanceof Content content && _contentTypesHelper.isInstanceOf(content, ExtractionConstants.DESCRIPTION_CONTENT_TYPE_ID);
    }
    
    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        if (user.equals(_getAuthor(object)))
        {
            return __AUTHOR_RIGHTS.contains(rightId) ? AccessResult.USER_ALLOWED : AccessResult.UNKNOWN;
        }
        
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return user.equals(_getAuthor(object)) ? AccessResult.USER_ALLOWED : AccessResult.UNKNOWN;
    }

    /**
     * If creator, access to a list of rights
     */
    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        Map<String, AccessResult> permissionByRight = new HashMap<>();
        
        if (user.equals(_getAuthor(object)))
        {
            for (String rightId : __AUTHOR_RIGHTS)
            {
                permissionByRight.put(rightId, AccessResult.USER_ALLOWED);
            }
        }
        
        return permissionByRight;
    }

    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    /**
     * If right requested is in the list, the creator is added the list of USER_ALLOWED
     */
    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        Map<UserIdentity, AccessResult> permissionByUser = new HashMap<>();
        
        if (__AUTHOR_RIGHTS.contains(rightId))
        {
            UserIdentity extractionAuthor = _getAuthor(object);
            permissionByUser.put(extractionAuthor, AccessResult.USER_ALLOWED);
        }
        return permissionByUser;
    }

    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        return MapUtils.EMPTY_MAP;
    }

    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return MapUtils.EMPTY_MAP;
    }

    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        return MapUtils.EMPTY_MAP;
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return false;
    }

    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }

    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    @Override
    protected AccessExplanation _getAccessExplanation(AccessResult result, Object object, UserIdentity user, Set<GroupIdentity> groups, String rightId)
    {
        switch (result)
        {
            case USER_ALLOWED:
            case UNKNOWN:
                if (object instanceof Content)
                {
                    return new AccessExplanation(
                            getId(),
                            result,
                            new I18nizableText("plugin.extraction", "PLUGINS_EXTRACTION_CONTENT_AUTHOR_ACCESS_CONTROLLER_" + result.name() + "_EXPLANATION",
                                    Map.of("title", getObjectLabel(object)))
                            );
                }
                else
                {
                    return new AccessExplanation(
                            getId(),
                            result,
                            new I18nizableText("plugin.extraction", "PLUGINS_EXTRACTION_AUTHOR_ACCESS_CONTROLLER_" + result.name() + "_EXPLANATION",
                                    Map.of("title", getObjectLabel(object)))
                            );
                }
            default:
                return AccessController.getDefaultAccessExplanation(getId(), result);
        }
    }
    
    private UserIdentity _getAuthor(Object object)
    {
        if (object instanceof Extraction extraction)
        {
            return extraction.getAuthor();
        }
        else if (object instanceof FileSource fileSource)
        {
            return _extractionDAO.getAuthor(fileSource);
        }
        else if (object instanceof Content content)
        {
            return content.getCreator();
        }
        return null;
    }
    
    private String _getExtractionName(Object object)
    {
        if (object instanceof FileSource fileSource)
        {
            String target = StringUtils.substringAfter(_extractionDAO.getExtractionRightPath(fileSource), ExtractionAccessController.ROOT_CONTEXT);
            target = StringUtils.replace(target.substring(1), "/", " > ");
            return target;
        }
        // We can't include the hierarchy from the extraction or content
        // But it shouldn't be visible as there is nothing display the explanation for it
        else if (object instanceof Extraction extraction)
        {
            return extraction.getFileName();
        }
        else if (object instanceof Content content)
        {
            return content.getTitle();
        }
        return null;
    }

    public I18nizableText getObjectLabel(Object object)
    {
        String extractionName = _getExtractionName(object);
        if (extractionName != null)
        {
            return new I18nizableText(extractionName);
        }
        throw new RightsException("Unsupported context: " + object.toString());
    }

    public I18nizableText getObjectCategory(Object object)
    {
        return ExtractionAccessController.EXTRACTION_CONTEXT_CATEGORY;
    }

    @Override
    protected Iterable< ? extends Object> getHandledObjects(UserIdentity identity, Set<GroupIdentity> groups, Set<Object> workspacesContexts)
    {
        if (workspacesContexts.contains("/cms"))
        {
            try
            {
                FileSource rootDir = (FileSource) _srcResolver.resolveURI(ExtractionConstants.DEFINITIONS_DIR);
                if (rootDir.getFile().exists())
                {
                    Stream<FileSource> definitions = _getDefinitions(rootDir);
                    return definitions.toList();
                }
            }
            catch (IOException e)
            {
                getLogger().warn("Failed to compute the list of extractions");
            }
        }

        return List.of();
    }

    private Stream<FileSource> _getDefinitions(FileSource source)
    {
        if (source.isCollection())
        {
            try
            {
                return source.getChildren().stream()
                    .filter(FileSource.class::isInstance)
                    .flatMap(src -> _getDefinitions((FileSource) src));
            }
            catch (SourceException e)
            {
                getLogger().warn("Failed to compute the list of extractions");
                return Stream.of();
            }
        }
        else
        {
            return Stream.of(source);
        }
    }
    
    public ExplanationObject getExplanationObject(Object object)
    {
        if (object instanceof FileSource source)
        {
            return new ExplanationObject(
                    // we convert the source to the right path to be able to merge with ExtractionAccessController
                    _extractionDAO.getExtractionRightPath(source),
                    getObjectLabel(object),
                    getObjectCategory(object)
                    );
        }
        return super.getExplanationObject(object);
    }
    
    @Override
    protected Collection<String> getHandledRights()
    {
        return __AUTHOR_RIGHTS;
    }
}
