/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.ametys.core.schedule.Runnable;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * A {@link Runnable} which schedules a {@link ExecuteExtractionSchedulable} task for the given definition file.
 */
public class ExecuteExtractionRunnable implements Runnable
{
    private String _definitionFilePath;
    private String _variables;
    private String _recipient;
    private String _pipeline;
    
    /**
     * Constructor
     * @param definitionFilePath The extraction definition file path
     * @param variables clauses variables and optional columns
     * @param recipient An email will be sent at this address when the extraction is complete
     * @param pipeline The pipeline id
     */
    public ExecuteExtractionRunnable(String definitionFilePath, String variables, String recipient, String pipeline)
    {
        _definitionFilePath = definitionFilePath;
        _variables = variables;
        _recipient = recipient;
        _pipeline = pipeline;
    }

    public String getId()
    {
        return getClass().getName() + "." + _definitionFilePath;
    }

    public I18nizableText getLabel()
    {
        return new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_SCHEDULABLE_LABEL", Collections.singletonList(_definitionFilePath));
    }

    public I18nizableText getDescription()
    {
        return new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_SCHEDULABLE_DESCRIPTION", Collections.singletonList(_definitionFilePath));
    }

    public FireProcess getFireProcess()
    {
        return FireProcess.NOW;
    }

    public String getCronExpression()
    {
        return null;
    }

    public String getSchedulableId()
    {
        return "org.ametys.plugins.extraction.execution.ExecuteExtractionSchedulable";
    }

    public boolean isRemovable()
    {
        return false;
    }

    public boolean isModifiable()
    {
        return false;
    }

    public boolean isDeactivatable()
    {
        return false;
    }

    public MisfirePolicy getMisfirePolicy()
    {
        return null;
    }

    public boolean isVolatile()
    {
        return false;
    }

    public UserIdentity getUserIdentity()
    {
        return UserPopulationDAO.SYSTEM_USER_IDENTITY;
    }

    public Map<String, Object> getParameterValues()
    {
        Map<String, Object> values = new HashMap<>();
        values.put(ExecuteExtractionSchedulable.DEFINITION_FILE_PATH_KEY, _definitionFilePath);
        values.put(ExecuteExtractionSchedulable.VARIABLES_KEY, _variables);
        values.put(ExecuteExtractionSchedulable.RECIPIENT_KEY, _recipient);
        values.put(ExecuteExtractionSchedulable.PIPELINE_KEY, _pipeline);
        return values;
    }

}
