/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;

import org.ametys.cms.schedule.AbstractSendingMailSchedulable;
import org.ametys.core.schedule.Schedulable;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.directory.NotUniqueUserException;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.execution.pipeline.PipelineDescriptor;
import org.ametys.plugins.extraction.execution.pipeline.PipelineManager;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * A {@link Schedulable} job which execute an extraction
 */
public class ExecuteExtractionSchedulable extends AbstractSendingMailSchedulable
{
    /** The key for the extraction definition file */
    public static final String DEFINITION_FILE_PATH_KEY = "definitionFilePath";
    /** The key for the variables values */
    public static final String VARIABLES_KEY = "variables";
    /** The key for the recipient */
    public static final String RECIPIENT_KEY = "recipient";
    /** The key for the pipeline */
    public static final String PIPELINE_KEY = "pipeline";
    
    private static final String __JOBDATAMAP_DEFINITION_FILE_PATH_KEY = Scheduler.PARAM_VALUES_PREFIX + DEFINITION_FILE_PATH_KEY;
    private static final String __JOBDATAMAP_VARIABLES_KEY = Scheduler.PARAM_VALUES_PREFIX + VARIABLES_KEY;
    private static final String __JOBDATAMAP_RECIPIENT_KEY = Scheduler.PARAM_VALUES_PREFIX + RECIPIENT_KEY;
    private static final String __JOBDATAMAP_PIPELINE_KEY = Scheduler.PARAM_VALUES_PREFIX + PIPELINE_KEY;

    private static final String __RESULT_FILE_PATHS = "resultFilePaths";
    
    private static final DateTimeFormatter RESULT_FILE_NAME_DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyyMMdd'T'HH.mm.ss");
    
    private JSONUtils _jsonUtils;
    private PipelineManager _pipelineManager;
    private ExtractionExecutor _extractionExecutor;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _pipelineManager = (PipelineManager) manager.lookup(PipelineManager.ROLE);
        _extractionExecutor = (ExtractionExecutor) manager.lookup(ExtractionExecutor.ROLE);
    }
    
    @Override
    public void _doExecute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();

        String definitionFilePath = _getDefinitionFilePath(context);
        PipelineDescriptor pipeline = _pipelineManager.get(jobDataMap.getString(__JOBDATAMAP_PIPELINE_KEY));
        String defaultResultFileName = _getDefaultResultFileName(definitionFilePath, pipeline);

        Map<String, Object> parameters = _getExtractionParameters(jobDataMap);
        String lang = (String) parameters.get("lang");
        
        Set<Path> resultFilePaths = _extractionExecutor.execute(definitionFilePath, defaultResultFileName, lang, parameters, pipeline);
        context.put(__RESULT_FILE_PATHS, resultFilePaths);
    }

    private String _getDefaultResultFileName(String definitionFilePath, PipelineDescriptor pipeline)
    {
        String[] definitionFilePathSegments = definitionFilePath.split(Pattern.quote(File.separator));
        String definitionFileName = definitionFilePathSegments[definitionFilePathSegments.length - 1];
        
        int lastIndexOfDot = definitionFileName.lastIndexOf('.');
        if (-1 != lastIndexOfDot)
        {
            definitionFileName = definitionFileName.substring(0, lastIndexOfDot);
        }
        
        String extractionDate = ZonedDateTime.now().format(RESULT_FILE_NAME_DATE_TIME_FORMATTER);
        
        StringBuilder resultFileName = new StringBuilder();
        resultFileName.append(definitionFileName).append("-").append(extractionDate);
        resultFileName.append(".").append(pipeline.getDefaultExtension());
        return resultFileName.toString();
    }
    
    private Map<String, Object> _getExtractionParameters(JobDataMap jobDataMap)
    {
        // variables parameters
        String variablesAsString = jobDataMap.getString(__JOBDATAMAP_VARIABLES_KEY);
        Map<String, Object> variablesAsMap = _jsonUtils.convertJsonToMap(variablesAsString);
        return variablesAsMap;
    }
    
    @Override
    protected String _getRecipientLanguage(JobExecutionContext context)
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        String recipient = jobDataMap.getString(__JOBDATAMAP_RECIPIENT_KEY);
        
        if (jobDataMap.containsKey("userIdentity"))
        {
            String userIdentityString = jobDataMap.getString("userIdentity");
            UserIdentity userIdentity = UserIdentity.stringToUserIdentity(userIdentityString);

            // Try to retrieve the user that launched the job
            User user = _userManager.getUser(userIdentity);
            // If the launching user is also the recipient, get its language
            if (user != null && recipient != null && recipient.equals(user.getEmail()))
            {
                return user.getLanguage();
            }
            else
            {
                // Try to retrieve the recipient by its mail in the population of the launching user
                try
                {
                    user = _userManager.getUserByEmail(userIdentity.getPopulationId(), recipient);
                    if (user != null)
                    {
                        return user.getLanguage();
                    }
                }
                catch (NotUniqueUserException e)
                {
                    // Do Nothing
                }
            }
        }
        
        return null;
    }
    
    @Override
    protected Optional<String> _getRecipient(JobExecutionContext context)
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        return Optional.ofNullable(jobDataMap.getString(__JOBDATAMAP_RECIPIENT_KEY))
                .filter(StringUtils::isNotEmpty);
    }
    
    @Override
    protected boolean _isMailBodyInHTML(JobExecutionContext context)
    {
        return true;
    }
    
    @Override
    protected I18nizableText _getSuccessMailSubject(JobExecutionContext context)
    {
        String extractionName = _getDefinitionFilePath(context);
        return new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_MAIL_SUCCESS_SUBJECT", Map.of("extractionName", new I18nizableText(extractionName)));
    }
    
    @Override
    protected String _getSuccessMailBody(JobExecutionContext context, String language)
    {
        try
        {
            String extractionName = _getDefinitionFilePath(context);
            
            MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
                .withTitle(new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_MAIL_SUCCESS_BODY_TITLE", Map.of("extractionName", new I18nizableText(extractionName))))
                .withLanguage(language)
                .addMessage(new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_MAIL_SUCCESS_BODY", Map.of("extractionName", new I18nizableText(extractionName))));
            
            I18nizableText resultsToolLabel = new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_RESULTS_LIST_TOOL_LABEL");

            @SuppressWarnings("unchecked")
            Set<Path> resultFilePaths = (Set<Path>) context.get(__RESULT_FILE_PATHS);
            if (resultFilePaths.size() == 1)
            {
                String downloadLink = _getResultFileDownloadLink(resultFilePaths.iterator().next());
                
                bodyBuilder
                    .addMessage(new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_MAIL_SUCCESS_BODY_DOWNLOAD_RESULT", Map.of("link", new I18nizableText(downloadLink))))
                    .addMessage(new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_MAIL_SUCCESS_BODY_SEE_RESULT", Map.of("toolLabel", resultsToolLabel)));
            }
            else
            {
                StringBuilder downloadLinks = new StringBuilder("<ul>");
                for (Path resultFilePath : resultFilePaths)
                {
                    downloadLinks.append("<li>")
                                 .append(_getResultFileDownloadLink(resultFilePath))
                                 .append("</li>");
                }
                downloadLinks.append("</ul>");
                
                bodyBuilder
                    .addMessage(new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_MAIL_SUCCESS_BODY_DOWNLOAD_RESULTS", Map.of("links", new I18nizableText(downloadLinks.toString()))))
                    .addMessage(new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_MAIL_SUCCESS_BODY_SEE_RESULTS", Map.of("toolLabel", resultsToolLabel)));
            }
            
            return bodyBuilder.build();
        }
        catch (IOException e)
        {
            getLogger().error("Failed to build HTML body for extraction report mail", e);
            return null;
        }
    }
    
    @Override
    protected I18nizableText _getErrorMailSubject(JobExecutionContext context)
    {
        String extractionName = _getDefinitionFilePath(context);
        return new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_MAIL_ERROR_SUBJECT", Map.of("extractionName", new I18nizableText(extractionName)));
    }
    
    @Override
    protected String _getErrorMailBody(JobExecutionContext context, String language, Throwable throwable)
    {
        String extractionName = _getDefinitionFilePath(context);
        String error = ExceptionUtils.getStackTrace(throwable);
        
        try
        {
            return StandardMailBodyHelper.newHTMLBody()
                .withTitle(new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_MAIL_ERROR_BODY_TITLE", Map.of("extractionName", new I18nizableText(extractionName))))
                .withMessage(new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_MAIL_ERROR_BODY", Map.of("extractionName", new I18nizableText(extractionName))))
                .withDetails(new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_MAIL_ERROR_TITLE"), error, true)
                .withLanguage(language)
                .build();
        }
        catch (IOException e)
        {
            getLogger().error("Failed to build HTML body email for extraction report results", e);
            return null;
        }
    }
    
    private String _getDefinitionFilePath(JobExecutionContext context)
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        return jobDataMap.getString(__JOBDATAMAP_DEFINITION_FILE_PATH_KEY);
    }
    
    private String _getResultFileDownloadLink(Path resultFilePath)
    {
        Path rootPath = AmetysHomeHelper.getAmetysHomeData().toPath().resolve(ExtractionConstants.RESULT_EXTRACTION_DIR_NAME);
        
        String rootURI = rootPath.toUri().toString();
        String resultFileURI = resultFilePath.toUri().toString();
        
        String resultFileRelativeURI = resultFileURI.substring(rootURI.length());
        String downloadURL = Config.getInstance().getValue("cms.url") + "/plugins/extraction/result/download/" + resultFileRelativeURI;

        String resultFileRelativePath = rootPath.relativize(resultFilePath).toString();

        return "<a href=\"" + downloadURL + "\">" + resultFileRelativePath + "</a>";
    }
    
}
