/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceException;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.TraversableSource;
import org.apache.excalibur.source.impl.FileSource;

import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.file.FileHelper;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.ProfileAssignmentStorage.AnonymousOrAnyConnectedKeys;
import org.ametys.core.right.ProfileAssignmentStorage.UserOrGroup;
import org.ametys.core.right.ProfileAssignmentStorageExtensionPoint;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.rights.ExtractionAccessController;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Object representing the extraction definition file content
 */
public class ExtractionDAO extends AbstractLogEnabled implements Serviceable, Component, Initializable
{
    /** The Avalon role */
    public static final String ROLE = ExtractionDAO.class.getName();
    
    /** Extraction author cache id */
    private static final String EXTRACTION_AUTHOR_CACHE = ExtractionDAO.class.getName() + "$extractionAuthor";
    
    private CurrentUserProvider _userProvider;
    private RightManager _rightManager;
    private SourceResolver _sourceResolver;
    private ExtractionDefinitionReader _definitionReader;
    private ProfileAssignmentStorageExtensionPoint _profileAssignmentStorageEP;
    private CurrentUserProvider _currentUserProvider;
    private AbstractCacheManager _cacheManager;
    private UserHelper _userHelper;
    private TraversableSource _root;
    private FileHelper _fileHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _userProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _definitionReader = (ExtractionDefinitionReader) manager.lookup(ExtractionDefinitionReader.ROLE);
        _profileAssignmentStorageEP = (ProfileAssignmentStorageExtensionPoint) manager.lookup(ProfileAssignmentStorageExtensionPoint.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
        _userHelper = (UserHelper) manager.lookup(UserHelper.ROLE);
        _fileHelper = (FileHelper) manager.lookup(FileHelper.ROLE);
    }

    public void initialize() throws Exception
    {
        _root = (TraversableSource) _sourceResolver.resolveURI(ExtractionConstants.DEFINITIONS_DIR);

        _cacheManager.createRequestCache(EXTRACTION_AUTHOR_CACHE,
                new I18nizableText("plugin.extraction", "PLUGINS_EXTRACTION_CACHE_DEFINITION_AUTHOR_LABEL"),
                new I18nizableText("plugin.extraction", "PLUGINS_EXTRACTION_CACHE_DEFINITION_AUTHOR_DESCRIPTION"),
                true);
    }
    
    /**
     * Get the root container properties
     * @return The root container properties
     * @throws IOException If an error occurred while reading folder
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getRootProperties() throws IOException
    {
        String rootURI = ExtractionConstants.DEFINITIONS_DIR;
        TraversableSource rootDir = (TraversableSource) _sourceResolver.resolveURI(rootURI);
        Map<String, Object> infos = getExtractionContainerProperties(rootDir);
        return infos;
    }

    /**
     * Get extraction container properties
     * @param folder the source of the extraction container
     * @return The extraction container properties
     */
    public Map<String, Object> getExtractionContainerProperties(TraversableSource folder)
    {
        Map<String, Object> infos = new HashMap<>();
        
        UserIdentity currentUser = _userProvider.getUser();
        
        infos.put("canRead", canRead(currentUser, folder));
        infos.put("canRename", canRename(currentUser, folder));
        infos.put("canWrite", canWrite(currentUser, folder));
        infos.put("canDelete", canDelete(currentUser, folder));
        infos.put("canAssignRights", canAssignRights(currentUser, folder));

        return infos;
    }
    
    /**
     * Get extraction properties
     * @param extraction the extraction
     * @param file the source of the extraction
     * @return The extraction properties
     */
    public Map<String, Object> getExtractionProperties(Extraction extraction, TraversableSource file)
    {
        Map<String, Object> infos = new HashMap<>();
        
        UserIdentity currentUser = _userProvider.getUser();
        infos.put("descriptionId", extraction.getDescriptionId());
        
        UserIdentity author = extraction.getAuthor();
        infos.put("author", _userHelper.user2json(author));
  
        infos.put("canRead", canRead(currentUser, file));
        infos.put("canWrite", canWrite(currentUser, file));
        infos.put("canDelete", canDelete(currentUser, file));
        infos.put("canAssignRights", canAssignRights(currentUser, file));

        return infos;
    }

    /**
     * Check if a folder has a descendant in read access for a given user
     * @param userIdentity the user
     * @param folder the source of the extraction container
     * @return <code>true</code> if the folder has a descendant in read access, <code>false</code> otherwise
     */
    public Boolean hasAnyReadableDescendant(UserIdentity userIdentity, TraversableSource folder)
    {
        try
        {
            if (folder.exists())
            {
                for (TraversableSource child : (Collection<TraversableSource>) folder.getChildren())
                {
                    if (child.isCollection())
                    {
                        if (canRead(userIdentity, child) || hasAnyReadableDescendant(userIdentity, child))
                        {
                            return true;
                        }
                    }
                    else if (child.getName().endsWith(".xml") && canRead(userIdentity, child))
                    {
                        return true;
                    }
                }
            }
            
            return false;
        }
        catch (SourceException e)
        {
            throw new RuntimeException("Cannot list child elements of " + folder.getURI(), e);
        }
    }

    /**
     * Check if a folder have descendant in write access for a given user
     * @param userIdentity the user identity
     * @param folder the source of the extraction container
     * @return true if the user have write right for at least one child of this container
     */
    public Boolean hasAnyWritableDescendant(UserIdentity userIdentity, TraversableSource folder)
    {
        return hasAnyWritableDescendant(userIdentity, folder, false);
    }
    
    /**
     * Check if a folder have descendant in write access for a given user
     * @param userIdentity the user identity
     * @param folder the source of the extraction container
     * @param ignoreExtraction true to ignore extraction file from search (rights will check only on containers)
     * @return true if the user have write right for at least one child of this container
     */
    public Boolean hasAnyWritableDescendant(UserIdentity userIdentity, TraversableSource folder, boolean ignoreExtraction)
    {
        try
        {
            if (folder.exists())
            {
                for (TraversableSource child : (Collection<TraversableSource>) folder.getChildren())
                {
                    if (child.isCollection())
                    {
                        if (canWrite(userIdentity, child) || hasAnyWritableDescendant(userIdentity, child))
                        {
                            return true;
                        }
                    }
                    else if (!ignoreExtraction && child.getName().endsWith(".xml") && canWrite(userIdentity, child))
                    {
                        return true;
                    }
                }
            }
            
            return false;
        }
        catch (SourceException e)
        {
            throw new RuntimeException("Cannot list child elements of " + folder.getURI(), e);
        }
    }
    
    /**
     * Checks if a folder has descendants in the right assignment access for a given user
     * @param userIdentity the user
     * @param folder the source of the extraction container
     * @return <code>true</code> if the folder has descendant, <code>false</code> otherwise
     */
    public boolean hasAnyAssignableDescendant(UserIdentity userIdentity, TraversableSource folder)
    {
        try
        {
            if (folder.exists())
            {
                for (TraversableSource child : (Collection<TraversableSource>) folder.getChildren())
                {
                    if (child.isCollection())
                    {
                        if (canAssignRights(userIdentity, child) || hasAnyAssignableDescendant(userIdentity, child))
                        {
                            return true;
                        }
                    }
                    else if (child.getName().endsWith(".xml"))
                    {
                        if (canAssignRights(userIdentity, child))
                        {
                            return true;
                        }
                    }
                }
            }
            
            return false;
        }
        catch (SourceException e)
        {
            throw new RuntimeException("Cannot list child elements of " + folder.getURI(), e);
        }
    }
    
    /**
     * Check if a user has read rights on an extraction container or file
     * @param userIdentity the user
     * @param source the source of the extraction container or file
     * @return <code>true</code> if the user has read rights on an extraction container, <code>false</code> otherwise
     */
    public boolean canRead(UserIdentity userIdentity, TraversableSource source)
    {
        return _rightManager.hasReadAccess(userIdentity, source) || canWrite(userIdentity, source);
    }
    
    /**
     * Check if a user has write rights on an extraction container or an extraction
     * @param userIdentity the user
     * @param source the source of the extraction file or extration container
     * @return <code>true</code> if the user has write rights on an extraction container, <code>false</code> otherwise
     */
    public boolean canWrite(UserIdentity userIdentity, TraversableSource source)
    {
        return canWrite(userIdentity, source, false);
    }
    
    /**
     * Determines if the user can rename an extraction container
     * @param userIdentity the user
     * @param folder the extraction container
     * @return true if the user can delete the extraction container
     */
    public boolean canRename(UserIdentity userIdentity, TraversableSource folder)
    {
        try
        {
            return !_isRoot(folder) // is not root
                    && canWrite(userIdentity, folder) // has write access
                    && canWrite(userIdentity, (TraversableSource) folder.getParent()); // has write access on parent
        }
        catch (SourceException e)
        {
            throw new RuntimeException("Unable to determine user rights on the extraction container " + folder.getURI(), e);
        }
    }
    
    /**
     * Determines if the user can delete an extraction container or the extraction file
     * @param userIdentity the user
     * @param source the extraction container or the extraction file
     * @return true if the user can delete the extraction container
     */
    public boolean canDelete(UserIdentity userIdentity, TraversableSource source)
    {
        try
        {
            return !_isRoot(source) // is not root
                && canWrite(userIdentity, (TraversableSource) source.getParent()) // has write access on parent
                && canWrite(userIdentity, source, true); // has write access on itselft and each descendant
        }
        catch (SourceException e)
        {
            throw new RuntimeException("Unable to determine user rights on extraction container or file at uri " + source.getURI(), e);
        }
    }
    
    /**
     * Check if a user has write access on an extraction container
     * @param userIdentity the user user identity
     * @param source the extraction container or the extraction file
     * @param recursively true to check write access on all descendants recursively
     * @return true if the user has write access on the extraction container
     */
    public boolean canWrite(UserIdentity userIdentity, TraversableSource source, boolean recursively)
    {
        boolean hasRight = _rightManager.hasRight(userIdentity, ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID, source) == RightResult.RIGHT_ALLOW;
        if (!hasRight)
        {
            return false;
        }
           
        try
        {
            if (recursively && source.isCollection())
            {
                for (TraversableSource child : (Collection<TraversableSource>) source.getChildren())
                {
                    hasRight = hasRight && canWrite(userIdentity, child);
                    
                    if (!hasRight)
                    {
                        return false;
                    }
                }
            }
            
            return hasRight;
        }
        catch (SourceException e)
        {
            throw new RuntimeException("Unable to determine user rights on extraction container " + source.getURI(), e);
        }
    }
    
    /**
     * Check if a user can edit rights on an extraction container or an extraction file
     * @param userIdentity the user
     * @param source the source of the extraction container or file
     * @return true if the user can edit rights on an extraction container or file
     */
    public boolean canAssignRights(UserIdentity userIdentity, TraversableSource source)
    {
        try
        {
            return _rightManager.hasRight(userIdentity, "Runtime_Rights_Rights_Handle", "/cms") == RightResult.RIGHT_ALLOW
                    || !_isRoot(source) // is not root
                    && canWrite(userIdentity, (TraversableSource) source.getParent()) // has write access on parent
                    && canWrite(userIdentity, source, true); // has write access on itselft and each descendant
        }
        catch (SourceException e)
        {
            throw new RuntimeException("Unable to determine the user rights on the extraction container or file at uri " + source.getURI(), e);
        }
    }
    
    /**
     * Determines if the extraction container is the root node
     * @param folder the extraction container
     * @return true if is root
     */
    protected boolean _isRoot(TraversableSource folder)
    {
        return trimLastFileSeparator(_root.getURI()).equals(trimLastFileSeparator(folder.getURI()));
    }

    /**
     * Get the path for rights of an extraction container or file
     * @param source the source of extraction container or file
     * @return the path for rights
     */
    public String getExtractionRightPath(TraversableSource source)
    {
        String rootURI = trimLastFileSeparator(_root.getURI());
        String sourceURI = source.getURI();
        
        if (!sourceURI.startsWith(rootURI))
        {
            // The source is an extraction source
            return null;
        }
        
        // Get only the part after the root folder to get the relative path
        String relPath = StringUtils.substringAfter(trimLastFileSeparator(sourceURI), rootURI);

        // In some case, relPath can start with a /, we need to trim it to test if it is an empty path corresponding to the root
        if (relPath.startsWith("/"))
        {
            relPath = StringUtils.substringAfter(relPath, "/");
        }
        
        return StringUtils.isEmpty(relPath) ? ExtractionAccessController.ROOT_CONTEXT : ExtractionAccessController.ROOT_CONTEXT + "/" + relPath;
    }
    
    /**
     * Get the source corresponding to the right context of an extraction container or file
     * @param rightContext The rights context such as '/extraction-dir/path/to/file
     * @return the resolved source file or null if the given context is not an extraction context
     * @throws IOException if an error occured
     */
    public TraversableSource getExtractionSource(String rightContext) throws IOException
    {
        if (rightContext.startsWith(ExtractionAccessController.ROOT_CONTEXT))
        {
            String relPath = StringUtils.substringAfter(rightContext, ExtractionAccessController.ROOT_CONTEXT);
            String fileUri = ExtractionConstants.DEFINITIONS_DIR + relPath;
            return (TraversableSource) _sourceResolver.resolveURI(fileUri);
        }
        
        return null;
    }
    
    /**
     * Copy rights from one context to another one
     * @param sourceContext the source context
     * @param targetContext the target context
     */
    public void copyRights(String sourceContext, String targetContext)
    {
        // Get the mapping between users and profiles
        Map<UserIdentity, Map<UserOrGroup, Set<String>>> profilesForUsers = _profileAssignmentStorageEP.getProfilesForUsers(sourceContext, null);
        // Copy allowed user assignment profiles to new context
        profilesForUsers.entrySet()
            .forEach(entry -> _copyAllowedUsers(entry.getKey(), entry.getValue().get(UserOrGroup.ALLOWED), targetContext));
        // Copy denied user assignment profiles to new context
        profilesForUsers.entrySet()
            .forEach(entry -> _copyDeniedUsers(entry.getKey(), entry.getValue().get(UserOrGroup.DENIED), targetContext));
        
        // Get the mapping between groups and profiles
        Map<GroupIdentity, Map<UserOrGroup, Set<String>>> profilesForGroups = _profileAssignmentStorageEP.getProfilesForGroups(sourceContext, null);
        // Copy allowed group assignment profiles to new context
        profilesForGroups.entrySet()
            .forEach(entry -> _copyAllowedGroups(entry.getKey(), entry.getValue().get(UserOrGroup.ALLOWED), targetContext));
        // Copy denied group assignment profiles to new context
        profilesForGroups.entrySet()
            .forEach(entry -> _copyDeniedGroups(entry.getKey(), entry.getValue().get(UserOrGroup.DENIED), targetContext));
        
        // Get the mapping between anonymous or any connected user and profiles
        Map<AnonymousOrAnyConnectedKeys, Set<String>> profilesForAnonymousOrAnyConnectedUser = _profileAssignmentStorageEP.getProfilesForAnonymousAndAnyConnectedUser(sourceContext);
        // Copy allowed anonymous user assignment profiles to new context
        profilesForAnonymousOrAnyConnectedUser.get(AnonymousOrAnyConnectedKeys.ANONYMOUS_ALLOWED)
            .forEach(profileId -> _profileAssignmentStorageEP.allowProfileToAnonymous(profileId, targetContext));
        // Copy denied anonymous user assignment profiles to new context
        profilesForAnonymousOrAnyConnectedUser.get(AnonymousOrAnyConnectedKeys.ANONYMOUS_DENIED)
            .forEach(profileId -> _profileAssignmentStorageEP.denyProfileToAnonymous(profileId, targetContext));
        // Copy allowed any connected user assignment profiles to new context
        profilesForAnonymousOrAnyConnectedUser.get(AnonymousOrAnyConnectedKeys.ANYCONNECTEDUSER_ALLOWED)
            .forEach(profileId -> _profileAssignmentStorageEP.allowProfileToAnyConnectedUser(profileId, targetContext));
        // Copy denied any connected user assignment profiles to new context
        profilesForAnonymousOrAnyConnectedUser.get(AnonymousOrAnyConnectedKeys.ANYCONNECTEDUSER_DENIED)
            .forEach(profileId -> _profileAssignmentStorageEP.denyProfileToAnyConnectedUser(profileId, targetContext));
    }
    
    private void _copyAllowedUsers(UserIdentity userIdentity, Set<String> profiles, String context)
    {
        profiles.forEach(profile -> _profileAssignmentStorageEP.allowProfileToUser(userIdentity, profile, context));
    }
    
    private void _copyDeniedUsers(UserIdentity userIdentity, Set<String> profiles, String context)
    {
        profiles.forEach(profile -> _profileAssignmentStorageEP.denyProfileToUser(userIdentity, profile, context));
    }
    
    private void _copyAllowedGroups(GroupIdentity groupIdentity, Set<String> profiles, String context)
    {
        profiles.forEach(profile -> _profileAssignmentStorageEP.allowProfileToGroup(groupIdentity, profile, context));
    }
    
    private void _copyDeniedGroups(GroupIdentity groupIdentity, Set<String> profiles, String context)
    {
        profiles.forEach(profile -> _profileAssignmentStorageEP.denyProfileToGroup(groupIdentity, profile, context));
    }
    
    /**
     * Delete rights from a context
     * @param context the context
     */
    public void deleteRights(String context)
    {
        // Get the mapping between users and profiles
        Map<UserIdentity, Map<UserOrGroup, Set<String>>> profilesForUsers = _profileAssignmentStorageEP.getProfilesForUsers(context, null);
        // Copy allowed user assignment profiles to new context
        profilesForUsers.entrySet()
            .forEach(entry -> _removeAllowedUsers(entry.getKey(), entry.getValue().get(UserOrGroup.ALLOWED), context));
        // Copy denied user assignment profiles to new context
        profilesForUsers.entrySet()
            .forEach(entry -> _removeDeniedUsers(entry.getKey(), entry.getValue().get(UserOrGroup.DENIED), context));
        
        // Get the mapping between groups and profiles
        Map<GroupIdentity, Map<UserOrGroup, Set<String>>> profilesForGroups = _profileAssignmentStorageEP.getProfilesForGroups(context, null);
        // Copy allowed group assignment profiles to new context
        profilesForGroups.entrySet()
            .forEach(entry -> _removeAllowedGroups(entry.getKey(), entry.getValue().get(UserOrGroup.ALLOWED), context));
        // Copy denied group assignment profiles to new context
        profilesForGroups.entrySet()
            .forEach(entry -> _removeDeniedGroups(entry.getKey(), entry.getValue().get(UserOrGroup.DENIED), context));
        
        // Get the mapping between anonymous or any connected user and profiles
        Map<AnonymousOrAnyConnectedKeys, Set<String>> profilesForAnonymousOrAnyConnectedUser = _profileAssignmentStorageEP.getProfilesForAnonymousAndAnyConnectedUser(context);
        // Copy allowed anonymous user assignment profiles to new context
        profilesForAnonymousOrAnyConnectedUser.get(AnonymousOrAnyConnectedKeys.ANONYMOUS_ALLOWED)
            .forEach(profileId -> _profileAssignmentStorageEP.removeAllowedProfileFromAnonymous(profileId, context));
        // Copy denied anonymous user assignment profiles to new context
        profilesForAnonymousOrAnyConnectedUser.get(AnonymousOrAnyConnectedKeys.ANONYMOUS_DENIED)
            .forEach(profileId -> _profileAssignmentStorageEP.removeDeniedProfileFromAnonymous(profileId, context));
        // Copy allowed any connected user assignment profiles to new context
        profilesForAnonymousOrAnyConnectedUser.get(AnonymousOrAnyConnectedKeys.ANYCONNECTEDUSER_ALLOWED)
            .forEach(profileId -> _profileAssignmentStorageEP.removeAllowedProfileFromAnyConnectedUser(profileId, context));
        // Copy denied any connected user assignment profiles to new context
        profilesForAnonymousOrAnyConnectedUser.get(AnonymousOrAnyConnectedKeys.ANYCONNECTEDUSER_DENIED)
            .forEach(profileId -> _profileAssignmentStorageEP.removeDeniedProfileFromAnyConnectedUser(profileId, context));
    }
    
    private void _removeAllowedUsers(UserIdentity userIdentity, Set<String> profiles, String context)
    {
        profiles.forEach(profile -> _profileAssignmentStorageEP.removeAllowedProfileFromUser(userIdentity, profile, context));
    }
    
    private void _removeDeniedUsers(UserIdentity userIdentity, Set<String> profiles, String context)
    {
        profiles.forEach(profile -> _profileAssignmentStorageEP.removeDeniedProfileFromUser(userIdentity, profile, context));
    }
    
    private void _removeAllowedGroups(GroupIdentity groupIdentity, Set<String> profiles, String context)
    {
        profiles.forEach(profile -> _profileAssignmentStorageEP.removeAllowedProfileFromGroup(groupIdentity, profile, context));
    }
    
    private void _removeDeniedGroups(GroupIdentity groupIdentity, Set<String> profiles, String context)
    {
        profiles.forEach(profile -> _profileAssignmentStorageEP.removeDeniedProfileFromGroup(groupIdentity, profile, context));
    }

    /**
     * Move an extraction file or folder inside a given directory
     * 
     * @param srcRelPath The relative URI of file/folder to move
     * @param targetRelPath The target relative URI of file/folder to move
     * @return a result map with the name and uri of moved file in case of
     *         success.
     * @throws IOException If an error occurred manipulating the source
     */
    @Callable (rights = ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID)
    public Map<String, Object> moveOrRenameExtractionDefinitionFile(String srcRelPath, String targetRelPath) throws IOException
    {
        Map<String, Object> result = new HashMap<>();

        FileSource srcFile = null;
        FileSource targetFile = null;
        try
        {
            srcFile = (FileSource) _sourceResolver.resolveURI(ExtractionConstants.DEFINITIONS_DIR + srcRelPath);
            targetFile = (FileSource) _sourceResolver.resolveURI(ExtractionConstants.DEFINITIONS_DIR + targetRelPath);

            String sourceContext = ExtractionAccessController.ROOT_CONTEXT + "/" + srcRelPath;
            String targetContext = ExtractionAccessController.ROOT_CONTEXT + "/" + targetRelPath;

            result = _moveOrRenameSource(srcFile, targetFile, sourceContext, targetContext);
            
            if (result.containsKey("uri"))
            {
                String newURI = (String) result.get("uri");
                String path = newURI.substring(_root.getURI().length());
                result.put("path", path);
            }
        }
        finally
        {
            _sourceResolver.release(srcFile);
            _sourceResolver.release(targetFile);
        }
        
        return result;
    }
        
    /**
     * Move a file or folder
     * 
     * @param sourceFile The file/folder to move
     * @param targetFile The target file
     * @param sourceContext the source context
     * @param targetContext the target context
     * @return a result map with the name and uri of moved file in case of
     *         success.
     * @throws IOException If an error occurred manipulating the source
     */
    private Map<String, Object> _moveOrRenameSource(FileSource sourceFile, FileSource targetFile, String sourceContext, String targetContext) throws IOException
    {
        Map<String, Object> result = new HashMap<>();
            
        // Check if the user try to move files outside the root folder
        if (!StringUtils.startsWith(sourceFile.getURI(), _root.getURI()) || !StringUtils.startsWith(targetFile.getURI(), _root.getURI()))
        {
            result.put("success", false);
            result.put("error", "no-exists");
            
            getLogger().error("User '{}' tried to  move parameter file outside of the root extraction directory.", _currentUserProvider.getUser());
            
            return result;
        }
        
        if (!sourceFile.exists())
        {
            result.put("success", false);
            result.put("error", "no-exists");
            return result;
        }
        
        if (targetFile.exists())
        {
            // If both files are equals, there is no need to rename or move it
            if (sourceFile.getFile().equals(targetFile.getFile()))
            {
                result.put("success", true);
                result.put("name", targetFile.getName());
                result.put("uri", targetFile.getURI());
                return result;
            }
            else
            {
                result.put("success", false);
                result.put("error", "already-exists");
                return result;
            }
        }

        copyRightsRecursively(sourceContext, targetContext, sourceFile);
        if (sourceFile.getFile().isFile())
        {
            FileUtils.moveFile(sourceFile.getFile(), targetFile.getFile());
        }
        else
        {
            FileUtils.moveDirectory(sourceFile.getFile(), targetFile.getFile());
        }
        deleteRightsRecursively(sourceContext, targetFile);

        result.put("success", true);
        result.put("name", targetFile.getName());
        result.put("uri", targetFile.getURI());

        return result;
    }
    
    /**
     * Copy rights from one context to another one
     * @param sourceContext the source context
     * @param targetContext the target context
     * @param file the source of the file to copy
     */
    public void copyRightsRecursively(String sourceContext, String targetContext, TraversableSource file)
    {
        copyRights(sourceContext, targetContext);
        if (file.isCollection())
        {
            try
            {
                for (TraversableSource child : (Collection<TraversableSource>) file.getChildren())
                {
                    copyRightsRecursively(sourceContext + "/" + child.getName(), targetContext + "/" + child.getName(), child);
                }
            }
            catch (SourceException e)
            {
                throw new RuntimeException("Cannot list child elements of " + file.getURI(), e);
            }
        }
    }

    /**
     * Copy rights from one context to another one
     * @param context the context
     * @param file the source of the file to copy
     */
    public void deleteRightsRecursively(String context, TraversableSource file)
    {
        deleteRights(context);
        if (file.isCollection())
        {
            try
            {
                for (TraversableSource child : (Collection<TraversableSource>) file.getChildren())
                {
                    deleteRightsRecursively(context + "/" + child.getName(), child);
                }
            }
            catch (SourceException e)
            {
                throw new RuntimeException("Cannot list child elements of " + file.getURI(), e);
            }
        }
    }
    
    /**
     * Get the author of extraction
     * @param extractionPath the path of the extraction
     * @return the author
     */
    public UserIdentity getAuthor(FileSource extractionPath)
    {
        return _getExtractionAuthorCache().get(extractionPath, path -> _getUserIdentityByExtractionFile(path));
        
    }
    
    private UserIdentity _getUserIdentityByExtractionFile(FileSource extractionPath)
    {
        try
        {
            Extraction extraction = _definitionReader.readExtractionDefinitionFile(extractionPath.getFile());
            return extraction.getAuthor();
        }
        catch (Exception e)
        {
            throw new RuntimeException("Cannot read extraction " + extractionPath, e);
        }
    }
    
    private Cache<FileSource, UserIdentity> _getExtractionAuthorCache()
    {
        return this._cacheManager.get(EXTRACTION_AUTHOR_CACHE);
    }
    
    /**
     * Remove the last separator from the uri if it has any
     * @param uri the uri
     * @return the uri without any ending separator
     */
    public static String trimLastFileSeparator(String uri)
    {
        return StringUtils.endsWith(uri, "/") ? StringUtils.substringBeforeLast(uri, "/") : uri;
    }
    
    /**
     * Get the path of all children that match the provided value.
     * @param path the path to the extraction to consider as root
     * @param value the value
     * @return the list of path
     */
    @Callable(rights = {"Runtime_Rights_Rights_Handle", "Extraction_Rights_ExecuteExtraction"}) // assignment for the tree in assignment tool
    public List<String> getFilteredPath(String path, String value)
    {
        try
        {
            TraversableSource currentSrc = (TraversableSource) _sourceResolver.resolveURI(ExtractionConstants.DEFINITIONS_DIR + (path.length() > 0 ? "/" + path : ""));
            
            List<String> result = _fileHelper.filterSources(currentSrc, value);
            return result.stream()
                  .map(this::_toRelativePath)
                  .toList();
        }
        catch (IOException e)
        {
            getLogger().error("Failed to filter extraction definition at path '" + path + "'", e);
            return List.of();
        }
    }

    private String _toRelativePath(String absoluteURI)
    {
        // the root URI has a trailing slash or not depending on the existence of the folder at start time
        // always remove the trailing slash so that we have a consistent behavior
        return StringUtils.substringAfter(trimLastFileSeparator(absoluteURI), trimLastFileSeparator(_root.getURI()));
    }
}
