/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.ametys.plugins.extraction.execution.Extraction.ClausesVariable;

/**
 * Context of extraction components execution
 */
public class ExtractionExecutionContext
{
    private Map<String, Boolean> _displayOptionalColumns = new HashMap<>();
    private Map<ClausesVariable, List<String>> _clausesVariablesValues = new HashMap<>();
    private List<ExtractionExecutionContextHierarchyElement> _hierarchyElements = new ArrayList<>();
    private Locale _defaultLocale;
    
    /**
     * Default constructor
     */
    public ExtractionExecutionContext()
    {
    }
    
    /**
     * Creates a context by copy
     * @param context context to copy
     */
    public ExtractionExecutionContext(ExtractionExecutionContext context)
    {
        setDefaultLocale(context.getDefaultLocale());
        setDisplayOptionalColumns(context.getDisplayOptionalColumns());
        setClausesVariablesValues(context.getClausesVariablesValues());
        setHierarchyElements(context.getHierarchyElements());
    }
    
    /**
     * Set the default locale for this context of extraction
     * The default locale is used for export of localized values such as multilingual strings or contents.
     * @param defaultLocale the default locale
     */
    public void setDefaultLocale(Locale defaultLocale)
    {
        _defaultLocale = defaultLocale;
    }
    
    /**
     * Get the default locale for this context of extraction
     * @return the default locale. Can be null.
     */
    public Locale getDefaultLocale()
    {
        return _defaultLocale;
    }

    /**
     * Retrieves the variables controlling display of optional columns
     * @return The variables controlling display of optional columns
     */
    public Map<String, Boolean> getDisplayOptionalColumns()
    {
        return _displayOptionalColumns;
    }
    
    /**
     * Sets the variables controlling display of optional columns
     * @param displayOptionalColumns The variables to set
     */
    public void setDisplayOptionalColumns(Map<String, Boolean> displayOptionalColumns)
    {
        _displayOptionalColumns = displayOptionalColumns;
    }

    /**
     * Retrieves the variables values to use in clauses
     * @return the variables values to use in clauses
     */
    public Map<ClausesVariable, List<String>> getClausesVariablesValues()
    {
        return _clausesVariablesValues;
    }
    
    /**
     * Sets the variables values to use in clauses
     * @param clausesVariablesValues the variables values to set
     */
    public void setClausesVariablesValues(Map<ClausesVariable, List<String>> clausesVariablesValues)
    {
        _clausesVariablesValues = clausesVariablesValues;
    }
    
    /**
     * Retrieves the parents elements of the context
     * @return the elements of the context
     */
    public List<ExtractionExecutionContextHierarchyElement> getHierarchyElements()
    {
        return _hierarchyElements;
    }
    
    /**
     * Sets the parents elements of the context
     * @param hierarchyElements the hierarchy elements to set
     */
    public void setHierarchyElements(List<ExtractionExecutionContextHierarchyElement> hierarchyElements)
    {
        _hierarchyElements = hierarchyElements;
    }
    
}
