/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

const DefaultConfigurationBuilder = Java.type('org.apache.avalon.framework.configuration.DefaultConfigurationBuilder');

const InputStream = Java.type('java.io.InputStream');
const Files = Java.type('java.nio.file.Files');
const File = Java.type('java.io.File');
const FileInputStream = Java.type('java.io.FileInputStream');
const Configuration = Java.type('org.apache.avalon.framework.configuration.Configuration');

const READER = "READER";
const EXTRACTIONMIGRATIONPROFILELABEL = "Extraction migration";
const EXTRACTIONMIGRATIONPROFILEID = "extraction-migration";

const GroupIdentity = Java.type('org.ametys.core.group.GroupIdentity');
const UserIdentity = Java.type('org.ametys.core.user.UserIdentity');

const _srcResolver = Ametys.serviceManager.lookup("org.apache.excalibur.source.SourceResolver");
var rootDir = _srcResolver.resolveURI(org.ametys.plugins.extraction.ExtractionConstants.DEFINITIONS_DIR);

const _profileAssignmentStorageExtensionPoint = Ametys.serviceManager.lookup("org.ametys.core.right.ProfileAssignmentStorageExtensionPoint");
const _rightProfileDAO = Ametys.serviceManager.lookup("org.ametys.core.right.RightProfilesDAO");

/** read access profile tag */
const READ_ACCESS_TAG = "read-access";

/** write access profile tag */
const WRITE_ACCESS_TAG = "write-access";

/** groups tag, used for granted groups */
const GROUPS_TAG = "groups";

/** group tag, used for a granted group */
const GROUP_TAG = "group";

/** users tag, used for granted users */
const USERS_TAG = "users";

/** user tag, used for a granted user */
const USER_TAG = "user";

let extractionProfile = _rightProfileDAO.getProfile(EXTRACTIONMIGRATIONPROFILEID);
if (extractionProfile == null)
{
    extractionProfile = _rightProfileDAO.addProfile(EXTRACTIONMIGRATIONPROFILELABEL);
}

_rightProfileDAO.updateRights(extractionProfile, ["Extraction_Rights_EditExtraction"]);

parseDirectory(rootDir);

function parseDirectory(directory)
{
    if (directory.exists())
    {
        directory.getChildren().forEach(
            child =>
            {
                if (!isIgnoredSource(child))
                {
                    if (child.isCollection())
                    {
                        parseDirectory(child);
                    }
                    else
                    {
                        parseExtractionFile(child.getFile())
                    }
                }
            }
        );
    }
}

function isIgnoredSource(source)
{
    return !source.isCollection() && !source.getName().endsWith(".xml");
}

function parseExtractionFile(file)
{
    logger.info("Handling file: " + file);
    
    let configuration = new DefaultConfigurationBuilder().buildFromFile(file);
    let visibility = configuration.getChild("visibility", false);

    // If there is no visibility, the configuration is public
    if (visibility == null || visibility.getValue() == "public")
    {
        _profileAssignmentStorageExtensionPoint.allowProfileToAnyConnectedUser(READER, getExtractionFileContext(file));
    }
    else if (visibility.getValue() == "shared")
    {
        let readAccessRight = configuration.getChild(READ_ACCESS_TAG);
        
        let readAccessGroups = readAccessRight.getChild(GROUPS_TAG).getChildren(GROUP_TAG);
        for (let i = 0; i < readAccessGroups.length; i++)
        {
            let groupIdentity = new GroupIdentity(readAccessGroups[i].getAttribute("id"), readAccessGroups[i].getAttribute("groupDirectory"));
            _profileAssignmentStorageExtensionPoint.allowProfileToGroup(groupIdentity, READER, getExtractionFileContext(file));
        }
        
        let readAccessUsers = readAccessRight.getChild(USERS_TAG).getChildren(USER_TAG);
        for (let i = 0; i < readAccessUsers.length; i++) {
            let login = readAccessUsers[i].getAttribute("login");
            let population = readAccessUsers[i].getAttribute("population");
            let userIdentity = new UserIdentity(login, population);

            _profileAssignmentStorageExtensionPoint.allowProfileToUser(userIdentity, READER, getExtractionFileContext(file));
        };
        
        let writeAccessRight = configuration.getChild(WRITE_ACCESS_TAG);
        
        let writeAccessGroups = writeAccessRight.getChild(GROUPS_TAG).getChildren(GROUP_TAG);
        for (let i = 0; i < writeAccessGroups.length; i++)
        {
            let groupIdentity = new GroupIdentity(writeAccessGroups[i].getAttribute("id"), writeAccessGroups[i].getAttribute("groupDirectory"));
            
            _profileAssignmentStorageExtensionPoint.allowProfileToGroup(groupIdentity, READER, getExtractionFileContext(file));
            
            _profileAssignmentStorageExtensionPoint.allowProfileToGroup(groupIdentity, EXTRACTIONMIGRATIONPROFILEID, getExtractionFileContext(file));
        }
        
        let writeAccessUsers = writeAccessRight.getChild(USERS_TAG).getChildren(USER_TAG);
        for (let i = 0; i < writeAccessUsers.length; i++)
        {
            let login = writeAccessUsers[i].getAttribute("login");
            let population = writeAccessUsers[i].getAttribute("population");
            let userIdentity = new UserIdentity(login, population);

            _profileAssignmentStorageExtensionPoint.allowProfileToUser(userIdentity, READER, getExtractionFileContext(file));

            _profileAssignmentStorageExtensionPoint.allowProfileToUser(userIdentity, EXTRACTIONMIGRATIONPROFILEID, getExtractionFileContext(file));
        }
    }
}

function getExtractionFileContext(file)
{
    let rootPath = rootDir.getFile().getPath();
    let relPath = "/extraction" + file.getPath().substring(rootPath.length);
    
    // As the file separator is system-dependent, it should be replaced by a single separator
    relPath = relPath.replaceAll(File.separator, '/');

    // remove the last "/" if present
    if (relPath.endsWith("/"))
    {
        relPath = relPath.substring(0, relPath.length() - 1);
    }
    
    return relPath;
}
