/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.edition;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.search.GetQueryFromJSONHelper;
import org.ametys.cms.search.QueryBuilder;
import org.ametys.cms.search.model.SearchModel;
import org.ametys.cms.search.query.AndQuery;
import org.ametys.cms.search.query.ContentTypeOrMixinTypeQuery;
import org.ametys.cms.search.query.ContentTypeQuery;
import org.ametys.cms.search.query.MixinTypeQuery;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.cms.search.ui.model.SearchUIModelExtensionPoint;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.JSONUtils;
import org.ametys.core.util.language.UserLanguagesManager;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.queriesdirectory.Query;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.thesaurus.Thesaurus;
import org.ametys.plugins.thesaurus.ThesaurusDAO;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Extraction node edition manager
 */
public class EditExtractionNodeManager extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Avalon role name */
    public static final String ROLE = EditExtractionNodeManager.class.getName();
    
    private ThesaurusDAO _thesaurusDAO;
    private AmetysObjectResolver _resolver;
    private JSONUtils _jsonUtils;
    private GetQueryFromJSONHelper _getQueryFromJSONHelper;
    private QueryBuilder _queryBuilder;
    private I18nUtils _i18nUtils;
    private ContentTypeExtensionPoint _contentTypeExtensionPoint;
    private SearchUIModelExtensionPoint _searchUIModelExtensionPoint;
    private CurrentUserProvider _currentUserProvider;
    private UserManager _userManager;
    private UserLanguagesManager _userLanguagesManager;

    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _thesaurusDAO = (ThesaurusDAO) serviceManager.lookup(ThesaurusDAO.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _jsonUtils = (JSONUtils) serviceManager.lookup(JSONUtils.ROLE);
        _getQueryFromJSONHelper = (GetQueryFromJSONHelper) serviceManager.lookup(GetQueryFromJSONHelper.ROLE);
        _queryBuilder = (QueryBuilder) serviceManager.lookup(QueryBuilder.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE);
        _searchUIModelExtensionPoint = (SearchUIModelExtensionPoint) serviceManager.lookup(SearchUIModelExtensionPoint.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _userLanguagesManager = (UserLanguagesManager) serviceManager.lookup(UserLanguagesManager.ROLE);
    }
    
    /**
     * Retrieves configuration for extraction node edition
     * @return A map containing information about what is needed to create/edit an extraction node
     */
    @Callable (rights = ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID)
    public Map<String, Object> getNodeEditionConfiguration()
    {
        Map<String, Object> result = new HashMap<>();
        
        // Manages Search UI Models
        Set<String> searchUIModelIds = _searchUIModelExtensionPoint.getExtensionsIds();
        List<Object> searchUIModelsConfigurations = new ArrayList<>();
        for (String searchUIModelId : searchUIModelIds)
        {
            Map<String, Object> searchUIModelMap = new HashMap<>();
            searchUIModelMap.put("value",  searchUIModelId);
            searchUIModelMap.put("label", searchUIModelId);
            
            searchUIModelsConfigurations.add(searchUIModelMap);
        }
        result.put("searchUIModels", searchUIModelsConfigurations);
        
        // Manages thesaurii
        AmetysObjectIterable<Thesaurus> thesaurii = _thesaurusDAO.getThesaurii();
        List<Object> thesauriiConfigurations = new ArrayList<>();
        for (Thesaurus thesaurus : thesaurii)
        {
            Map<String, Object> thesaurusMap = new HashMap<>();
            thesaurusMap.put("value",  thesaurus.getId());
            thesaurusMap.put("label", thesaurus.getLabel());
            
            thesauriiConfigurations.add(thesaurusMap);
        }
        result.put("thesaurii", thesauriiConfigurations);
        
        return result;
    }
    
    /**
     * Retrieves microthesaurii for the given thesaurus
     * @param thesaurusId identifier of the thesaurus
     * @return A map containing microthesaurii of the thesaurus
     */
    @Callable (rights = ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID)
    public Map<String, Object> getMicroThesaurii(String thesaurusId)
    {
        Map<String, Object> result = new HashMap<>();
        
        if (thesaurusId != null)
        {
            String thesaurusLabel = _thesaurusDAO.getThesaurusLabel(thesaurusId);
            Thesaurus thesaurus = _thesaurusDAO.findThesaurusByLabel(thesaurusLabel);
            
            if (thesaurus != null)
            {
                List<String> microthesaurii = _thesaurusDAO.getMicrothesaurii(thesaurus.getId());
                
                String language = _userLanguagesManager.getDefaultLanguage();
                UserIdentity userIdentity = _currentUserProvider.getUser();
                if (userIdentity != null)
                {
                    User user = _userManager.getUser(userIdentity);
                    if (user != null)
                    {
                        language = user.getLanguage();
                    }
                }
                
                List<Map<String, Object>> microthesauriiConfigurations = new ArrayList<>();
                for (String microthesaurus : microthesaurii)
                {
                    I18nizableText i18nLabel = _contentTypeExtensionPoint.getExtension(microthesaurus).getLabel();
                    
                    String label = _i18nUtils.translate(i18nLabel, language);
    
                    Map<String, Object> thesaurusMap = new HashMap<>();
                    thesaurusMap.put("value",  microthesaurus);
                    thesaurusMap.put("text", label);
                    
                    microthesauriiConfigurations.add(thesaurusMap);
                }
                
                result.put("microthesaurii", microthesauriiConfigurations);
            }
        }
        return result;
    }
    
    /**
     * Retrieves content types configured on the given saved query
     * @param savedQueryId saved query identifier
     * @return A list containing the content types
     */
    @SuppressWarnings("unchecked")
    @Callable (rights = ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID)
    public List<String> getSavedQueryContentTypes(String savedQueryId)
    {
        List<String> result = new ArrayList<>();
        
        if (savedQueryId != null && !savedQueryId.isEmpty())
        {
            Query referencedQuery = _resolver.resolveById(savedQueryId);
            
            Map<String, Object> contentMap = _jsonUtils.convertJsonToMap(referencedQuery.getContent());
            Map<String, Object> exportParams = (Map<String, Object>) contentMap.get("exportParams");
            String modelId = (String) exportParams.get("model");
            
            if (modelId.contains("solr"))
            {
                Map<String, Object> values = (Map<String, Object>) exportParams.get("values");
                result.addAll((List<String>) values.get("contentTypes"));
            }
            else
            {
                SearchModel model = _getQueryFromJSONHelper.getSearchModel(exportParams);
                Map<String, Object> contextualParameters = Optional.ofNullable((Map<String, Object>) exportParams.get("contextualParameters")).orElseGet(HashMap::new);
                
                Map<String, Object> values = (Map<String, Object>) exportParams.get("values");
                String searchMode = Objects.toString(exportParams.get("searchMode"), "simple");

                org.ametys.cms.search.query.Query query = _queryBuilder.build(model, searchMode , values, contextualParameters);
                if (query instanceof AndQuery)
                {
                    Set<org.ametys.cms.search.query.Query> subqueries = new LinkedHashSet<>(((AndQuery) query).getQueries());
                    Predicate<org.ametys.cms.search.query.Query> isCTypeOrMixinOrBothQuery = q -> ContentTypeQuery.class.isInstance(q) || MixinTypeQuery.class.isInstance(q) || ContentTypeOrMixinTypeQuery.class.isInstance(q);
                    List<org.ametys.cms.search.query.Query> matchingQueries = subqueries.stream().distinct().filter(isCTypeOrMixinOrBothQuery).collect(Collectors.toList());
                    
                    for (org.ametys.cms.search.query.Query matchingQuery : matchingQueries)
                    {
                        if (matchingQuery instanceof ContentTypeQuery)
                        {
                            ContentTypeQuery cTypeQuery = (ContentTypeQuery) matchingQuery;
                            if (cTypeQuery.getOperator() == Operator.EQ)
                            {
                                result.addAll(cTypeQuery.getValue());
                            }
                        }
                        else if (matchingQuery instanceof MixinTypeQuery)
                        {
                            MixinTypeQuery mixinQuery = (MixinTypeQuery) matchingQuery;
                            if (mixinQuery.getOperator() == Operator.EQ)
                            {
                                result.addAll(mixinQuery.getValue());
                            }
                        }
                        else if (matchingQuery instanceof ContentTypeOrMixinTypeQuery)
                        {
                            ContentTypeOrMixinTypeQuery cTypeOrMixinQuery = (ContentTypeOrMixinTypeQuery) matchingQuery;
                            if (cTypeOrMixinQuery.getOperator() == Operator.EQ)
                            {
                                result.addAll(cTypeOrMixinQuery.getIds());
                            }
                        }
                    }
                }
            }
        }
        
        return result;
    }
}
