/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class provides a widget to select a metadata.
 */
Ext.define('Ametys.plugins.extraction.execution.widget.SelectPipeline', {
    extend: 'Ametys.form.widget.ComboBox',
    
    /**
     * @cfg {String} [extraction] The extraction. If {@link #cfg-extractionField} is used, this will be ignored.
     */
    
    /**
     * @cfg {String} [extractionField] The relative path the extraction field,
     *      to allow the list of pipelines to be updated given the
     *      value of the extraction field. To force the extraction, use {@link #cfg-extraction} instead.
     */
    /**
     * @private
     * @property {String} _extractionFieldName The property related to {@link #cfg-extractionField}
     */

    constructor: function (config)
    {
        config.proxyStore = {
            type: 'ametys',
            role: "org.ametys.plugins.extraction.execution.pipeline.PipelineManager",
            methodName: "getAvailablePipelines",
            methodArguments: ['extractionId'],
            reader: {
                type: 'json',
                rootProperty: 'pipelines'
            }
        };
        config.autoLoad = true;
        
        this.callParent(arguments);
        
        this._extractionFieldName = config.extractionField || null;
        this.form.onRelativeFieldsChange(this._extractionFieldName, this, this._relativeFieldChange);
       
        if (!Ext.isEmpty(this.extraction))
        {
            this.form.executeFormReady(this._relativeFieldChange, this);
        }
    },
    
    /**
     * Get the store configuration
     * @param {Object} config The current configuration object
     * @return {Object} The store configuration
     */
    getStoreCfg: function(config)
    {
        var storeCfg = this.callParent(arguments);

        Ext.apply(storeCfg, {
            listeners: {
                beforeload: {fn: this._onStoreBeforeLoad, scope: this}
            }
        });
        
        return storeCfg;
    },
    
    /**
     * Set the request parameters before loading the store.
     * @param {Ext.data.Store} store The store.
     * @param {Ext.data.operation.Operation} operation The Ext.data.Operation object that will be passed to the Proxy to load the Store.
     * @private
     */
    _onStoreBeforeLoad: function(store, operation)
    {
        var extractionField = this.form.getRelativeField(this._extractionFieldName, this);
        if (extractionField && extractionField.getValue() == null)
        {
            return false; // cancel the load
        }
            
        operation.setParams(Ext.apply(operation.getParams() || {}, {
            extractionId: extractionField ? extractionField.getValue() : this.extraction
        }));
    },
    
    /**
     * @private
     * Listener called when the value of a relative field changes
     */
    _relativeFieldChange: function()
    {
        this.getStore().load();
    }
});