/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.component;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.solr.client.solrj.util.ClientUtils;
import org.xml.sax.ContentHandler;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.data.type.ModelItemTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.cms.search.GetQueryFromJSONHelper;
import org.ametys.cms.search.content.ContentSearcherFactory;
import org.ametys.cms.search.content.ContentSearcherFactory.SimpleContentSearcher;
import org.ametys.cms.search.model.SearchModel;
import org.ametys.cms.search.model.SearchModelHelper;
import org.ametys.cms.search.model.SystemProperty;
import org.ametys.cms.search.model.SystemPropertyExtensionPoint;
import org.ametys.cms.search.query.QuerySyntaxException;
import org.ametys.cms.search.solr.SolrContentQueryHelper;
import org.ametys.core.util.JSONUtils;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.extraction.execution.Extraction.ClausesVariable;
import org.ametys.plugins.extraction.execution.Extraction.ClausesVariableType;
import org.ametys.plugins.extraction.execution.ExtractionExecutionContext;
import org.ametys.plugins.extraction.execution.ExtractionExecutionContextHierarchyElement;
import org.ametys.plugins.queriesdirectory.Query;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.EmptyIterable;
import org.ametys.plugins.thesaurus.ThesaurusDAO;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.model.type.ModelItemType;

/**
 * This class represents an extraction component with a solr query
 */
public abstract class AbstractSolrExtractionComponent extends AbstractExtractionComponent
{
    /** Regex used to extract variables from an expression. A variable is inside a ${} */
    private static final String __EXTRACT_VARIABLES_REGEX = "\\$\\{([^{}]+)\\}";
    
    /**
     * Regex used to check variables from a join expression: \.\.(?:\/\.\.)*(?:\/[^\/}]+)?
     * variable starts with .. (to get the direct parent),
     * has several /.. (to get parent of parent of (...))
     * and can have a /metadataName (to specify the metadata to join on)
     */
    private static final String __CHECK_JOIN_VARIABLES_REGEX = "\\.\\.(?:\\/\\.\\.)*(?:\\/[^\\/}]+)?";
    
    /** Content types concerned by the solr search */
    protected Set<String> _contentTypes = new HashSet<>();
    
    /** Reference id of a recorded query */
    protected String _queryReferenceId;
    
    /** The list of clauses */
    protected List<ExtractionClause> _clauses = new ArrayList<>();
    
    /** Helper to resolve referenced query infos */
    protected GetQueryFromJSONHelper _getQueryFromJSONHelper;
    
    /** Util class to manipulate JSON String */
    protected JSONUtils _jsonUtils;
    
    private AmetysObjectResolver _resolver;
    private SystemPropertyExtensionPoint _systemPropertyExtensionPoint;
    private ContentHelper _contentHelper;
    private ContentSearcherFactory _contentSearcherFactory;
    private SearchModelHelper _searchModelHelper;
    private ModelItemTypeExtensionPoint _contentAttributeTypeExtensionPoint;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _jsonUtils = (JSONUtils) serviceManager.lookup(JSONUtils.ROLE);
        _getQueryFromJSONHelper = (GetQueryFromJSONHelper) serviceManager.lookup(GetQueryFromJSONHelper.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _systemPropertyExtensionPoint = (SystemPropertyExtensionPoint) serviceManager.lookup(SystemPropertyExtensionPoint.ROLE);
        _contentHelper = (ContentHelper) serviceManager.lookup(ContentHelper.ROLE);
        _contentSearcherFactory = (ContentSearcherFactory) serviceManager.lookup(ContentSearcherFactory.ROLE);
        _searchModelHelper = (SearchModelHelper) serviceManager.lookup(SearchModelHelper.ROLE);
        _contentAttributeTypeExtensionPoint = (ModelItemTypeExtensionPoint) serviceManager.lookup(ModelItemTypeExtensionPoint.ROLE_CONTENT_ATTRIBUTE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        
        Configuration clauses = configuration.getChild("clauses");
        for (Configuration clause : clauses.getChildren("clause"))
        {
            addClauses(clause.getValue());
        }

        _contentTypes = new HashSet<>();
        if (Arrays.asList(configuration.getAttributeNames()).contains("ref"))
        {
            if (Arrays.asList(configuration.getAttributeNames()).contains("contentTypes"))
            {
                throw new IllegalArgumentException(getLogsPrefix() + "a component with a query reference should not specify a content type");
            }
            
            _queryReferenceId = configuration.getAttribute("ref");
        }
        else
        {
            String contentTypesString = configuration.getAttribute("contentTypes");
            _contentTypes.addAll(org.ametys.core.util.StringUtils.stringToCollection(contentTypesString));
        }
    }
    
    @Override
    public void prepareComponentExecution(ExtractionExecutionContext context) throws Exception
    {
        super.prepareComponentExecution(context);
        
        if (_queryReferenceId != null && !_queryReferenceId.isEmpty())
        {
            Query referencedQuery = _resolver.resolveById(_queryReferenceId);
            computeReferencedQueryInfos(referencedQuery.getContent());
        }
        
        _computeClausesInfos(context);
    }

    /**
     * Manages the stored query referenced by the component
     * @param refQueryContent referenced query content
     * @throws QuerySyntaxException if there is a syntax error in the referenced query
     */
    @SuppressWarnings("unchecked")
    protected void computeReferencedQueryInfos(String refQueryContent) throws QuerySyntaxException
    {
        Map<String, Object> contentMap = _jsonUtils.convertJsonToMap(refQueryContent);
        Map<String, Object> exportParams = (Map<String, Object>) contentMap.get("exportParams");
        String modelId = (String) exportParams.get("model");
        
        String q;
        if (modelId.contains("solr"))
        {
            Map<String, Object> values = (Map<String, Object>) exportParams.get("values");
            String baseQuery = (String) values.get("query");
            
            _contentTypes = new HashSet<>((List<String>) values.get("contentTypes"));
            
            q = SolrContentQueryHelper.buildQuery(_searchModelHelper, baseQuery, _contentTypes, Collections.emptySet());
        }
        else
        {
            SearchModel model = _getQueryFromJSONHelper.getSearchModel(exportParams);
            List<String> contentTypesToFill = new ArrayList<>();
            org.ametys.cms.search.query.Query query = _getQueryFromJSONHelper.getQueryFromModel(model, exportParams, contentTypesToFill);
            
            q = query.build();
            _contentTypes = new HashSet<>(contentTypesToFill);
        }
        
        ExtractionClause clause = new ExtractionClause();
        clause.setExpression(q);
        _clauses.add(0, clause);
    }

    private void _computeClausesInfos(ExtractionExecutionContext context)
    {
        for (ExtractionClause clause : _clauses)
        {
            String clauseExpression = clause.getExpression();
            clause.setExpression(clauseExpression);
            
            List<ExtractionClauseGroup> groups = _extractGroupExpressionsFromClause(clauseExpression);
            if (!groups.isEmpty())
            {
                Collection<String> groupExpressions = groups.stream()
                        .map(ExtractionClauseGroup::getCompleteExpression)
                        .collect(Collectors.toList());
                if (_hasVariablesOutsideGroups(clauseExpression, groupExpressions, context.getClausesVariablesValues().keySet()))
                {
                    throw new IllegalArgumentException(getLogsPrefix() + "if there's at least one group, every variable should be in a group.");
                }
            }
            else
            {
                // The only group is the entire expression
                // The complete expression is the same as the classic one (there is no characters used to identify the group)
                ExtractionClauseGroup group = new ExtractionClauseGroup();
                group.setCompleteExpression(clauseExpression);
                group.setExpression(clauseExpression);
                groups.add(group);
            }
            
            for (ExtractionClauseGroup group : groups)
            {
                Set<String> variables = new HashSet<>(_extractVariablesFromClauseExpression(group.getExpression(), context.getClausesVariablesValues().keySet()));
                if (!variables.isEmpty())
                {
                    if (variables.size() > 1)
                    {
                        throw new IllegalArgumentException(getLogsPrefix() + "only variables with same name are allowed within a single group");
                    }
                    
                    for (String variable : variables)
                    {
                        String[] pathSegments = variable.split(JOIN_HIERARCHY_SEPARATOR);
                        String fieldPath = pathSegments[pathSegments.length - 1];
    
                        group.setVariable(variable);
                        group.setFieldPath(fieldPath);
                    }
                }
                
                clause.addGroup(group);
            }
        }
    }
    
    private boolean _hasVariablesOutsideGroups(String clauseExpression, Collection<String> groupExpressions, Collection<ClausesVariable> clausesVariable)
    {
        List<String> variablesInClause = _extractVariablesFromClauseExpression(clauseExpression, clausesVariable);
        List<String> variablesInGroups = new ArrayList<>();
        for (String groupExpression : groupExpressions)
        {
            variablesInGroups.addAll(_extractVariablesFromClauseExpression(groupExpression, clausesVariable));
        }
        return variablesInClause.size() > variablesInGroups.size();
    }

    List<ExtractionClauseGroup> _extractGroupExpressionsFromClause(String expression)
    {
        List<ExtractionClauseGroup> groupExpressions = new ArrayList<>();
        int indexOfGroup = expression.indexOf("#{");
        while (indexOfGroup != -1)
        {
            StringBuilder currentGroupSb = new StringBuilder();
            int endIndex = indexOfGroup;
            int braceLevel = 0;
            for (int i = indexOfGroup + 2; i < expression.length(); i++)
            {
                endIndex = i;
                char currentChar = expression.charAt(i);
                if ('{' == currentChar)
                {
                    braceLevel++;
                }
                else if ('}' == currentChar)
                {
                    if (0  == braceLevel)
                    {
                        ExtractionClauseGroup group = new ExtractionClauseGroup();
                        String currentGroup = currentGroupSb.toString();
                        group.setCompleteExpression("#{" + currentGroup + "}");
                        group.setExpression(currentGroup);
                        groupExpressions.add(group);
                        break;
                    }
                    braceLevel--;
                }
                currentGroupSb.append(currentChar);
            }
            
            indexOfGroup = expression.indexOf("#{", endIndex);
        }
        return groupExpressions;
    }

    List<String> _extractVariablesFromClauseExpression(String expression, Collection<ClausesVariable> clausesVariable)
    {
        List<String> variableNames = new ArrayList<>();
        
        Pattern variablePattern = Pattern.compile(__EXTRACT_VARIABLES_REGEX);
        Matcher variableMatcher = variablePattern.matcher(expression);
        
        while (variableMatcher.find())
        {
            String variableName = variableMatcher.group(1);
            Pattern joinPattern = Pattern.compile(__CHECK_JOIN_VARIABLES_REGEX);
            Matcher joinMatcher = joinPattern.matcher(variableName);
            if (clausesVariable.stream().map(var -> var.name()).anyMatch(variableName::equals) || joinMatcher.matches())
            {
                variableNames.add(variableName);
            }
        }
        
        return variableNames;
    }

    @Override
    public void executeComponent(ContentHandler contentHandler, ExtractionExecutionContext context) throws Exception
    {
        Iterable<Content> contents = getContents(context);
        processContents(contents, contentHandler, context);
    }
    
    List<String> _getClauseQueries(ExtractionExecutionContext context)
    {
        List<String> clauseQueries = new ArrayList<>();
        
        for (ExtractionClause clause : _clauses)
        {
            String expression = clause.getExpression();
            
            // Resolve all groups
            for (ExtractionClauseGroup group : clause.getGroups())
            {
                String variableName = group.getVariable();
                
                if (StringUtils.isNotEmpty(variableName))
                {
                    Collection<String> groupExpressions = new ArrayList<>();
                    
                    if (context.getClausesVariablesValues()
                               .keySet()
                               .stream()
                               .anyMatch(clausesVariable -> clausesVariable.name().equals(variableName)))
                    {
                        Entry<ClausesVariable, List<String>> clausesVariableValues = context.getClausesVariablesValues()
                                                                                            .entrySet()
                                                                                            .stream()
                                                                                            .filter(entry -> entry.getKey().name().equals(variableName))
                                                                                            .findFirst()
                                                                                            .get();
                        
                        if (ClausesVariableType.SELECT_CONTENTS.equals(clausesVariableValues.getKey().type()))
                        {
                            String attributeTypeId = ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID;
                            Collection<Object> values = clausesVariableValues.getValue()
                                                                             .stream()
                                                                             .map(Object.class::cast)
                                                                             .collect(Collectors.toList());
                        
                            for (Object value : values)
                            {
                                String valueAsString = _getValueAsString(value, attributeTypeId, group.getFieldPath());
                                groupExpressions.add("(" + group.getExpression().replace("${" + variableName + "}", valueAsString) + ")");
                            }
                        }
                        else
                        {
                            List<String> solrRequests = clausesVariableValues.getValue();
                            if (solrRequests.size() == 1)
                            {
                                groupExpressions.add("(" + group.getExpression().replace("${" + variableName + "}", solrRequests.get(0)) + ")");
                            }
                        }
                    }
                    else
                    {
                        ExtractionExecutionContextHierarchyElement currentContextHierarchyElement = _getCurrentContextElementFromVariable(variableName, context.getHierarchyElements());
                        
                        ExtractionComponent contextComponent = currentContextHierarchyElement.getComponent();
                        String attributeTypeId = _getAttributeTypeId(group.getFieldPath(), contextComponent.getContentTypes());
                        Collection<Object> values = _getValuesFromVariable(group.getFieldPath(), attributeTypeId, currentContextHierarchyElement, context.getDefaultLocale());
                    
                        for (Object value : values)
                        {
                            String valueAsString = _getValueAsString(value, attributeTypeId, group.getFieldPath());
                            groupExpressions.add("(" + group.getExpression().replace("${" + variableName + "}", valueAsString) + ")");
                        }
                    }
                    
                    if (groupExpressions.isEmpty())
                    {
                        getLogger().warn(getLogsPrefix() + "no value found for field '" + group.getFieldPath() + "'. The query of this component can't be achieved");
                        return null;
                    }
                    
                    String groupReplacement =  StringUtils.join(groupExpressions, " OR ");
                    expression = expression.replace(group.getCompleteExpression(), "(" + groupReplacement + ")");
                }
            }
            
            clauseQueries.add(expression);
        }
        
        return clauseQueries;
    }

    private ExtractionExecutionContextHierarchyElement _getCurrentContextElementFromVariable(String variable, List<ExtractionExecutionContextHierarchyElement> context)
    {
        int lastIndexOfSlash = variable.lastIndexOf(JOIN_HIERARCHY_SEPARATOR);
        int indexOfCurrentContext = -1;
        if (lastIndexOfSlash == -1)
        {
            indexOfCurrentContext = context.size() - 1;
        }
        else
        {
            int hierarchicalLevel = (lastIndexOfSlash + 1) / 3;
            indexOfCurrentContext = context.size() - hierarchicalLevel;
            if (variable.endsWith(JOIN_HIERARCHY_ELEMENT))
            {
                indexOfCurrentContext--;
            }
        }
        if (indexOfCurrentContext < 0 || indexOfCurrentContext >= context.size())
        {
            throw new IllegalArgumentException(getLogsPrefix() + "join on '" + variable + "' does not refer to an existing parent");
        }
        return context.get(indexOfCurrentContext);
    }
    
    /**
     * Retrieves the field path's attribute type identifier from content types
     * @param fieldPath the field path
     * @param contentTypeIds the content types identifiers
     * @return the attribute type identifier
     */
    protected String _getAttributeTypeId(String fieldPath, Collection<String> contentTypeIds)
    {
        // Manage direct content references
        if (JOIN_HIERARCHY_ELEMENT.equals(fieldPath))
        {
            return ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID;
        }
        
        // Manage System Properties
        String[] pathSegments = fieldPath.split(EXTRACTION_ITEM_PATH_SEPARATOR);
        String propertyName = pathSegments[pathSegments.length - 1];
        if (_systemPropertyExtensionPoint.hasExtension(propertyName))
        {
            SystemProperty systemProperty = _systemPropertyExtensionPoint.getExtension(propertyName);
            return systemProperty.getType().getId();
        }
        
        String fieldPathWthClassicSeparator = fieldPath.replaceAll(EXTRACTION_ITEM_PATH_SEPARATOR, ModelItem.ITEM_PATH_SEPARATOR);
        Collection<ContentType> contentTypes = contentTypeIds.stream()
                .map(_contentTypeExtensionPoint::getExtension)
                .collect(Collectors.toList());
        
        if (ModelHelper.hasModelItem(fieldPathWthClassicSeparator, contentTypes))
        {
            ModelItem modelItem = ModelHelper.getModelItem(fieldPathWthClassicSeparator, contentTypes);
            return modelItem.getType().getId();
        }
        
        throw new IllegalArgumentException(getLogsPrefix() + "join on '" + fieldPath + "'. This attribute is not available");
    }

    private Collection<Object> _getValuesFromVariable(String fieldPath, String attributeTypeId, ExtractionExecutionContextHierarchyElement contextHierarchyElement, Locale defaultLocale)
    {
        Collection<Object> values = new LinkedHashSet<>();
        
        Iterable<Content> contents = contextHierarchyElement.getContents();
        for (Content content: contents)
        {
            boolean isAutoposting = contextHierarchyElement.isAutoposting();
            Collection<Object> contentValues = _getContentValuesFromVariable(content, fieldPath, attributeTypeId, isAutoposting, defaultLocale);
            values.addAll(contentValues);
        }
        
        return values;
    }
    
    private Collection<Object> _getContentValuesFromVariable(Content content, String fieldPath, String attributeTypeId, boolean isAutoposting, Locale defaultLocale)
    {
        Collection<Object> values = new LinkedHashSet<>();
        
        Object value = _getContentValue(content, fieldPath);
        if (value == null)
        {
            return Collections.emptyList();
        }
        
        if (value instanceof Collection<?>)
        {
            values.addAll((Collection<?>) value);
        }
        else
        {
            values.add(value);
        }
        
        Collection<Object> result = new LinkedHashSet<>(values);
        
        if (isAutoposting && ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID.equals(attributeTypeId))
        {
            for (Object object : values)
            {
                Optional<? extends Content> parent = object instanceof ContentValue ? ((ContentValue) object).getContentIfExists() : Optional.of((Content) object);
                ContentType contentType = parent.map(_contentTypesHelper::getFirstContentType)
                                                .orElse(null);
                
                // Manage autoposting only if the current value is a thesaurus term
                if (contentType != null && Arrays.asList(_contentTypesHelper.getSupertypeIds(contentType.getId()).getLeft()).contains(ThesaurusDAO.MICROTHESAURUS_ABSTRACT_CONTENT_TYPE))
                {
                    AmetysObjectIterable<Content> chidren = _thesaurusDAO.getChildTerms(contentType.getId(), parent.get().getId());
                    for (Content child : chidren)
                    {
                        Collection<Object> childValues = _getContentValuesFromVariable(child, JOIN_HIERARCHY_ELEMENT, attributeTypeId, isAutoposting, defaultLocale);
                        result.addAll(childValues);
                    }
                }
            }
        }

        return result;
    }
    
    private Object _getContentValue(Content content, String fieldPath)
    {
        if (JOIN_HIERARCHY_ELEMENT.equals(fieldPath))
        {
            return content;
        }
        else 
        {
            String fieldPathWthClassicSeparator = fieldPath.replaceAll(EXTRACTION_ITEM_PATH_SEPARATOR, ModelItem.ITEM_PATH_SEPARATOR);
            return _contentHelper.getValue(content, fieldPathWthClassicSeparator);
        }
    }
    
    private <T> String _getValueAsString(Object value, String attributeTypeId, String fieldPath)
    {
        ModelItemType modelItemType = _contentAttributeTypeExtensionPoint.getExtension(attributeTypeId);
        if (modelItemType instanceof ElementType)
        {
            @SuppressWarnings("unchecked")
            ElementType<T> elementType = (ElementType<T>) modelItemType;
            T typedValue = elementType.castValue(value);
            String valueAsString = elementType.toString(typedValue);
            return ClientUtils.escapeQueryChars(valueAsString);
        }
        else
        {
            throw new IllegalArgumentException(getLogsPrefix() + "join on '" + fieldPath + "'. Attribute type '" + attributeTypeId + "' is not supported by extraction module");
        }
    }
    
    /**
     * Retrieves the content searcher to use for solr search
     * @return the content searcher
     */
    protected SimpleContentSearcher getContentSearcher()
    {
        return _contentSearcherFactory.create(_contentTypes);
    }
    
    /**
     * Gets the content results from Solr
     * @param context component execution context
     * @return the content results from Solr
     * @throws Exception if an error occurs
     */
    protected Iterable<Content> getContents(ExtractionExecutionContext context) throws Exception
    {
        List<String> clauseQueries = _getClauseQueries(context);
        if (clauseQueries == null)
        {
            return new EmptyIterable<>();
        }
        else
        {
            List<String> filterQueryStrings = clauseQueries
                    .stream()
                    .map(LambdaUtils.wrap(clauseQuery -> SolrContentQueryHelper.buildQuery(_searchModelHelper, clauseQuery, Collections.emptySet(), Collections.emptySet())))
                    .collect(Collectors.toList());
            return getContentSearcher()
                    .withFilterQueryStrings(filterQueryStrings)
                    .setCheckRights(false)
                    .search("*:*");
        }
    }

    /**
     * Process result contents to format the result document
     * @param contents search results
     * @param contentHandler result document
     * @param context component execution context
     * @throws Exception if an error occurs
     */
    protected abstract void processContents(Iterable<Content> contents, ContentHandler contentHandler, ExtractionExecutionContext context) throws Exception;

    @Override
    public Map<String, Object> getComponentDetailsForTree()
    {
        Map<String, Object> details = super.getComponentDetailsForTree();
        
        @SuppressWarnings("unchecked")
        Map<String, Object> data = (Map<String, Object>) details.get("data");
        
        List<String> clauses = new ArrayList<>();
        for (ExtractionClause clause : this.getClauses())
        {
            clauses.add(clause.getExpression());
        }
        data.put("clauses", clauses);
        
        data.put("useQueryRef", StringUtils.isNotEmpty(_queryReferenceId));
        data.put("contentTypes", this.getContentTypes());
        data.put("queryReferenceId", this.getQueryReferenceId());
        
        return details;
    }
    
    public Set<String> getContentTypes()
    {
        return _contentTypes;
    }

    /**
     * Add content types to component
     * @param contentTypes Array of content types to add
     */
    public void addContentTypes(String... contentTypes)
    {
        _contentTypes.addAll(Arrays.asList(contentTypes));
    }

    /**
     * Retrieves the id of the referenced query
     * @return the id of the referenced query
     */
    public String getQueryReferenceId()
    {
        return _queryReferenceId;
    }
    
    /**
     * Sets the id of the referenced query
     * @param queryReferenceId The id of the referenced query to set
     */
    public void setQueryReferenceId(String queryReferenceId)
    {
        _queryReferenceId = queryReferenceId;
    }

    /**
     * Retrieves the component clauses
     * @return the component clauses
     */
    public List<ExtractionClause> getClauses()
    {
        return _clauses;
    }

    /**
     * Add clauses to the component. Do not manage clauses' groups
     * @param expressions Array clauses expressions to add
     */
    public void addClauses(String... expressions)
    {
        for (String expression : expressions)
        {
            ExtractionClause clause = new ExtractionClause();
            clause.setExpression(expression);
            _clauses.add(clause);
        }
    }
}
