/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.component;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringJoiner;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;

import org.ametys.cms.repository.Content;
import org.ametys.cms.search.SortOrder;
import org.ametys.cms.search.cocoon.GroupSearchContent;
import org.ametys.cms.search.content.ContentSearcherFactory.ContentSearchSort;
import org.ametys.cms.search.content.ContentSearcherFactory.SimpleContentSearcher;
import org.ametys.cms.search.query.QuerySyntaxException;
import org.ametys.cms.search.ui.model.ColumnHelper;
import org.ametys.cms.search.ui.model.ColumnHelper.Column;
import org.ametys.core.util.StringUtils;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.execution.ExtractionExecutionContext;
import org.ametys.plugins.extraction.execution.ExtractionExecutionContextHierarchyElement;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ViewItemContainer;
import org.ametys.runtime.model.type.DataContext;

/**
 * This class represents a query component of the extraction module
 */
public class QueryExtractionComponent extends AbstractGroupExtractionComponent implements TwoStepsExecutingExtractionComponent
{
    private boolean _overrideColumns;
    private List<ExtractionColumn> _columns = new ArrayList<>();
    private boolean _overrideSorts;
    private Map<String, SortOrder> _sortMap = new LinkedHashMap<>();
    private List<ContentSearchSort> _sorts = new ArrayList<>();
    
    private ViewItemContainer _resultItems;
    
    private ColumnHelper _columnHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _columnHelper = (ColumnHelper) serviceManager.lookup(ColumnHelper.ROLE);
    }
    
    @Override
    public void configure(Configuration query) throws ConfigurationException
    {
        super.configure(query);
        
        Configuration columnsConfiguration = query.getChild("columns");
        _overrideColumns = columnsConfiguration.getAttributeAsBoolean("override", false);
        for (Configuration columnConfiguration : columnsConfiguration.getChildren("column"))
        {
            ExtractionColumn column = new ExtractionColumn();
            column.setFieldPath(columnConfiguration.getValue());
            column.setDisplayOptionalName(columnConfiguration.getAttribute("optional", null));
            _columns.add(column);
        }
        
        Configuration sorts = query.getChild("sorts");
        _overrideSorts = sorts.getAttributeAsBoolean("override", false);
        for (Configuration sort : sorts.getChildren("sort"))
        {
            String orderAsString = sort.getAttribute("order", "ASC");
            SortOrder order = orderAsString.equalsIgnoreCase("ASC") ? SortOrder.ASC : SortOrder.DESC;
            String fieldPath = sort.getValue();
            _sortMap.put(fieldPath, order);
        }
    }
    
    @Override
    public void prepareComponentExecution(ExtractionExecutionContext context) throws Exception
    {
        super.prepareComponentExecution(context);
        
        List<String> columnFieldPaths = _getColumnsToDisplay(context);
        _resultItems = _buildResulItems(columnFieldPaths);
        
        // Replace attribute separator for sorts coming from configuration
        for (Map.Entry<String, SortOrder> entry : _sortMap.entrySet())
        {
            String fieldPath = entry.getKey().replaceAll(EXTRACTION_ITEM_PATH_SEPARATOR, ModelItem.ITEM_PATH_SEPARATOR);
            _sorts.add(new ContentSearchSort(fieldPath, entry.getValue()));
        }
        
        // Sorts can come from configuration or from referenced query
        for (ContentSearchSort sort : _sorts)
        {
            // get attribute type just to throw an Exception if the field is not available for content types 
            this._getAttributeTypeId(sort.sortField(), _contentTypes);
        }
    }

    private List<String> _getColumnsToDisplay(ExtractionExecutionContext context)
    {
        Map<String, Boolean> displayOptionalColumns = context.getDisplayOptionalColumns();
        List<String> columnFieldPaths = new ArrayList<>();
        for (ExtractionColumn column : _columns)
        {
            String displayOptionalColumnName = column.getDisplayOptionalName();
            if (!displayOptionalColumns.containsKey(displayOptionalColumnName) || displayOptionalColumns.get(displayOptionalColumnName))
            {
                columnFieldPaths.add(column.getFieldPath());
            }
        }
        return columnFieldPaths;
    }
    
    private ViewItemContainer _buildResulItems(List<String> columnFieldPaths)
    {
        Set<String> contentTypeIds = _contentTypesHelper.getCommonAncestors(_contentTypes);
        List<Column> columns = _columnHelper.getColumns(columnFieldPaths, contentTypeIds);
        
        return _columnHelper.createViewFromColumns(contentTypeIds, columns, true);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    protected void computeReferencedQueryInfos(String refQueryContent) throws QuerySyntaxException
    {
        super.computeReferencedQueryInfos(refQueryContent);
        
        Map<String, Object> contentMap = _jsonUtils.convertJsonToMap(refQueryContent);
        Map<String, Object> exportParams = (Map<String, Object>) contentMap.get("exportParams");
        
        if (!_overrideColumns)
        {
            Map<String, Object> values = (Map<String, Object>) exportParams.get("values");
            Object columnFieldPathsAsObject = values.get("columns");
            
            Collection<String> columnFieldPaths; 
            if (columnFieldPathsAsObject instanceof String)
            {
                columnFieldPaths = StringUtils.stringToCollection((String) columnFieldPathsAsObject);
            }
            else
            {
                columnFieldPaths = (List<String>) columnFieldPathsAsObject;
            }
            
            for (String fieldPath :  columnFieldPaths)
            {
                ExtractionColumn column = new ExtractionColumn();
                column.setFieldPath(fieldPath);
                _columns.add(column);
            }
        }
        
        if (!_overrideSorts)
        {
            _sorts.addAll(0, _getQueryFromJSONHelper.getSort(exportParams));
        }
    }

    @Override
    protected SimpleContentSearcher getContentSearcher()
    {
        return super.getContentSearcher().withSort(_sorts);
    }

    @Override
    protected void processContents(Iterable<Content> contents, ContentHandler contentHandler, ExtractionExecutionContext context) throws Exception
    {
        if (contents.iterator().hasNext())
        {
            XMLUtils.startElement(contentHandler, _tagName);
            
            GroupSearchContent rootGroup = organizeContentsInGroups(contents, context.getDefaultLocale());
            saxGroup(contentHandler, rootGroup, 0, context, _resultItems);
            
            XMLUtils.endElement(contentHandler, _tagName);
        }
    }
    
    @Override
    public Iterable<Content> computeFirstLevelResults(ExtractionExecutionContext context) throws Exception
    {
        return getContents(context);
    }
    
    @Override
    public void executeFor(ContentHandler contentHandler, Iterable<Content> firstLevelResults, ExtractionExecutionContext context) throws Exception
    {
        processContents(firstLevelResults, contentHandler, context);
    }

    @Override
    protected void saxContents(ContentHandler contentHandler, ExtractionExecutionContext context, ViewItemContainer resultItems, List<Content> contents) throws Exception
    {
        for (int currentContentIndex = 0; currentContentIndex < contents.size(); currentContentIndex++)
        {
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug(getLogsPrefix() + "executing content " + (currentContentIndex + 1) + "/" + contents.size());
            }
            
            Content content = contents.get(currentContentIndex);
            
            AttributesImpl attributes = new AttributesImpl();
            attributes.addCDATAAttribute("id", content.getId());
            attributes.addCDATAAttribute("name", content.getName());
            attributes.addCDATAAttribute("title", content.getTitle(context.getDefaultLocale()));
            if (content.getLanguage() != null)
            {
                attributes.addCDATAAttribute("language", content.getLanguage());
            }
                
            XMLUtils.startElement(contentHandler, "content", attributes);
            
            DataContext dataContext = DataContext.newInstance()
                                                 .withLocale(context.getDefaultLocale())
                                                 .withEmptyValues(false);
            content.dataToSAX(contentHandler, resultItems, dataContext);
            
            ExtractionExecutionContextHierarchyElement currentContext = new ExtractionExecutionContextHierarchyElement(this, Collections.singleton(content));
            executeSubComponents(contentHandler, context, currentContext);
            XMLUtils.endElement(contentHandler, "content");
        }
    }
    
    @Override
    public Map<String, Object> getComponentDetailsForTree()
    {
        Map<String, Object> details = super.getComponentDetailsForTree();
        details.put("tag", ExtractionConstants.QUERY_COMPONENT_TAG);
        
        @SuppressWarnings("unchecked")
        Map<String, Object> data = (Map<String, Object>) details.get("data");
        
        StringJoiner columns = new StringJoiner(ExtractionConstants.STRING_COLLECTIONS_INPUT_DELIMITER);
        for (ExtractionColumn column : this.getColumns())
        {
            StringBuilder builder = new StringBuilder();
            builder.append(column.getFieldPath());
            if (null != column.getDisplayOptionalName())
            {
                builder.append(" (").append(column.getDisplayOptionalName()).append(")");
            }
            columns.add(builder);
        }
        data.put("columns", columns.toString());
        data.put("overrideColumns", this.overrideColumns());
        
        StringJoiner sorts = new StringJoiner(ExtractionConstants.STRING_COLLECTIONS_INPUT_DELIMITER);
        for (Map.Entry<String, SortOrder> sort : this.getSorts().entrySet())
        {
            StringBuilder builder = new StringBuilder();
            builder.append(sort.getKey()).append(" (");
            if (SortOrder.DESC.equals(sort.getValue()))
            {
                builder.append("DESC");
            }
            else
            {
                builder.append("ASC");
            }
            builder.append(")");
            sorts.add(builder);
        }
        data.put("sorts", sorts.toString());
        data.put("overrideSorts", this.overrideSorts());

        details.put("iconCls", "ametysicon-query-search");
        
        return details;
    }

    @Override
    protected String getDefaultTagName()
    {
        return "query";
    }
    
    @Override
    protected String getLogsPrefix()
    {
        return "Query component '" + _tagName + "': ";
    }
    
    /**
     * the referenced query's columns
     * @return <code>true</code> if the referenced query's columns are overridden on this component, <code>false</code> otherwise
     */
    public boolean overrideColumns()
    {
        return _overrideColumns;
    }
    
    /**
     * Set the boolean to override the referenced query's columns or not
     * @param overrideColumns <code>true</code> to override columns, <code>false</code> otherwise
     */
    public void setOverrideColumns(boolean overrideColumns)
    {
        _overrideColumns = overrideColumns;
    }

    /**
     * Retrieves the component columns
     * @return The component columns
     */
    public List<ExtractionColumn> getColumns()
    {
        return _columns;
    }

    /**
     * Add columns to the components. Do not manage optional columns' variables
     * @param fieldPaths Array of columns' field paths to add
     */
    public void addColumns(String... fieldPaths)
    {
        for (String fieldPath : fieldPaths)
        {
            ExtractionColumn column = new ExtractionColumn();
            column.setFieldPath(fieldPath);
            _columns.add(column);
        }
    }

    /**
     * Add an optional column to the component
     * @param fieldPath The column's field path
     * @param displayOptionalColumnName The name of the variable that manage the display of this optional column
     */
    public void addColumn(String fieldPath, String displayOptionalColumnName)
    {
        ExtractionColumn column = new ExtractionColumn();
        column.setFieldPath(fieldPath);
        column.setDisplayOptionalName(displayOptionalColumnName);
        _columns.add(column);
    }
    
    /**
     * the referenced query's sorts
     * @return <code>true</code> if the referenced query's sorts are overridden on this component, <code>false</code> otherwise
     */
    public boolean overrideSorts()
    {
        return _overrideSorts;
    }
    
    /**
     * Set the boolean to override the referenced query's sorts or not
     * @param overrideSorts <code>true</code> to override sorts, <code>false</code> otherwise
     */
    public void setOverrideSorts(boolean overrideSorts)
    {
        _overrideSorts = overrideSorts;
    }

    /**
     * Retrieves the component sorts
     * @return The component sorts
     */
    public Map<String, SortOrder> getSorts()
    {
        return _sortMap;
    }

    /**
     * Add sort to the component
     * @param filedPath Field on which apply sort 
     * @param order sort order to apply for field
     */
    public void addSort(String filedPath, SortOrder order)
    {
        _sortMap.put(filedPath, order);
    }
}
