/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.component;

import org.xml.sax.ContentHandler;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.extraction.execution.ExtractionExecutionContext;

/**
 * {@link ExtractionComponent} whose execution sometimes needs to be done in two times 
 * (in order to avoid calling {@link #execute(ContentHandler, ExtractionExecutionContext) execute} n times  for performance reasons):
 * <ol>
 * <li>by {@link #computeFirstLevelResults(ExtractionExecutionContext)} for returning all the first level content results;</li>
 * <li>by {@link #executeFor(ContentHandler, Iterable, ExtractionExecutionContext)} for SAXing with the given {@link ContentHandler} and the given iterable on contents, subset of the one returned by the first method.</li>
 * </ol>
 * Thus, instead of calling {@link #execute(ContentHandler, ExtractionExecutionContext) execute} n times,
 * {@link #computeFirstLevelResults(ExtractionExecutionContext) computeFirstLevelResults} is called one time 
 * and {@link #executeFor(ContentHandler, Iterable, ExtractionExecutionContext) executeFor} is called n times with a different handler and iterable.
 * <br>
 * <br><b>Important: </b>The implementations must ensure that calling <code>execute</code> is equivalent to successively call <code>computeFirstLevelResults</code> and then <code>executeFor</code> with that result.
 */
public interface TwoStepsExecutingExtractionComponent extends ExtractionComponent
{
    /**
     * Computes first level results for preparing multiple extraction executions
     * @param context context of the extraction component
     * @return The first level content results
     * @throws Exception if an error occurs
     */
    public Iterable<Content> computeFirstLevelResults(ExtractionExecutionContext context) throws Exception;
    
    /**
     * Execute the extraction of the component for the given first level results
     * @param contentHandler result document
     * @param subsetResults A subset of the first level content results previously returned by {@link #computeFirstLevelResults(ExtractionExecutionContext)}
     * @param context context of the extraction component
     * @throws Exception if an error occurs
     */
    public void executeFor(ContentHandler contentHandler, Iterable<Content> subsetResults, ExtractionExecutionContext context) throws Exception;
    
    @Override
    default void execute(ContentHandler contentHandler, ExtractionExecutionContext context) throws Exception
    {
        Iterable<Content> firstLevelResults = computeFirstLevelResults(context);
        executeFor(contentHandler, firstLevelResults, context);
    }
}
