/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.edition;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.impl.FileSource;
import org.apache.xml.serializer.OutputPropertiesFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import org.ametys.cms.repository.ContentDAO;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.execution.ExtractionDAO;
import org.ametys.plugins.extraction.rights.ExtractionAccessController;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * This client site element manages a button to create an extraction definition file
 */
public class EditExtractionClientSideElement extends StaticClientSideElement
{
    /** The Avalon role name */
    public static final String ROLE = EditExtractionClientSideElement.class.getName();
    
    private SourceResolver _sourceResolver;
    private ContentWorkflowHelper _contentWorkflowHelper;
    private I18nUtils _i18nUtils;
    private ContentDAO _contentDAO;
    private UserHelper _userHelper;
    private ExtractionDAO _extractionDAO;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _sourceResolver = (SourceResolver) serviceManager.lookup(SourceResolver.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) serviceManager.lookup(ContentWorkflowHelper.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
        _contentDAO = (ContentDAO) serviceManager.lookup(ContentDAO.ROLE);
        _userHelper = (UserHelper) serviceManager.lookup(UserHelper.ROLE);
        _extractionDAO = (ExtractionDAO) serviceManager.lookup(ExtractionDAO.ROLE);
        
    }

    /**
     * Creates an extraction definition file.
     * @param relativeDefinitionFilePath The path of the extraction definition file to create. This path has to be relative to the base definition directory.
     * @param language the language used to create the description
     * @return Map containing success boolean and the created extraction informations, or error codes if one occurs
     * @throws Exception if an error occurs
     */
    @Callable (rights = ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID)
    public Map<String, Object> createExtraction(String relativeDefinitionFilePath, String language) throws Exception
    {
        // Create extraction definitions directory
        Source definitionsSrc = _sourceResolver.resolveURI(ExtractionConstants.DEFINITIONS_DIR);
        File definitionsDir = ((FileSource) definitionsSrc).getFile();
        definitionsDir.mkdirs();
        
        String absoluteDefinitionFilePath = ExtractionConstants.DEFINITIONS_DIR + relativeDefinitionFilePath;
        Source definitionSrc = _sourceResolver.resolveURI(absoluteDefinitionFilePath);
        File definitionFile = ((FileSource) definitionSrc).getFile();

        if (definitionFile.exists())
        {
            getLogger().error("A definition file already exists at path '{}'", relativeDefinitionFilePath);
            return Map.of(
                    "success", false,
                    "error", "already-exists");
        }
        
        try (OutputStream os = Files.newOutputStream(Paths.get(definitionFile.getAbsolutePath())))
        {
            // Create the description content
            String extractionName = _getExtractionNameFromFileName(definitionFile.getName());
            I18nizableText descriptionTitle = new I18nizableText(ExtractionConstants.PLUGIN_NAME, ExtractionConstants.DESCRIPTION_DEFAULT_TITLE_KEY, Collections.singletonList(extractionName));
            
            Map<String, Object> contentInfos = _contentWorkflowHelper.createContent(
                    ExtractionConstants.DESCRIPTION_CONTENT_WORKFLOW_NAME,
                    ExtractionConstants.DESCRIPTION_CONTENT_INITIAL_ACTION_ID,
                    extractionName,
                    _i18nUtils.translate(descriptionTitle, language),
                    new String[] {ExtractionConstants.DESCRIPTION_CONTENT_TYPE_ID},
                    new String[0],
                    language
                );
            String descriptionId = (String) contentInfos.get("contentId");
            
            // Create a transformer for saving sax into a file
            TransformerHandler handler = ((SAXTransformerFactory) TransformerFactory.newInstance()).newTransformerHandler();
            
            StreamResult result = new StreamResult(os);
            handler.setResult(result);

            // create the format of result
            Properties format = new Properties();
            format.put(OutputKeys.METHOD, "xml");
            format.put(OutputKeys.INDENT, "yes");
            format.put(OutputKeys.ENCODING, "UTF-8");
            format.put(OutputPropertiesFactory.S_KEY_INDENT_AMOUNT, "4");
            handler.getTransformer().setOutputProperties(format);
            handler.startDocument();

            // sax skeleton
            XMLUtils.startElement(handler, ExtractionConstants.EXTRACTION_TAG);
            
            // Generate SAX events for the extraction's description
            AttributesImpl attributes = new AttributesImpl();
            attributes.addCDATAAttribute(ExtractionConstants.DESCRIPTION_IDENTIFIER_ATTRIBUTE_NAME, descriptionId);
            XMLUtils.createElement(handler, ExtractionConstants.DESCRIPTION_TAG, attributes);
                        
            // Generate SAX events for the extraction's author
            UserIdentity author = _currentUserProvider.getUser();
            _userHelper.saxUserIdentity(author, handler, ExtractionConstants.AUTHOR_TAG);
            
            XMLUtils.endElement(handler, ExtractionConstants.EXTRACTION_TAG);
            
            handler.endDocument();
        
            return Map.of(
                    "success", true, 
                    "path", relativeDefinitionFilePath,
                    "name", definitionFile.getName(),
                    "descriptionId", descriptionId,
                    "author", _userHelper.user2json(author),
                    "canRead", true,
                    "canWrite", true,
                    "canAssignRights", true);
        }
        catch (Exception e)
        {
            getLogger().error("Error when trying to create the extraction definition file '{}'", relativeDefinitionFilePath, e);
            return Map.of(
                    "success", false,
                    "error", "other-error");
        }
    }
    
    private String _getExtractionNameFromFileName(String fileName)
    {
        return fileName.contains(".") ? fileName.substring(0, fileName.lastIndexOf(".")) : fileName;
    }
    
    /**
     * Adds a description to an extraction.
     * @param definitionFileName The extraction definition file name
     * @param descriptionId the identifier of the description 
     * @return Map containing success boolean and error codes if one occurs
     * @throws Exception if an error occurs
     */
    @Callable (rights = ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID)
    public Map<String, Object> addDescription(String definitionFileName, String descriptionId) throws Exception
    {
        return _modifyDefinitionFile(definitionFileName, descriptionId, this::_insertDescriptionInDocument);
    }

    private void _insertDescriptionInDocument(Document document, String descriptionId)
    {
        Element extractionRoot = document.getDocumentElement();
        
        // Delete the current description node if exists
        _getElements(document.getDocumentElement(), ExtractionConstants.DESCRIPTION_TAG)
            .forEach(extractionRoot::removeChild);

        // Insert the description in the extraction root node
        Element description = document.createElement(ExtractionConstants.DESCRIPTION_TAG);
        description.setAttribute(ExtractionConstants.DESCRIPTION_IDENTIFIER_ATTRIBUTE_NAME, descriptionId);
        extractionRoot.insertBefore(description, extractionRoot.getFirstChild());
    }
    
    private <T> Map<String, Object> _modifyDefinitionFile(String definitionFileName, T dataToModify, BiConsumer<Document, T> modifyingConsumer) throws Exception
    {
        String definitionFilePath = ExtractionConstants.DEFINITIONS_DIR + definitionFileName;
        Source definitionSrc = _sourceResolver.resolveURI(definitionFilePath);
        File definitionFile = ((FileSource) definitionSrc).getFile();
        
        if (!definitionFile.exists())
        {
            getLogger().error("Error while adding a description to the extraction '{}': this definition file doesn't exist.", definitionFileName);
            return Map.of(
                    "success", false,
                    "error", "unexisting");
        }
        
        String tmpFilePath = ExtractionConstants.DEFINITIONS_DIR + definitionFileName + ".tmp";
        Source tmpSrc = _sourceResolver.resolveURI(tmpFilePath);
        File tmpFile = ((FileSource) tmpSrc).getFile();
        
        try (OutputStream os = Files.newOutputStream(Paths.get(tmpFile.getAbsolutePath())))
        {
            // Parse existing definition file
            Document document = _parseDefinitionFile(definitionFile);
            
            // Apply the modification
            modifyingConsumer.accept(document, dataToModify);

            // Write the updated definition file
            DOMSource source = new DOMSource(document);
            Transformer transformer = TransformerFactory.newInstance().newTransformer();
            StreamResult result = new StreamResult(os);
            transformer.transform(source, result);
            
            Files.copy(tmpFile.toPath(), definitionFile.toPath(), StandardCopyOption.REPLACE_EXISTING);

            return Map.of("success", true);
        }
        catch (Exception e)
        {
            getLogger().error("Error when trying to modify the extraction '{}'", definitionFileName, e);
            return Map.of(
                    "success", false,
                    "error", "other-error");
        }
        finally
        {
            // delete the temporary file to keep the original one
            _deleteTemporaryFile(definitionFileName, tmpFile.toPath());
        }
    }
    
    private void _deleteTemporaryFile(String definitionFileName, Path temporaryFilePath)
    {
        try
        {
            Files.deleteIfExists(temporaryFilePath);
        }
        catch (IOException e)
        {
            getLogger().error("Error when deleting the temporary file for '{}'", definitionFileName, e);
        }
    }

    /**
     * Renames an extraction definition file.
     * @param relativeOldFilePath The extraction definition old file path, relative to the base definitions directory
     * @param newFileName The extraction definition new file name
     * @return Map containing success boolean and error codes if one occurs
     * @throws Exception if an error occurs
     */
    @Callable (rights = ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID)
    public Map<String, Object> renameExtraction(String relativeOldFilePath, String newFileName) throws Exception
    {
        String fileUri = ExtractionConstants.DEFINITIONS_DIR + relativeOldFilePath;
        FileSource file = (FileSource) _sourceResolver.resolveURI(fileUri);
        String relativeParentPath = StringUtils.removeEnd(relativeOldFilePath, file.getName());
        String relativeNewFilePath = relativeParentPath + newFileName;
        return _extractionDAO.moveOrRenameExtractionDefinitionFile(relativeOldFilePath, relativeNewFilePath);
    }
    
    /**
     * Deletes an extraction definition file.
     * @param definitionFileName The extraction definition file to delete
     * @return <code>true</code> if extraction deletion succeed, <code>false</code> otherwise
     * @throws Exception if an error occurs
     */
    @Callable (rights = ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID)
    public boolean deleteExtraction(String definitionFileName) throws Exception
    {
        String filePath = ExtractionConstants.DEFINITIONS_DIR + definitionFileName;
        Source source = _sourceResolver.resolveURI(filePath);
        File file = ((FileSource) source).getFile();
        
        if (!file.exists())
        {
            throw new IllegalArgumentException("Error while deleting '" + definitionFileName + "': this definition file doesn't exist.");
        }
        
        try
        {
            // Get description identifiers
            Document document = _parseDefinitionFile(file);
            List<String> descriptionIds = _getElements(document.getDocumentElement(), ExtractionConstants.DESCRIPTION_TAG)
                .map(description -> description.getAttribute(ExtractionConstants.DESCRIPTION_IDENTIFIER_ATTRIBUTE_NAME))
                .collect(Collectors.toList());
            
            // Delete descriptions contents
            _contentDAO.deleteContents(descriptionIds, ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID);

            Files.deleteIfExists(file.toPath());

            String context = ExtractionAccessController.ROOT_CONTEXT + "/" + definitionFileName;
            _extractionDAO.deleteRights(context);
        }
        catch (IOException e)
        {
            if (getLogger().isErrorEnabled())
            {
                getLogger().error("Error while deleting definition file '" + definitionFileName + "'.", e);
            }
            throw e;
        }
        
        return true;
    }
    
    private Document _parseDefinitionFile(File definitionFile) throws Exception
    {
        DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
        DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
        return documentBuilder.parse(definitionFile);
    }
    
    private Stream<Element> _getElements(Element extractionRoot, String nodeTagName)
    {
        NodeList nodeList = extractionRoot.getElementsByTagName(nodeTagName);
        return IntStream.range(0, nodeList.getLength())
                .mapToObj(nodeList::item)
                .map(Element.class::cast);
    }
    
}
