/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.FileUtils;

import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * This client site element manages a button to delete extraction result folders and files
 */
public class DeleteExtractionResultFilesClientSideElement extends StaticClientSideElement
{
    /**
     * Deletes multiple result folders and result files.
     * @param paths The paths of the result folders and files to delete
     * @return The result map
     */
    @Callable(rights = ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID)
    public Map<String, Object> deleteResults(List<String> paths)
    {
        Map<String, Object> result = new HashMap<>();
        List<String> deleted = new ArrayList<>();
        List<String> notDeleted = new ArrayList<>();
        List<String> doNotExist = new ArrayList<>();
        
        Path rootResults = AmetysHomeHelper.getAmetysHomeData().toPath().resolve(ExtractionConstants.RESULT_EXTRACTION_DIR_NAME);
        for (String path : paths)
        {
            _tryToDelete(path, rootResults, deleted, notDeleted, doNotExist);
        }
        
        result.put("deleted", deleted);
        result.put("errors", notDeleted);
        result.put("not-exist", doNotExist);
        return result;
    }
    
    private static void _tryToDelete(String path, Path rootResults, List<String> deleted, List<String> notDeleted, List<String> doNotExist)
    {
        Path fileOrFolder = rootResults.resolve(Paths.get(path));
        if (Files.exists(fileOrFolder))
        {
            try
            {
                _delete(fileOrFolder);
                deleted.add(path);
            }
            catch (IOException e)
            {
                notDeleted.add(path);
            }
        }
        else
        {
            doNotExist.add(path);
        }
    }
    
    private static void _delete(Path fileOrFolder) throws IOException
    {
        if (Files.isDirectory(fileOrFolder))
        {
            FileUtils.deleteDirectory(fileOrFolder.toFile());
        }
        else
        {
            Files.delete(fileOrFolder);
        }
    }
}

