/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.quartz.JobKey;
import org.quartz.SchedulerException;

import org.ametys.core.schedule.Runnable;
import org.ametys.core.schedule.Runnable.FireProcess;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.core.impl.schedule.DefaultRunnable;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for {@link Runnable} executing an extraction
 */
public class ExecuteExtractionRunnableHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** Avalon Role */
    public static final String ROLE = ExecuteExtractionRunnableHelper.class.getName();
    
    private static final String __RUNNABLE_PREFIX = "org.ametys.plugins.extraction.execution.ExecuteExtractionRunnable$";
    
    private JSONUtils _jsonUtils;
    private CurrentUserProvider _currentUserProvider;
    private Scheduler _scheduler;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _scheduler = (Scheduler) manager.lookup(Scheduler.ROLE);
    }
   
    /**
     * Execute the extraction
     * @param definitionFilePath The extraction definition file path
     * @param variables clauses variables and optional columns
     * @param recipient An email will be sent at this address when the extraction is complete
     * @param pipelineId The id of the extraction pipeline
     * @return a Map with error if one occurs
     */
    public Map<String, Object> executeExtraction(String definitionFilePath, Map<String, Object> variables, String recipient, String pipelineId)
    {
        try
        {
            Map<String, Object> jobParams = new HashMap<>();
            jobParams.put(ExecuteExtractionSchedulable.DEFINITION_FILE_PATH_KEY, definitionFilePath);
            jobParams.put(ExecuteExtractionSchedulable.VARIABLES_KEY, _jsonUtils.convertObjectToJson(variables));
            jobParams.put(ExecuteExtractionSchedulable.RECIPIENT_KEY, recipient);
            jobParams.put(ExecuteExtractionSchedulable.PIPELINE_KEY, pipelineId);
            
            Runnable executeExtractionRunnable = new DefaultRunnable(__RUNNABLE_PREFIX + definitionFilePath,
                    new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_SCHEDULABLE_LABEL", Collections.singletonList(definitionFilePath)),
                    new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_SCHEDULABLE_DESCRIPTION", Collections.singletonList(definitionFilePath)),
                    FireProcess.NOW,
                    null /* cron*/,
                    "org.ametys.plugins.extraction.execution.ExecuteExtractionSchedulable",
                    true /* removable */,
                    true /* modifiable */,
                    false /* deactivatable */,
                    null /* misfire policy */,
                    false /* isVolatile */,
                    _currentUserProvider.getUser(),
                    jobParams
                );
            
            JobKey jobKey = new JobKey(executeExtractionRunnable.getId(), Scheduler.JOB_GROUP);
            if (_scheduler.getScheduler().checkExists(jobKey))
            {
                _scheduler.getScheduler().deleteJob(jobKey);
            }
            _scheduler.scheduleJob(executeExtractionRunnable);
            
            getLogger().info("Scheduled extraction execution of " + definitionFilePath);
            return Map.of("definitionFilePath", definitionFilePath);
        }
        catch (SchedulerException e)
        {
            getLogger().error("An error occured when trying to schedule the extraction execution of " + definitionFilePath, e);
            return Map.of("error", "scheduler-error");
        }
    }
    
    /**
     * Remove the scheduled task of extraction execution of the given file
     * @param definitionFilePath The extraction definition file path
     */
    public void deleteRunnableJob(String definitionFilePath)
    {
        try
        {
            String runnableId = __RUNNABLE_PREFIX + definitionFilePath;
            // Remove job corresponding to the runnable id (existing only if the extraction has been launched via the extraction execution tool) 
            JobKey jobKey = new JobKey(runnableId, Scheduler.JOB_GROUP);
            if (_scheduler.getScheduler().checkExists(jobKey))
            {
                _scheduler.getScheduler().deleteJob(jobKey);
            }
        }
        catch (SchedulerException e)
        {
            getLogger().error("An error occured when trying to remove the scheduled task of extraction execution of " + definitionFilePath, e);
        }
    }
}
