/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.io.File;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.impl.FileSource;

import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.component.ExtractionComponent;
import org.ametys.plugins.extraction.edition.SaveExtractionHelper;
import org.ametys.plugins.extraction.execution.Extraction.ClausesVariable;
import org.ametys.runtime.i18n.I18nizableText;

/**
 *  Tool client side element for extraction edition tool
 */
public class ExtractionDetailsToolElement extends StaticClientSideElement
{
    private ExtractionDefinitionReader _reader;
    private SourceResolver _sourceResolver;
    private SaveExtractionHelper _saveHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _reader = (ExtractionDefinitionReader) serviceManager.lookup(ExtractionDefinitionReader.ROLE);
        _sourceResolver = (SourceResolver) serviceManager.lookup(SourceResolver.ROLE);
        _saveHelper = (SaveExtractionHelper) serviceManager.lookup(SaveExtractionHelper.ROLE);
    }
    
    /**
     * Retrieve extraction definition details.
     * @param relativeDefinitionFilePath The extraction's definition file path, relative to the base definitions directory
     * @return a <code>Map</code> containing the extraction definition details
     * @throws Exception if an error occurs
     */
    @Callable (rights = ExtractionConstants.EXECUTE_EXTRACTION_RIGHT_ID)
    public Map<String, Object> getExtractionDefinitionDetails(String relativeDefinitionFilePath) throws Exception
    {
        Map<String, Object> extractionDefinitionDetails = new LinkedHashMap<>();

        String absoluteDefinitionFilePath = ExtractionConstants.DEFINITIONS_DIR + relativeDefinitionFilePath;
        Source src = _sourceResolver.resolveURI(absoluteDefinitionFilePath);
        File file = ((FileSource) src).getFile();
        
        if (!file.exists())
        {
            if (getLogger().isWarnEnabled())
            {
                getLogger().warn("The file " + relativeDefinitionFilePath + " does not exist.");
            }
            
            extractionDefinitionDetails.put("success", false);
            extractionDefinitionDetails.put("file-error", "unexisting");
            return extractionDefinitionDetails;
        }
        
        Extraction extraction = _reader.readExtractionDefinitionFile(file);
        
        List<Map<String, Object>> extractionNodes = new ArrayList<>();
        
        Map<String, Object> clausesVariablesNode = _getClausesVariablesNode(extraction);
        extractionNodes.add(clausesVariablesNode);
        
        Map<String, Object> optionalColumnsNode = _getOptionalColumnsNode(extraction);
        extractionNodes.add(optionalColumnsNode);
        
        List<Map<String, Object>> componentsNodes = _getComponentNodes(extraction.getExtractionComponents());
        if (!componentsNodes.isEmpty())
        {
            extractionNodes.addAll(componentsNodes);
        }
        
        extractionDefinitionDetails.put("success", true);
        extractionDefinitionDetails.put("descriptionId", extraction.getDescriptionId());
        extractionDefinitionDetails.put("children", extractionNodes);
        return extractionDefinitionDetails;
    }

    private Map<String, Object> _getClausesVariablesNode(Extraction extraction)
    {
        Map<String, Object> clausesVariablesNode = new LinkedHashMap<>();
        List<ClausesVariable> clausesVariables = extraction.getClausesVariables();
            
        List<Map<String, Object>> variables = new ArrayList<>();
        for (ClausesVariable clausesVariable : clausesVariables)
        {
            Map<String, Object> clausesVariableData = new LinkedHashMap<>();

            clausesVariableData.put("name", clausesVariable.name());
            clausesVariableData.put("type", clausesVariable.type().getStringValue());
            clausesVariableData.put("contentTypeIds", clausesVariable.contentTypeIds());
            clausesVariable.searchModelId()
                           .ifPresent(searchModelId -> clausesVariableData.put("searchModelId", searchModelId));
            clausesVariable.solrRequest()
                           .ifPresent(solrRequest -> clausesVariableData.put("solrRequest", solrRequest));
            
            variables.add(clausesVariableData);
        }
        
        Map<String, Object> clausesVariablesData = new LinkedHashMap<>();
        clausesVariablesData.put("variables", variables);
        
        clausesVariablesNode.put("text", new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_TREE_CLAUSES_VARIABLES_NODE_TEXT"));
        clausesVariablesNode.put("data", clausesVariablesData);
        clausesVariablesNode.put("leaf", true);
        clausesVariablesNode.put("tag", ExtractionConstants.CLAUSES_VARIABLES_TAG);
        clausesVariablesNode.put("iconCls", "ametysicon-symbol-x");
        
        return clausesVariablesNode;
    }

    private Map<String, Object> _getOptionalColumnsNode(Extraction extraction)
    {
        Map<String, Object> optionalColumnsNode = new LinkedHashMap<>();
        List<String> optionalColumns = extraction.getDisplayOptionalColumnsNames();
        
        Map<String, Object> optionalColumnsData = new LinkedHashMap<>();
        optionalColumnsData.put("names", optionalColumns);
        
        optionalColumnsNode.put("text", new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_TREE_OPTIONAL_COLUMNS_NODE_TEXT"));
        optionalColumnsNode.put("data", optionalColumnsData);
        optionalColumnsNode.put("leaf", true);
        optionalColumnsNode.put("tag", ExtractionConstants.OPTIONAL_COLUMNS_TAG);
        optionalColumnsNode.put("iconCls", "ametysicon-table28");

        return optionalColumnsNode;
    }

    private List<Map<String, Object>> _getComponentNodes(List<ExtractionComponent> components)
    {
        List<Map<String, Object>> componentNodes = new ArrayList<>();
        for (ExtractionComponent component : components)
        {
            Map<String, Object> componentNode = component.getComponentDetailsForTree();
            if (component.getSubComponents().isEmpty())
            {
                componentNode.put("leaf", true);
            }
            else
            {
                
                componentNode.put("children", _getComponentNodes(component.getSubComponents()));
            }
            componentNodes.add(componentNode);
        }
        return componentNodes;
    }
    
    /**
     * Saves modifications on extraction. Creates the definition file if it doesn't exist
     * @param definitionFileName The extraction definition file name
     * @param extraction A <code>Map</code> containing definition information
     * @return <code>true</code> if extraction saving succeed, <code>false</code> otherwise
     * @throws Exception if an error occurs
     */
    @Callable (rights = ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID)
    public boolean saveExtraction(String definitionFileName, Map<String, Object> extraction) throws Exception
    {
        return _saveHelper.saveExtraction(definitionFileName, extraction);
    }
}
