/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution.pipeline.impl;

import java.io.OutputStream;
import java.util.Map;
import java.util.Properties;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.TransformerException;
import javax.xml.transform.sax.SAXResult;
import javax.xml.transform.sax.TransformerHandler;

import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.xml.sax.ContentHandlerProxy;
import org.apache.xml.serializer.OutputPropertiesFactory;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.plugins.extraction.execution.pipeline.PipelineSerializerModel;
import org.ametys.plugins.extraction.execution.pipeline.Pipelines;

/**
 * Model for XML pipeline serializers
 */
public class XmlPipelineSerializerModel implements PipelineSerializerModel
{
    @Override
    public String getDefaultFileExtension()
    {
        return "xml";
    }
    
    @Override
    public PipelineSerializer newSerializer(TransformerHandler handler, OutputStream out, Map<String, String> outputParameters)
    {
        return new XmlPipelineSerializer(handler, out, outputParameters);
    }
    
    private static final class XmlPipelineSerializer extends AbstractSerializerImpl
    {
        XmlPipelineSerializer(TransformerHandler handler, OutputStream out, Map<String, String> outputParameters)
        {
            super(handler, out);
            // Strip whitespaces to avoid rich texts indentation if indent param is set to "no"
            _wrapHandlerToStripEOL();
            _setOutputProps(outputParameters);
        }
        
        private void _wrapHandlerToStripEOL()
        {
            try
            {
                TransformerHandler transformerHandler = Pipelines.getSaxTransformerFactory().newTransformerHandler();
                ContentHandler stripWhitespacesContentHandler = new StripEOLContentHandler(transformerHandler);
                _handler.setResult(new SAXResult(stripWhitespacesContentHandler));
                _handler = transformerHandler;
            }
            catch (TransformerException e)
            {
                throw new RuntimeException("Unable to strip EOL", e);
            }
        }
        
        private void _setOutputProps(Map<String, String> outputParameters)
        {
            Properties properties = new Properties();
            properties.put(OutputKeys.METHOD, outputParameters.getOrDefault("method", "xml"));
            properties.put(OutputKeys.ENCODING, outputParameters.getOrDefault("encoding", "UTF-8"));
            properties.put(OutputKeys.INDENT, outputParameters.getOrDefault("indent", "yes"));
            properties.put(OutputPropertiesFactory.S_KEY_INDENT_AMOUNT, outputParameters.getOrDefault("indent-amount", "4"));
            _handler.getTransformer().setOutputProperties(properties);
        }
        
        @Override
        public void serialize() throws Exception
        {
            // Nothing
        }
    }
    
    private static class StripEOLContentHandler extends ContentHandlerProxy
    {
        StripEOLContentHandler(ContentHandler contentHandler)
        {
            super(contentHandler);
        }
        
        @Override
        public void characters(char[] ch, int start, int len) throws SAXException
        {
            String data  =  new String(ch, start, len).replaceAll("[\\r\\n]", StringUtils.EMPTY);
            super.characters(data.toCharArray(), 0, data.length());
        }
    }
}
