/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.io.File;
import java.io.OutputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.LocaleUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.impl.FileSource;
import org.xml.sax.ContentHandler;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.component.ExtractionComponent;
import org.ametys.plugins.extraction.component.TwoStepsExecutingExtractionComponent;
import org.ametys.plugins.extraction.execution.Extraction.ClausesVariable;
import org.ametys.plugins.extraction.execution.Extraction.ClausesVariableType;
import org.ametys.plugins.extraction.execution.pipeline.Pipeline;
import org.ametys.plugins.extraction.execution.pipeline.PipelineDescriptor;
import org.ametys.plugins.extraction.execution.pipeline.Pipelines;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.util.AmetysHomeHelper;

import com.google.common.base.Predicates;

/**
 * Extracts query results form a XML definition file
 */
public class ExtractionExecutor extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Avalon role. */
    public static final String ROLE = ExtractionExecutor.class.getName();
    
    private RightManager _rightManager;
    private ExtractionDefinitionReader _reader;
    private CurrentUserProvider _currentUserProvider;
    private AmetysObjectResolver _resolver;
    private ContentTypesHelper _contentTypesHelper;
    private SourceResolver _sourceResolver;
    private PathResolver _resultPathResolver;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
        _reader = (ExtractionDefinitionReader) serviceManager.lookup(ExtractionDefinitionReader.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _contentTypesHelper = (ContentTypesHelper) serviceManager.lookup(ContentTypesHelper.ROLE);
        _sourceResolver = (SourceResolver) serviceManager.lookup(SourceResolver.ROLE);
        _resultPathResolver = (PathResolver) serviceManager.lookup(PathResolver.ROLE);
    }
    
    /**
     * Executes the given extraction
     * @param relativeFilePath The path of the extraction file to execute
     * @param defaultResultFileName The default file name for the result (it can be unused, for instance if resultSubFolder is a file, i.e. contains a '.' in its last element)
     * @param lang The language
     * @param parameters The parameters
     * @param pipeline The execution pipeline
     * @return The set of the result files path
     * @throws Exception if an errors occurred
     */
    public Set<Path> execute(String relativeFilePath, String defaultResultFileName, String lang, Map<String, Object> parameters, PipelineDescriptor pipeline) throws Exception
    {
        return _execute(relativeFilePath, defaultResultFileName, null, lang, parameters, pipeline);
    }
    
    /**
     * Executes the given extraction
     * @param relativeFilePath The path of the extraction file to execute
     * @param resultOutputStream The result output stream
     * @param lang The language
     * @param parameters The parameters
     * @param pipeline The execution pipeline
     * @throws Exception if an errors occurred
     */
    public void execute(String relativeFilePath, OutputStream resultOutputStream, String lang, Map<String, Object> parameters, PipelineDescriptor pipeline) throws Exception
    {
        _execute(relativeFilePath, null, resultOutputStream, lang, parameters, pipeline);
    }
    
    private Set<Path> _execute(
            String relativeFilePath,
            String defaultResultFileName,
            OutputStream resultOutputStream,
            String lang,
            Map<String, Object> parameters,
            PipelineDescriptor pipeline) throws Exception
    {
        _checkRights();
        
        Pair<Extraction, String> extractionAndName = _getExtraction(relativeFilePath);
        Extraction extraction = extractionAndName.getLeft();
        String definitionFilename = extractionAndName.getRight();
        
        AttributesImpl attributes = _getAttrs(definitionFilename);

        ExtractionExecutionContext context = _getContext(extraction, lang, parameters);
        
        if (resultOutputStream != null)
        {
            _doExecuteForPathWithNoVar(attributes, extraction, context, resultOutputStream, pipeline);
            return Set.of();
        }
        else
        {
            return _doExecute(attributes, extraction, relativeFilePath, context, defaultResultFileName, pipeline);
        }
    }
    
    private void _checkRights()
    {
        if (_rightManager.hasRight(_currentUserProvider.getUser(), ExtractionConstants.EXECUTE_EXTRACTION_RIGHT_ID, "/admin") != RightResult.RIGHT_ALLOW)
        {
            String errorMessage = "User " + _currentUserProvider.getUser() + " tried to execute extraction with no sufficient rights";
            getLogger().error(errorMessage);
            throw new IllegalStateException(errorMessage);
        }
    }
    
    private Pair<Extraction, String> _getExtraction(String relativeFilePath) throws Exception
    {
        String absoluteFilePath = ExtractionConstants.DEFINITIONS_DIR + relativeFilePath;
        Source src = _sourceResolver.resolveURI(absoluteFilePath);

        try
        {
            File file = ((FileSource) src).getFile();
            
            if (!file.exists())
            {
                throw new IllegalArgumentException("The file " + relativeFilePath + " does not exist.");
            }
            
            Extraction extraction = _reader.readExtractionDefinitionFile(file);
            return Pair.of(extraction, file.getName());
        }
        catch (Exception e)
        {
            throw new IllegalStateException("An unexpected error occured.", e);
        }
        finally
        {
            _sourceResolver.release(src);
        }
    }
    
    private AttributesImpl _getAttrs(String definitionFilename)
    {
        AttributesImpl attributes = new AttributesImpl();
        attributes.addCDATAAttribute("user", _currentUserProvider.getUser().getLogin());
        attributes.addCDATAAttribute("date", ZonedDateTime.now().format(DateUtils.getISODateTimeFormatter()));
        attributes.addCDATAAttribute("name", definitionFilename);
        return attributes;
    }
    
    private ExtractionExecutionContext _getContext(Extraction extraction, String lang, Map<String, Object> parameters)
    {
        ExtractionExecutionContext context = new ExtractionExecutionContext();
        if (StringUtils.isNotEmpty(lang))
        {
            context.setDefaultLocale(LocaleUtils.toLocale(lang));
        }
        context.setDisplayOptionalColumns(_getDisplayOptionalColumns(extraction.getDisplayOptionalColumnsNames(), parameters));
        context.setClausesVariablesValues(_getClausesVariablesValues(extraction.getClausesVariables(), parameters));
        return context;
    }

    Map<String, Boolean> _getDisplayOptionalColumns(List<String> displayOptionalColumnsNames, Map<String, Object> parameters)
    {
        Map<String, Boolean> result = new HashMap<>();
        for (String name : displayOptionalColumnsNames)
        {
            Boolean value = _getDipslayOptionalColumn(name, parameters);
            if (value == null)
            {
                throw new IllegalArgumentException("Extraction - There is a variable named '" + name + "' but there is no corresponding value");
            }
            else
            {
                result.put(name, value);
            }
        }
        return result;
    }
    
    private Boolean _getDipslayOptionalColumn(String optionalColumnName, Map<String, Object> parameters)
    {
        Object value = parameters.get(optionalColumnName);
        
        if (value instanceof Boolean)
        {
            return (Boolean) value;
        }
        else if (value instanceof String)
        {
            return Boolean.valueOf((String) value);
        }
        else
        {
            return null;
        }
    }

    private Map<ClausesVariable, List<String>> _getClausesVariablesValues(List<ClausesVariable> clausesVariables, Map<String, Object> parameters)
    {
        Map<ClausesVariable, List<String>> result = new HashMap<>();
        for (ClausesVariable clausesVariable : clausesVariables)
        {
            if (ClausesVariableType.SELECT_CONTENTS.equals(clausesVariable.type()))
            {
                // Only one content type for SELECT_CONTENTS clauses variables
                Optional<String> contentTypeId = clausesVariable.contentTypeIds()
                                                                .stream()
                                                                .findFirst();
                
                @SuppressWarnings("unchecked")
                List<String> contentIds = (List<String>) parameters.get(clausesVariable.name());
                if (contentIds == null || contentIds.isEmpty())
                {
                    throw new IllegalArgumentException("Extraction - There is a variable named '" + clausesVariable.name() + "' but there is no corresponding value");
                }
                
                // If a content type is provided, check all contents' types
                if (contentTypeId.isPresent())
                {
                    for (String contentId : contentIds)
                    {
                        Content content = _resolver.resolveById(contentId);
                        if (!_contentTypesHelper.isInstanceOf(content, contentTypeId.get()))
                        {
                            throw new IllegalArgumentException("Extraction - content '" + contentId + "' is not an instance of content type '" + contentTypeId.get() + "', defined by the variable named '" + clausesVariable.name() + "'");
                        }
                    }
                }
                
                result.put(clausesVariable, contentIds);
            }
            else
            {
                String solrRequest = (String) parameters.get(clausesVariable.name());
                if (StringUtils.isEmpty(solrRequest))
                {
                    throw new IllegalArgumentException("Extraction - There is a variable named '" + clausesVariable.name() + "' but there is no corresponding value");
                }
                
                result.put(clausesVariable, List.of(solrRequest));
            }
        }
        return result;
    }
    
    private Set<Path> _doExecute(AttributesImpl attributes, Extraction extraction, String extractionFilePath, ExtractionExecutionContext context, String defaultResultFileName, PipelineDescriptor pipeline) throws Exception
    {
        String unresolvedPath = pipeline.getResultSubfolder();
        Path basePath = Paths.get(AmetysHomeHelper.getAmetysHomeData().toPath().toString(), ExtractionConstants.RESULT_EXTRACTION_DIR_NAME);
       
        if (_resultPathResolver.hasVariable(unresolvedPath))
        {
            List<ExtractionComponent> extractionComponents = extraction.getExtractionComponents();
            boolean allAreNotTwoStepsComponent = extractionComponents.stream()
                    .filter(Predicates.not(TwoStepsExecutingExtractionComponent.class::isInstance))
                    .findFirst()
                    .isPresent();
            if (allAreNotTwoStepsComponent)
            {
                throw new IllegalArgumentException("The extraction " + extractionFilePath + " has an invalid component at first level which does not support a subfolder containing variables.");
            }
            
            List<TwoStepsExecutingExtractionComponent> components = extractionComponents.stream()
                    .map(TwoStepsExecutingExtractionComponent.class::cast)
                    .collect(Collectors.toList());
            return _doExecuteForPathWithVar(attributes, extraction, context, components, basePath, unresolvedPath, pipeline);
        }
        else
        {
            Path fileOrFolderPath = _resultPathResolver.resolvePath(unresolvedPath, null, extraction, basePath).keySet().iterator().next();
            Path filePath = _filePathWhenNoVar(fileOrFolderPath, defaultResultFileName);
            try (OutputStream fileOs = Pipelines.getOutputStream(filePath);)
            {
                _doExecuteForPathWithNoVar(attributes, extraction, context, fileOs, pipeline);
                return Set.of(filePath);
            }
        }
    }
    
    private Path _filePathWhenNoVar(Path fileOrFolderPath, String defaultFileName)
    {
        Path fileName = fileOrFolderPath.getFileName();
        if (fileName.toString().contains("."))
        {
            // is already a file, do not use default file name
            return fileOrFolderPath;
        }
        return Paths.get(fileOrFolderPath.toString(), defaultFileName);
    }
    
    private void _doExecuteForPathWithNoVar(AttributesImpl attributes, Extraction extraction, ExtractionExecutionContext context, OutputStream outputStream, PipelineDescriptor pipelineDescriptor) throws Exception
    {
        try (Pipeline pipeline = pipelineDescriptor.newPipeline(outputStream);)
        {
            ContentHandler contentHandler = pipeline.getHandler();
            _noVarExecute(contentHandler, attributes, extraction, context);
            pipeline.serialize();
        }
    }
    
    private void _noVarExecute(ContentHandler contentHandler, AttributesImpl attributes, Extraction extraction, ExtractionExecutionContext context) throws Exception
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, ExtractionConstants.RESULT_EXTRACTION_TAG, attributes);
        
        for (ExtractionComponent component : extraction.getExtractionComponents())
        {
            component.prepareComponentExecution(context);
            component.execute(contentHandler, context);
        }
        
        XMLUtils.endElement(contentHandler, ExtractionConstants.RESULT_EXTRACTION_TAG);
        contentHandler.endDocument();
    }
    
    private Set<Path> _doExecuteForPathWithVar(AttributesImpl attributes, Extraction extraction, ExtractionExecutionContext context, List<TwoStepsExecutingExtractionComponent> components, Path basePath, String unresolvedPath, PipelineDescriptor pipelineDescriptor) throws Exception
    {
        Map<TwoStepsExecutingExtractionComponent, List<Content>> firstLevelResultsByComponent = new HashMap<>();
        for (TwoStepsExecutingExtractionComponent component : components)
        {
            component.prepareComponentExecution(context);
            List<Content> firstLevelResults = StreamSupport.stream(component.computeFirstLevelResults(context).spliterator(), false)
                    .collect(Collectors.toList());
            firstLevelResultsByComponent.put(component, firstLevelResults);
        }
        
        List<Content> allContents = firstLevelResultsByComponent.values()
                .stream()
                .flatMap(List::stream)
                .collect(Collectors.toList());
        
        String unresolvedFilePath = _unresolvedFilePathWhenVar(unresolvedPath, pipelineDescriptor);
        Map<Path, List<Content>> paths = _resultPathResolver.resolvePath(unresolvedFilePath, allContents, extraction, basePath);
        for (Path filePath : paths.keySet())
        {
            List<Content> involvedContentsForPath = paths.get(filePath);
            try (OutputStream fileOs = Pipelines.getOutputStream(filePath);
                 Pipeline pipeline = pipelineDescriptor.newPipeline(fileOs);)
            {
                ContentHandler contentHandler = pipeline.getHandler();
                _withVarExecute(contentHandler, attributes, context, components, involvedContentsForPath, firstLevelResultsByComponent);
                pipeline.serialize();
            }
        }
        
        return paths.keySet();
    }
    
    private String _unresolvedFilePathWhenVar(String unresolvedFolderOrFilePath, PipelineDescriptor pipelineDescriptor)
    {
        if (_resultPathResolver.isFolder(unresolvedFolderOrFilePath))
        {
            return unresolvedFolderOrFilePath + "/${title}." + pipelineDescriptor.getDefaultExtension();
        }
        else
        {
            return unresolvedFolderOrFilePath;
        }
    }
    
    private void _withVarExecute(ContentHandler contentHandler, AttributesImpl attributes, ExtractionExecutionContext context, List<TwoStepsExecutingExtractionComponent> components, List<Content> involvedContentsForPath, Map<TwoStepsExecutingExtractionComponent, List<Content>> firstLevelResultsByComponent) throws Exception
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, ExtractionConstants.RESULT_EXTRACTION_TAG, attributes);
        
        for (TwoStepsExecutingExtractionComponent component : components)
        {
            List<Content> firstLevelResults = firstLevelResultsByComponent.get(component);
            List<Content> involvedFirstLevelResults = ListUtils.intersection(firstLevelResults, involvedContentsForPath);
            component.executeFor(contentHandler, involvedFirstLevelResults, context);
        }
        
        XMLUtils.endElement(contentHandler, ExtractionConstants.RESULT_EXTRACTION_TAG);
        contentHandler.endDocument();
    }
}
