/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.xml.sax.ContentHandler;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.plugins.extraction.execution.ExtractionExecutionContext;
import org.ametys.plugins.extraction.execution.ExtractionExecutionContextHierarchyElement;
import org.ametys.plugins.thesaurus.ThesaurusDAO;

/**
 * This class represents an extraction component
 */
public abstract class AbstractExtractionComponent extends AbstractLogEnabled implements ExtractionComponent, Serviceable, Configurable
{
    /** Hierarchy level separator for join expression */
    protected static final String JOIN_HIERARCHY_SEPARATOR = "/";
    
    /** Hierarchy level element representation for join expression */
    protected static final String JOIN_HIERARCHY_ELEMENT = "..";
    
    /** Metadata path separator used in extraction definition */
    protected static final String EXTRACTION_ITEM_PATH_SEPARATOR = "\\.";
    
    /** The list of sub components */
    protected List<ExtractionComponent> _subComponents = new ArrayList<>();
    
    /** the tag name */
    protected String _tagName;
    
    /** Helper for content types */
    protected ContentTypesHelper _contentTypesHelper;
    
    /** Content types extension point */
    protected ContentTypeExtensionPoint _contentTypeExtensionPoint;
    
    /** Thesaurus DAO */
    protected ThesaurusDAO _thesaurusDAO;
    
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE);
        _contentTypesHelper = (ContentTypesHelper) serviceManager.lookup(ContentTypesHelper.ROLE);
        _thesaurusDAO = (ThesaurusDAO) serviceManager.lookup(ThesaurusDAO.ROLE);
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _tagName = configuration.getAttribute("tagName", getDefaultTagName());
    }

    /**
     * Retrieves the default tag name
     * @return the default tag name
     */
    protected abstract String getDefaultTagName();
    
    public void prepareComponentExecution(ExtractionExecutionContext context) throws Exception
    {
        for (ExtractionComponent subComponent : _subComponents)
        {
            subComponent.prepareComponentExecution(context);
        }
    }
    
    public final void execute(ContentHandler contentHandler, ExtractionExecutionContext context) throws Exception
    {
        long startTime = -1;
        if (getLogger().isDebugEnabled())
        {
            startTime = System.currentTimeMillis();
            getLogger().debug(getLogsPrefix() + "executing component.");
        }
        
        executeComponent(contentHandler, context);
        
        if (getLogger().isDebugEnabled())
        {
            long endTime = System.currentTimeMillis();
            getLogger().debug(getLogsPrefix() + "executed component in " + (endTime - startTime) + "ms");
        }
    }
    
    /**
     * Execute the extraction of the component
     * @param contentHandler result document
     * @param context context of the extraction component
     * @throws Exception if an error occurs
     */
    protected abstract void executeComponent(ContentHandler contentHandler, ExtractionExecutionContext context) throws Exception;

    /**
     * Execute the sub components
     * @param contentHandler result document
     * @param context context of the extraction component
     * @param currentContextHierarchyElement the current context
     * @throws Exception if an error occurs
     */
    public void executeSubComponents(ContentHandler contentHandler, ExtractionExecutionContext context, ExtractionExecutionContextHierarchyElement currentContextHierarchyElement) throws Exception
    {
        ExtractionExecutionContext newContext = new ExtractionExecutionContext(context);
        
        List<ExtractionExecutionContextHierarchyElement> subComponentsContextElements = new ArrayList<>();
        subComponentsContextElements.addAll(context.getHierarchyElements());
        subComponentsContextElements.add(currentContextHierarchyElement);
        newContext.setHierarchyElements(subComponentsContextElements);
        
        for (ExtractionComponent subComponent : _subComponents)
        {
            subComponent.execute(contentHandler, newContext);
        }
    }
    
    public void addSubComponent(ExtractionComponent subComponent)
    {
        _subComponents.add(subComponent);
    }

    public List<ExtractionComponent> getSubComponents()
    {
        return _subComponents;
    }
    
    public Map<String, Object> getComponentDetailsForTree()
    {
        Map<String, Object> details = new HashMap<>();
        details.put("text", this.getTagName());
        
        Map<String, Object> data = new HashMap<>();
        data.put("componentTagName", this.getTagName());
        details.put("data", data);
        
        return details;
    }

    /**
     * Retrieves the prefix to use in exceptions thrown by this component
     * @return the prefix for exceptions
     */
    protected abstract String getLogsPrefix();
    
    /**
     * Retrieves the component tag name
     * @return the tag name
     */
    public String getTagName()
    {
        return _tagName;
    }

    /**
     * Sets the component tag name
     * @param tagName The tag name to set
     */
    public void setTagName(String tagName)
    {
        _tagName = tagName;
    }
}
