/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.nio.file.Path;
import java.time.Instant;
import java.time.temporal.ChronoUnit;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;

import org.ametys.cms.schedule.AbstractDeleteFilesSchedulable;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.schedule.Schedulable;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * A {@link Schedulable} job for deleting old extractions
 */
public class DeleteExtractionsSchedulable extends AbstractDeleteFilesSchedulable implements Initializable
{
    /** Parameter for workspace */
    public static final String LIFETIME_KEY = "lifetime";
    
    private static final String __JOBDATAMAP_LIFETIME_KEY = Scheduler.PARAM_VALUES_PREFIX + LIFETIME_KEY;
    
    /** Right Manager */
    protected RightManager _rightManager;
    
    /** Current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    private Path _rootPath;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        _rootPath = AmetysHomeHelper.getAmetysHomeData().toPath().resolve(ExtractionConstants.RESULT_EXTRACTION_DIR_NAME);
    }
    
    @Override
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        UserIdentity user = _currentUserProvider.getUser();
        if (_rightManager.hasRight(user, ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID, "/${WorkspaceName}") != RightResult.RIGHT_ALLOW)
        {
            throw new IllegalStateException("User " + user + " try to delete extractions with no sufficient rights");
        }
        
        super.execute(context, progressionTracker);
    }
    
    @Override
    protected DeleteFilesConfiguration _getConfiguration(JobExecutionContext context)
    {
        DeleteFilesConfiguration configuration = new DeleteFilesConfiguration(_rootPath);
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        long lifetime = jobDataMap.getLong(__JOBDATAMAP_LIFETIME_KEY);
        configuration.setAgeLimit(Instant.now().minus(lifetime, ChronoUnit.DAYS));
        return configuration;
    }
}
