/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution.pipeline.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;
import java.util.UUID;

import javax.xml.transform.sax.TransformerHandler;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.SourceUtil;

import org.ametys.core.util.URIUtils;
import org.ametys.plugins.extraction.execution.pipeline.PipelineSerializerModel;
import org.ametys.plugins.extraction.execution.pipeline.Pipelines;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * Model for PDF pipeline serializers
 */
public class PdfPipelineSerializerModel implements PipelineSerializerModel, Serviceable
{
    private SourceResolver _sourceResolver;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
    }
    
    @Override
    public String getDefaultFileExtension()
    {
        return "pdf";
    }
    
    @Override
    public PipelineSerializer newSerializer(TransformerHandler handler, OutputStream out, Map<String, String> outputParameters)
    {
        return new PdfPipelineSerializer(handler, out, _sourceResolver);
    }
    
    private static final class PdfPipelineSerializer extends AbstractSerializerImpl
    {
        private SourceResolver _sourceResolver;
        private OutputStream _tmpFileOs;
        private Path _tmpFile;
        
        PdfPipelineSerializer(TransformerHandler handler, OutputStream out, SourceResolver sourceResolver)
        {
            super(handler, out);
            _sourceResolver = sourceResolver;
        }
        
        @Override
        public void prepare() throws Exception
        {
            String tmpFilename = UUID.randomUUID().toString();
            _tmpFile = AmetysHomeHelper.getAmetysHomeTmp().toPath().resolve(Paths.get("extraction", tmpFilename));
            _tmpFileOs = Pipelines.getOutputStream(_tmpFile);
            _streamResult = Pipelines.setResult(_handler, _tmpFileOs);
        }
        
        @Override
        public void serialize() throws Exception
        {
            StringBuilder uri = new StringBuilder("cocoon://_admin/plugins/extraction/pdf-serialize");
            uri.append("?file=").append(URIUtils.encodeParameter(_tmpFile.toString()));
            SitemapSource src = (SitemapSource) _sourceResolver.resolveURI(uri.toString());
            try (InputStream sourceIs = src.getInputStream())
            {
                SourceUtil.copy(sourceIs, _out);
            }
            finally
            {
                _sourceResolver.release(src);
            }
        }
        
        @Override
        public void close() throws IOException
        {
            super.close();
            _tmpFileOs.close();
            Files.delete(_tmpFile);
        }
    }
}
