/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class controls a ribbon button that is able to check for conditions on a form element
 * @private
 */
Ext.define('Ametys.plugins.forms.controllers.FormController', {
    extend: 'Ametys.ribbon.element.ui.ButtonController',
    
    constructor: function (config)
    {
        this.callParent(arguments);
        
        Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onModified, this);
    },
    
    /**
     * Listener on modified message.
     * Update the state of the controller accordingly.
     * @param {Ametys.message.Message} message the message of type modified.
     * @private
     */
    _onModified: function(message)
    {
        if (this.updateTargetsInCurrentSelectionTargets(message))
        {
            this.refresh();
        }
    },
    
    updateState: function ()
    {
        this.refreshing();
        this.disable();
            
        this._getStatus(this.getMatchingTargets());
    },
    
    /**
     * Refresh the controller from the form informations state of given targets
     * @param {Ametys.message.MessageTarget[]} targets The form targets
     * @protected
     */
    _getStatus: function (targets)
    {
        var description = this.getInitialConfig("description") || this.getInitialConfig("default-description") || '';
        
        var enabledOnPrivateOnly = this.getConfig("enable-on-private-only") == "true";
        var enabledOnPublishedOnly = this.getConfig("enable-on-published-only") == "true";
        var enabledOnUnpublishedOnly = this.getConfig("enable-on-unpublished-only") == "true";
        var enabledOnNoEntryOnly = this.getConfig("enable-on-noentry-only") == "true";
        var enabledOnEntryOnly = this.getConfig("enable-on-entry-only") == "true";
        
        // Multiselection is currently not supported
        var matchingTarget = targets[0];
        
        var errorDescription = "";
        
        if (enabledOnPrivateOnly && matchingTarget.getParameters().isAnonymous)
        {
            errorDescription = this.getConfig("noprivate-description");
        }
        else if (enabledOnPublishedOnly && !matchingTarget.getParameters().isPublished)
        {
            errorDescription = this.getConfig("unpublished-description");
        }
        else if (enabledOnUnpublishedOnly && matchingTarget.getParameters().isPublished)
        {
            errorDescription = this.getConfig("published-description");
        }
        else if (enabledOnNoEntryOnly && matchingTarget.getParameters().hasEntries)
        {
            errorDescription = this.getConfig("has-entries-description");
        }
        else if (enabledOnEntryOnly && !matchingTarget.getParameters().hasEntries)
        {
            errorDescription = this.getConfig("noentry-description");
        }
        else
        {
            errorDescription = this.getErrorDescription(matchingTarget);
        }
        
        if (errorDescription == '')
        {
            this.enable();
            var additionnalDescription = this.getAdditionnalDescription(matchingTarget);
            if (description != '' && additionnalDescription != '')
            {
                description += "<br/><br/>";
            }
            description += additionnalDescription
        }
        else
        {
            if (description != '')
            {
                description += "<br/><br/>";
            }
            description += errorDescription;
            this.disable();
        }
        
        this.stopRefreshing();
        this.setDescription (description);
    },
    
    /**
     * Get an additional description to display when the button is enabled
     * @param {Ametys.message.MessageTarget} matchingTarget The matching target
     * @return {String} teh additional description
     */
    getAdditionnalDescription: function(matchingTarget)
    {
        return "";
    },
    
    /**
     * @protected
     * Get an additional error not handled by default form controller
     * @param {Ametys.message.MessageTarget} matchingTarget The matching target
     * @return {String} additional error or empty is there is no error for current selection
     */
    getErrorDescription: function(matchingTarget)
    {
        return "";
    }
});
