/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.data;

import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.reading.ServiceableReader;
import org.apache.commons.lang.StringUtils;
import org.apache.excalibur.source.SourceUtil;
import org.xml.sax.SAXException;

import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.core.datasource.dbtype.SQLDatabaseTypeExtensionPoint;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.content.table.FormTableManager;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.config.Config;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;

/**
 * Reads a BLOB value of a form entry.
 */
public class FormEntryFileReader extends ServiceableReader
{
    private CurrentUserProvider _currentUserProvider;
    
    private SQLDatabaseTypeExtensionPoint _sqlDatabaseTypeExtensionPoint;

    private RenderingContextHandler _renderingContextHandler;

    private SQLDatabaseTypeExtensionPoint getSQLDatabaseTypeExtensionPoint()
    {
        if (_sqlDatabaseTypeExtensionPoint == null)
        {
            try
            {
                _sqlDatabaseTypeExtensionPoint = (SQLDatabaseTypeExtensionPoint) manager.lookup(SQLDatabaseTypeExtensionPoint.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException(e);
            }
        }
        return _sqlDatabaseTypeExtensionPoint;
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _renderingContextHandler = (RenderingContextHandler) smanager.lookup(RenderingContextHandler.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        String siteName = parameters.getParameter("site", "");
        String formId = parameters.getParameter("form-id", "");
        int entryId = parameters.getParameterAsInteger("entry-id", Integer.MIN_VALUE);
        String fieldId = parameters.getParameter("field-id", "");
        
        if (StringUtils.isEmpty(siteName) || StringUtils.isEmpty(formId)
            || entryId == Integer.MIN_VALUE || StringUtils.isEmpty(fieldId))
        {
            throw new IllegalArgumentException("Site name, form id, entry id and field id must be provided.");
        }
        
        String tableName = FormTableManager.TABLE_PREFIX + formId;
        
        Connection connection = null;
        PreparedStatement stmt = null;
        ResultSet rs = null;
        
        try
        {
            String dataSourceId = Config.getInstance().getValue(FormTableManager.FORMS_POOL_CONFIG_PARAM);
            connection = ConnectionHelper.getConnection(dataSourceId);
            
            String dbType = ConnectionHelper.getDatabaseType(connection);
            String sql = "SELECT " + getSQLDatabaseTypeExtensionPoint().languageEscapeTableName(dbType, fieldId) 
                    + "," + FormTableManager.LOGIN_FIELD
                    + "," + FormTableManager.POPULATION_ID_FIELD
                    + " FROM " + getSQLDatabaseTypeExtensionPoint().languageEscapeTableName(dbType, tableName) 
                    + " WHERE id = ?";
            
            stmt = connection.prepareStatement(sql);
            
            stmt.setInt(1, entryId);
            
            // Execute the query.
            rs = stmt.executeQuery();
            
            // Extract the result.
            if (rs.next())
            {
                String login = rs.getString(FormTableManager.LOGIN_FIELD);
                String population = rs.getString(FormTableManager.POPULATION_ID_FIELD);
                
                _checkAccess(login, population);
                
                try (InputStream is = _sqlDatabaseTypeExtensionPoint.getBlob(dbType, rs, 1))
                {
                    if (is != null)
                    {
                        SourceUtil.copy(is, out);
                    }
                }
            }
        }
        catch (SQLException e)
        {
            getLogger().error("Error reading a form entry blob." + tableName, e);
            throw new ProcessingException("Error reading a form entry blob.", e);
        }
        finally
        {
            ConnectionHelper.cleanup(rs);
            ConnectionHelper.cleanup(stmt);
            ConnectionHelper.cleanup(connection);
        }
    }
    
    private void _checkAccess(String login, String population)
    {
        UserIdentity entryOwner = null;
        if (StringUtils.isNotEmpty(login) && StringUtils.isNotEmpty(population))
        {
            entryOwner = new UserIdentity(login, population);
        }
        
        RenderingContext context = _renderingContextHandler.getRenderingContext();
        if (context == RenderingContext.FRONT)
        {
            // If request come from front, check user is the file owner
            UserIdentity currentUser = _currentUserProvider.getUser();
            if (entryOwner == null || !entryOwner.equals(currentUser))
            {
                throw new AccessDeniedException("User '" + currentUser + "' is not allowed to access to user entry data.");
            }
        }
    }
}
