/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.computing;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.web.repository.page.Page;

/**
 * Class for current page id computing type
 */
public class CurrentPageIdComputingType extends AbstractStaticComputingType implements Contextualizable
{
    /** The context */
    protected Context _context;

    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public Object getComputedValue(FormQuestion question, FormEntry entry)
    {
        Request request = ContextHelper.getRequest(_context);
        return request.getParameter(question.getNameForForm());
    }

    public Map<String, ModelItem> getModelItems()
    {
        return Map.of();
    }

    public String getStorageType(FormQuestion question)
    {
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }

    public List<ViewItem> getViewElements()
    {
        return List.of();
    }
    
    @Override
    public void saxAdditionalValue(ContentHandler contentHandler, FormQuestion question, FormEntry entry) throws SAXException
    {
        String pageId = entry.getValue(question.getNameForForm());
        if (StringUtils.isNotBlank(pageId))
        {
            Page page = _silentlyResolve(pageId);
            if (page != null)
            {
                XMLUtils.startElement(contentHandler, "page");
                XMLUtils.createElement(contentHandler, "title", page.getTitle());
                XMLUtils.createElement(contentHandler, "id", page.getId());
                XMLUtils.endElement(contentHandler, "page");
            }
        }
    }
    
    private Page _silentlyResolve(String pageId)
    {
        try
        {
            return _resolver.resolveById(pageId);
        }
        catch (Exception e) 
        {
            return null;
        }
    }
    
    @Override
    public String getJSRenderer()
    {
        return "Ametys.plugins.forms.helper.SearchEntriesGridHelper.renderPage";
    }
    
    public Object valueToJSONForClient(Object value, FormQuestion question, FormEntry entry, ModelItem modelItem) throws Exception
    {
        String pageId = value != null ? (String) value : null;
        if (StringUtils.isNotBlank(pageId))
        {
            Page page = _silentlyResolve(pageId);
            if (page != null)
            {
                return Map.of("id", page.getId(), "title", page.getTitle());
            }
        }
        
        return value;
    }
    
    public boolean canEdit()
    {
        return false;
    }
}
