/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.computing;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Class for creating user computed field
 */
public class UserComputingType extends AbstractStaticComputingType
{
    /** The user info attribute */
    public static final String ATTRIBUTE_USER_INFO = "user-info";
    
    /** Name of email userInfoStaticEnumerator entry */
    public static final String EMAIL_USER_VALUE = "email";
    /** Name of id userInfoStaticEnumerator entry */
    public static final String ID_USER_VALUE = "id";
    /** Name of fullname userInfoStaticEnumerator entry */
    public static final String FULLNAME_USER_VALUE = "fullName";
    /** Name of firstName userInfoStaticEnumerator entry */
    public static final String FIRSTNAME_USER_VALUE = "firstName";
    /** Name of lastName userInfoStaticEnumerator entry */
    public static final String LASTNAME_USER_VALUE = "lastName";
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** The users manager */
    protected UserManager _userManager;
    /** Map of ModelItems specific to ManualSourceType */
    protected Map<String, ModelItem> _userComputingItems;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
    }
    
    public String getComputedValue(FormQuestion computedQuestion, FormEntry entry)
    {
        UserIdentity userIdentity = _currentUserProvider.getUser();
        User user = userIdentity != null ? _userManager.getUser(userIdentity) : null;
        return user != null ? _getUserInfoValue(computedQuestion, user) : null;
    }

    public Map<String, ModelItem> getModelItems()
    {
        _userComputingItems = new HashMap<>();
        
        ElementDefinition<String> userInfos = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_USER_INFO, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_COMPUTING_USERS", "PLUGINS_FORMS_QUESTIONS_DIALOG_COMPUTING_USERS_DESC", null);
        StaticEnumerator<String> userInfoStaticEnumerator = new StaticEnumerator<>();
        userInfoStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_EMAIL"), EMAIL_USER_VALUE);
        userInfoStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_ID"), ID_USER_VALUE);
        userInfoStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_FULLNAME"), FULLNAME_USER_VALUE);
        userInfoStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_FIRSTNAME"), FIRSTNAME_USER_VALUE);
        userInfoStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_LASTNAME"), LASTNAME_USER_VALUE);
        userInfos.setEnumerator(userInfoStaticEnumerator);
        userInfos.setDefaultValue(FULLNAME_USER_VALUE);
        
        _userComputingItems.put(userInfos.getName(), userInfos);
       
        return _userComputingItems;
    }

    public List<ViewItem> getViewElements()
    {
        List<ViewItem> viewElements = new ArrayList<>();
        
        ViewElement userSelect = new ViewElement();
        userSelect.setDefinition((ElementDefinition< ? >) _userComputingItems.get(ATTRIBUTE_USER_INFO));
        viewElements.add(userSelect);
        
        return viewElements;
    }

   /**
    * Get user value
    * @param question the question
    * @param user the user
    * @return the user value.<code>null</code> if there is not user value
    */
    protected String _getUserInfoValue(FormQuestion question, User user) 
    {
        String userInfo = question.getValue(ATTRIBUTE_USER_INFO);
        switch (userInfo)
        {
            case EMAIL_USER_VALUE:
                return user.getEmail();
            case ID_USER_VALUE:
                return user.getIdentity().getLogin(); 
            case FULLNAME_USER_VALUE:
                return user.getFullName();
            case FIRSTNAME_USER_VALUE:
                return user.getFirstName();
            case LASTNAME_USER_VALUE:
                return user.getLastName();
            default:
                return null;
        }
    }

    public String getStorageType(FormQuestion question)
    {
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }
    
    @Override
    public List<String> getFieldToDisableIfFormPublished()
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished();
        fieldNames.add(ATTRIBUTE_USER_INFO);
        return fieldNames;
    }
    
    public boolean canEdit()
    {
        return false;
    }
}
