/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.types;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Provide all the necessary methods to add a autocomplete attribute to a {@link FormQuestionType}
 */
public interface AutocompleteAwareQuestionType
{
    /** Constant for autocomplete attribute. */
    public static final String ATTRIBUTE_AUTOCOMPLETE = "autocomplete";

    /**
     * The type of the input
     */
    public enum InputType
    {
        /** A simple text input */
        TEXT,
        /** A textarea input */
        TEXTAREA,
        /** A date input */
        DATE,
        /** A select input */
        SELECT
    }
    
    private Map<AutocompleteOption, List<InputType>> _getAutocompleteOptions()
    {
        Map<AutocompleteOption, List<InputType>> autocompleteOptions = new HashMap<>();
        autocompleteOptions.put(
                new AutocompleteOption("name", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_FULLNAME")),
                List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("honorific-prefix", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_HONORIFIC_PREFIX")),
            List.of(InputType.SELECT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("given-name", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_GIVEN_NAME")),
            List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("family-name", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_FAMILY_NAME")),
            List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("email", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_EMAIL")),
            List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("organization-title", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_ORGANIZATION_TITLE")),
            List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("organization", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_ORGANIZATION")),
            List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("street-address", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_STREET_ADDRESS")),
            List.of(InputType.TEXT, InputType.TEXTAREA)
        );
        autocompleteOptions.put(
            new AutocompleteOption("country-name", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_COUNTRY_NAME")),
            List.of(InputType.TEXT, InputType.SELECT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("postal-code", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_POSTAL_CODE")),
            List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("bday", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_BIRTHDAY")),
            List.of(InputType.DATE)
        );
        autocompleteOptions.put(
            new AutocompleteOption("sex", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_SEX")),
            List.of(InputType.SELECT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("tel", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_TEL")),
            List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("url", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_URL")),
            List.of(InputType.TEXT)
        );
        
        return autocompleteOptions;
    }
    
    
    /**
     * Get a model item to include in the model return by FormQuestionType.getModel
     * @param inputType the input type
     * @return the item
     */
    public default ModelItem getAutocompleteModelItem(InputType inputType)
    {
        ElementDefinition<String> autocomplete = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_AUTOCOMPLETE, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_LABEL", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_DESC", null);
        StaticEnumerator<String> autocompleteStaticEnumerator = new StaticEnumerator<>();
        
        _getAutocompleteOptions().entrySet()
            .stream()
            .filter(e -> e.getValue().contains(inputType))
            .forEach(e -> autocompleteStaticEnumerator.add(e.getKey().label(), e.getKey().value()));
        
        autocomplete.setEnumerator(autocompleteStaticEnumerator);
        
        return autocomplete;
    }
    
    /**
     * Get a view element to include in the view returned by FormQuestionType.getView
     * @param model the question type model
     * @return the element
     */
    public default ViewElement getAutocompleteViewElement(Model model)
    {
        ViewElement description = new ViewElement();
        description.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_AUTOCOMPLETE));
        return description;
    }
    
    /**
     * Get the question description
     * @param question the question
     * @return the description
     */
    public default String getAutocomplete(FormQuestion question)
    {
        return question.getValue(AutocompleteAwareQuestionType.ATTRIBUTE_AUTOCOMPLETE);
    }
    
    /**
     * Record representing an autocomplete option
     * @param value the value of the option
     * @param label the label of the option
     */
    record AutocompleteOption(String value, I18nizableText label) { /** */ }
}
